import React from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  Box,
  Typography,
  IconButton,
  Chip,
  Grid,
  Paper,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Divider,
} from '@mui/material';
import {
  Close,
  CalendarMonth,
  AccessTime,
  CheckCircle,
  EventAvailable,
} from '@mui/icons-material';

const DAYS_OF_WEEK = [
  { short: 'Mon', full: 'Monday' },
  { short: 'Tue', full: 'Tuesday' },
  { short: 'Wed', full: 'Wednesday' },
  { short: 'Thu', full: 'Thursday' },
  { short: 'Fri', full: 'Friday' },
  { short: 'Sat', full: 'Saturday' },
  { short: 'Sun', full: 'Sunday' },
];

// Mock schedule data - In a real app, this would come from your backend
const getMentorSchedule = (mentorId) => {
  // This is sample data - replace with actual data from your backend
  return {
    duration: 60,
    availability: {
      Monday: ['09:00', '10:00', '14:00'],
      Tuesday: ['09:00', '10:00', '14:00', '15:00'],
      Wednesday: ['10:00', '11:00', '15:00'],
      Thursday: ['09:00', '14:00', '15:00', '16:00'],
      Friday: ['09:00', '10:00', '13:00', '14:00'],
      Saturday: [],
      Sunday: [],
    },
    totalSlotsPerWeek: 17,
    activeDays: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri'],
  };
};

export function ViewScheduleDialog({ open, onClose, mentor }) {
  const schedule = getMentorSchedule(mentor?.id);

  const getTotalSlots = () => {
    return Object.values(schedule.availability).reduce(
      (total, slots) => total + slots.length, 
      0
    );
  };

  const getActiveDaysCount = () => {
    return Object.values(schedule.availability).filter(
      slots => slots.length > 0
    ).length;
  };

  const formatTimeSlots = (slots) => {
    if (slots.length === 0) return 'No slots available';
    return slots.join(', ');
  };

  const getTimeRanges = (slots) => {
    if (slots.length === 0) return '-';
    if (slots.length === 1) return slots[0];
    
    // Group consecutive slots
    const sortedSlots = [...slots].sort();
    const ranges = [];
    let rangeStart = sortedSlots[0];
    let rangeEnd = sortedSlots[0];

    for (let i = 1; i < sortedSlots.length; i++) {
      const currentTime = sortedSlots[i];
      const prevTime = sortedSlots[i - 1];
      
      // Check if times are consecutive (assuming 1-hour slots)
      const currentHour = parseInt(currentTime.split(':')[0]);
      const prevHour = parseInt(prevTime.split(':')[0]);
      
      if (currentHour === prevHour + 1) {
        rangeEnd = currentTime;
      } else {
        ranges.push(rangeStart === rangeEnd ? rangeStart : `${rangeStart} - ${rangeEnd}`);
        rangeStart = currentTime;
        rangeEnd = currentTime;
      }
    }
    
    ranges.push(rangeStart === rangeEnd ? rangeStart : `${rangeStart} - ${rangeEnd}`);
    return ranges.join(', ');
  };

  return (
    <Dialog 
      open={open} 
      onClose={onClose} 
      maxWidth="lg" 
      fullWidth
      PaperProps={{
        sx: { 
          borderRadius: 3,
          minHeight: '70vh',
          maxHeight: '90vh',
        }
      }}
    >
      <DialogTitle sx={{ p: 3, pb: 2 }}>
        <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
          <Box sx={{ display: 'flex', alignItems: 'center' }}>
            <CalendarMonth sx={{ color: '#38aae1', mr: 2, fontSize: 28 }} />
            <Box>
              <Typography variant="h5" sx={{ fontWeight: 600, color: '#25476a' }}>
                View Availability Schedule
              </Typography>
              <Typography variant="body2" sx={{ color: '#64748b', mt: 0.5 }}>
                {mentor?.name}'s consultation availability schedule
              </Typography>
            </Box>
          </Box>
          <IconButton onClick={onClose} sx={{ color: '#64748b' }}>
            <Close />
          </IconButton>
        </Box>
      </DialogTitle>

      <DialogContent sx={{ p: 3, pt: 1 }}>
        {/* Schedule Summary */}
        <Grid container spacing={3} sx={{ mb: 4 }}>
          <Grid item xs={12} md={3}>
            <Paper 
              sx={{ 
                p: 3, 
                borderRadius: 2, 
                backgroundColor: '#f0f9ff',
                border: '1px solid #bae6fd',
              }}
            >
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                <AccessTime sx={{ color: '#38aae1', fontSize: 20, mr: 1 }} />
                <Typography variant="body2" sx={{ color: '#64748b' }}>
                  Session Duration
                </Typography>
              </Box>
              <Typography variant="h4" sx={{ fontWeight: 700, color: '#25476a' }}>
                {schedule.duration} min
              </Typography>
            </Paper>
          </Grid>

          <Grid item xs={12} md={3}>
            <Paper 
              sx={{ 
                p: 3, 
                borderRadius: 2, 
                backgroundColor: '#fff7ed',
                border: '1px solid #fed7aa',
              }}
            >
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                <EventAvailable sx={{ color: '#feb139', fontSize: 20, mr: 1 }} />
                <Typography variant="body2" sx={{ color: '#64748b' }}>
                  Total Slots
                </Typography>
              </Box>
              <Typography variant="h4" sx={{ fontWeight: 700, color: '#25476a' }}>
                {getTotalSlots()}
              </Typography>
              <Typography variant="caption" sx={{ color: '#64748b' }}>
                per week
              </Typography>
            </Paper>
          </Grid>

          <Grid item xs={12} md={3}>
            <Paper 
              sx={{ 
                p: 3, 
                borderRadius: 2, 
                backgroundColor: '#f0fdf4',
                border: '1px solid #bbf7d0',
              }}
            >
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                <CheckCircle sx={{ color: '#22c55e', fontSize: 20, mr: 1 }} />
                <Typography variant="body2" sx={{ color: '#64748b' }}>
                  Active Days
                </Typography>
              </Box>
              <Typography variant="h4" sx={{ fontWeight: 700, color: '#25476a' }}>
                {getActiveDaysCount()}
              </Typography>
              <Typography variant="caption" sx={{ color: '#64748b' }}>
                days per week
              </Typography>
            </Paper>
          </Grid>

          <Grid item xs={12} md={3}>
            <Paper 
              sx={{ 
                p: 3, 
                borderRadius: 2, 
                backgroundColor: '#faf5ff',
                border: '1px solid #e9d5ff',
              }}
            >
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                <AccessTime sx={{ color: '#a855f7', fontSize: 20, mr: 1 }} />
                <Typography variant="body2" sx={{ color: '#64748b' }}>
                  Total Hours
                </Typography>
              </Box>
              <Typography variant="h4" sx={{ fontWeight: 700, color: '#25476a' }}>
                {(getTotalSlots() * schedule.duration) / 60}h
              </Typography>
              <Typography variant="caption" sx={{ color: '#64748b' }}>
                per week
              </Typography>
            </Paper>
          </Grid>
        </Grid>

        <Divider sx={{ mb: 3 }} />

        {/* Weekly Schedule Table */}
        <Box sx={{ mb: 3 }}>
          <Typography variant="h6" sx={{ mb: 2, color: '#25476a', fontWeight: 600 }}>
            Weekly Availability
          </Typography>
          
          <TableContainer component={Paper} sx={{ borderRadius: 2, border: '1px solid #e2e8f0' }}>
            <Table>
              <TableHead sx={{ backgroundColor: '#f8fafc' }}>
                <TableRow>
                  <TableCell sx={{ fontWeight: 600, color: '#25476a', width: '120px' }}>
                    Day
                  </TableCell>
                  <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>
                    Available Time Slots
                  </TableCell>
                  <TableCell sx={{ fontWeight: 600, color: '#25476a', width: '100px', textAlign: 'center' }}>
                    Slots
                  </TableCell>
                  <TableCell sx={{ fontWeight: 600, color: '#25476a', width: '120px', textAlign: 'center' }}>
                    Status
                  </TableCell>
                </TableRow>
              </TableHead>
              <TableBody>
                {DAYS_OF_WEEK.map((day) => {
                  const daySlots = schedule.availability[day.full] || [];
                  const hasSlots = daySlots.length > 0;
                  
                  return (
                    <TableRow 
                      key={day.full}
                      sx={{ 
                        '&:hover': { backgroundColor: '#f8fafc' },
                        backgroundColor: hasSlots ? 'white' : '#fafbfc',
                      }}
                    >
                      <TableCell>
                        <Box sx={{ display: 'flex', alignItems: 'center' }}>
                          <Box
                            sx={{
                              width: 8,
                              height: 8,
                              borderRadius: '50%',
                              backgroundColor: hasSlots ? '#22c55e' : '#e2e8f0',
                              mr: 2,
                            }}
                          />
                          <Typography variant="body1" sx={{ fontWeight: 600, color: '#25476a' }}>
                            {day.full}
                          </Typography>
                        </Box>
                      </TableCell>
                      <TableCell>
                        <Typography 
                          variant="body2" 
                          sx={{ 
                            color: hasSlots ? '#25476a' : '#94a3b8',
                            fontFamily: 'monospace',
                          }}
                        >
                          {formatTimeSlots(daySlots)}
                        </Typography>
                      </TableCell>
                      <TableCell sx={{ textAlign: 'center' }}>
                        <Chip
                          label={daySlots.length}
                          size="small"
                          sx={{
                            backgroundColor: hasSlots ? '#dbeafe' : '#f1f5f9',
                            color: hasSlots ? '#1e40af' : '#64748b',
                            fontWeight: 600,
                            minWidth: 45,
                          }}
                        />
                      </TableCell>
                      <TableCell sx={{ textAlign: 'center' }}>
                        <Chip
                          label={hasSlots ? 'Available' : 'Off'}
                          size="small"
                          color={hasSlots ? 'success' : 'default'}
                          variant={hasSlots ? 'filled' : 'outlined'}
                          sx={{
                            fontWeight: 500,
                            minWidth: 85,
                          }}
                        />
                      </TableCell>
                    </TableRow>
                  );
                })}
              </TableBody>
            </Table>
          </TableContainer>
        </Box>

        {/* Time Range Summary */}
        <Paper 
          sx={{ 
            p: 3, 
            borderRadius: 2,
            backgroundColor: '#f8fafc',
            border: '1px solid #e2e8f0',
          }}
        >
          <Typography variant="h6" sx={{ mb: 2, color: '#25476a', fontWeight: 600 }}>
            Daily Time Ranges
          </Typography>
          <Grid container spacing={2}>
            {DAYS_OF_WEEK.map((day) => {
              const daySlots = schedule.availability[day.full] || [];
              if (daySlots.length === 0) return null;

              return (
                <Grid item xs={12} sm={6} md={4} key={day.full}>
                  <Box
                    sx={{
                      p: 2,
                      borderRadius: 1.5,
                      backgroundColor: 'white',
                      border: '1px solid #e2e8f0',
                    }}
                  >
                    <Typography 
                      variant="body2" 
                      sx={{ 
                        fontWeight: 600, 
                        color: '#25476a',
                        mb: 0.5,
                      }}
                    >
                      {day.short}
                    </Typography>
                    <Typography 
                      variant="body2" 
                      sx={{ 
                        color: '#64748b',
                        fontFamily: 'monospace',
                        fontSize: '0.85rem',
                      }}
                    >
                      {getTimeRanges(daySlots)}
                    </Typography>
                  </Box>
                </Grid>
              );
            })}
          </Grid>
        </Paper>
      </DialogContent>

      <DialogActions sx={{ p: 3, pt: 2, gap: 2 }}>
        <Button 
          onClick={onClose}
          variant="outlined"
          sx={{
            borderColor: '#e2e8f0',
            color: '#64748b',
            '&:hover': {
              borderColor: '#cbd5e1',
              backgroundColor: '#f8fafc',
            },
          }}
        >
          Close
        </Button>
        <Button 
          onClick={onClose}
          variant="contained"
          sx={{
            backgroundColor: '#38aae1',
            '&:hover': {
              backgroundColor: '#2e8bc4',
            },
          }}
        >
          Edit Schedule
        </Button>
      </DialogActions>
    </Dialog>
  );
}
