import React, { Component } from 'react';
import { Box } from '@mui/material';

interface VidstackPlayerProps {
  src: string;
  title?: string;
  poster?: string;
  aspectRatio?: string;
  autoplay?: boolean;
  controls?: boolean;
  className?: string;
}

interface VidstackPlayerState {
  isReady: boolean;
  hasError: boolean;
}

export class VidstackPlayer extends Component<VidstackPlayerProps, VidstackPlayerState> {
  static defaultProps = {
    title: 'Video',
    aspectRatio: '16/9',
    autoplay: false,
    controls: true,
  };

  private isMounted: boolean = false;
  private initTimeout: NodeJS.Timeout | null = null;
  private videoRef: React.RefObject<HTMLVideoElement> = React.createRef();

  constructor(props: VidstackPlayerProps) {
    super(props);
    this.state = {
      isReady: false,
      hasError: false,
    };
  }

  componentDidMount() {
    this.isMounted = true;
    // Delay rendering to ensure parent is fully mounted and stable
    this.initTimeout = setTimeout(() => {
      if (this.isMounted) {
        this.setState({ isReady: true });
      }
    }, 200);
  }

  componentWillUnmount() {
    this.isMounted = false;
    if (this.initTimeout) {
      clearTimeout(this.initTimeout);
      this.initTimeout = null;
    }
  }

  handleError = () => {
    if (this.isMounted) {
      this.setState({ hasError: true });
    }
  };

  render() {
    const { src, title, poster, aspectRatio, className, controls } = this.props;
    const { isReady, hasError } = this.state;

    if (!src || hasError) {
      return (
        <Box 
          className={className}
          sx={{ 
            width: '100%',
            aspectRatio: aspectRatio || '16/9',
            borderRadius: 2,
            overflow: 'hidden',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            bgcolor: '#f8fafc',
            color: '#64748b',
          }}
        >
          <p>Video player unavailable</p>
        </Box>
      );
    }

    if (!isReady) {
      return (
        <Box 
          className={className}
          sx={{ 
            width: '100%',
            aspectRatio: aspectRatio || '16/9',
            borderRadius: 2,
            overflow: 'hidden',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            bgcolor: '#f8fafc',
            color: '#64748b',
          }}
        >
          <p>Loading player...</p>
        </Box>
      );
    }

    return (
      <Box 
        className={className}
        sx={{ 
          width: '100%',
          aspectRatio: aspectRatio || '16/9',
          borderRadius: 2,
          overflow: 'hidden',
          bgcolor: '#000',
          '& video': {
            width: '100%',
            height: '100%',
            objectFit: 'contain',
          }
        }}
      >
        <video
          ref={this.videoRef}
          src={src}
          poster={poster}
          controls={controls}
          playsInline
          preload="metadata"
          title={title}
          onError={this.handleError}
          style={{
            width: '100%',
            height: '100%',
            borderRadius: '8px',
          }}
        />
      </Box>
    );
  }
}
