import React, { Component } from 'react';
import { Box } from '@mui/material';

export class VidstackPlayer extends Component {
  static defaultProps = {
    title: 'Video',
    aspectRatio: '16/9',
    autoplay: false,
    controls: true,
  };

  constructor(props) {
    super(props);
    this.state = {
      isReady: false,
      hasError: false,
    };
    this.isMounted = false;
    this.initTimeout = null;
    this.videoRef = React.createRef();
  }

  componentDidMount() {
    this.isMounted = true;
    // Delay rendering to ensure parent is fully mounted and stable
    this.initTimeout = setTimeout(() => {
      if (this.isMounted) {
        this.setState({ isReady: true });
      }
    }, 200);
  }

  componentWillUnmount() {
    this.isMounted = false;
    if (this.initTimeout) {
      clearTimeout(this.initTimeout);
      this.initTimeout = null;
    }
  }

  handleError = () => {
    if (this.isMounted) {
      this.setState({ hasError: true });
    }
  };

  render() {
    const { src, title, poster, aspectRatio, className, controls } = this.props;
    const { isReady, hasError } = this.state;

    if (!src || hasError) {
      return (
        <Box 
          className={className}
          sx={{ 
            width: '100%',
            aspectRatio: aspectRatio || '16/9',
            borderRadius: 2,
            overflow: 'hidden',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            bgcolor: '#f8fafc',
            color: '#64748b',
          }}
        >
          <p>Video player unavailable</p>
        </Box>
      );
    }

    if (!isReady) {
      return (
        <Box 
          className={className}
          sx={{ 
            width: '100%',
            aspectRatio: aspectRatio || '16/9',
            borderRadius: 2,
            overflow: 'hidden',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            bgcolor: '#f8fafc',
            color: '#64748b',
          }}
        >
          <p>Loading player...</p>
        </Box>
      );
    }

    return (
      <Box 
        className={className}
        sx={{ 
          width: '100%',
          aspectRatio: aspectRatio || '16/9',
          borderRadius: 2,
          overflow: 'hidden',
          bgcolor: '#000',
          '& video': {
            width: '100%',
            height: '100%',
            objectFit: 'contain',
          }
        }}
      >
        <video
          ref={this.videoRef}
          src={src}
          poster={poster}
          controls={controls}
          playsInline
          preload="metadata"
          title={title}
          onError={this.handleError}
          style={{
            width: '100%',
            height: '100%',
            borderRadius: '8px',
          }}
        />
      </Box>
    );
  }
}
