import React, { Component } from 'react';
import { Box, Typography } from '@mui/material';
import { CloudUpload } from '@mui/icons-material';

export class VideoUploadDragger extends Component {
  constructor(props) {
    super(props);
    this.state = {
      isDragging: false,
    };
    this.fileInputRef = React.createRef();
  }

  handleDragEnter = (e) => {
    e.preventDefault();
    e.stopPropagation();
    this.setState({ isDragging: true });
  };

  handleDragLeave = (e) => {
    e.preventDefault();
    e.stopPropagation();
    this.setState({ isDragging: false });
  };

  handleDragOver = (e) => {
    e.preventDefault();
    e.stopPropagation();
  };

  handleDrop = (e) => {
    e.preventDefault();
    e.stopPropagation();
    this.setState({ isDragging: false });

    const files = e.dataTransfer.files;
    if (files && files.length > 0) {
      this.handleFile(files[0]);
    }
  };

  handleFileInput = (e) => {
    const files = e.target.files;
    if (files && files.length > 0) {
      this.handleFile(files[0]);
    }
  };

  handleFile = (file) => {
    const { accept = 'video/*', maxSize = 500, onUpload } = this.props;

    // Validate file type
    if (accept === 'video/*' && !file.type.startsWith('video/')) {
      alert('You can only upload video files!');
      return;
    }

    // Validate file size
    const fileSizeMB = file.size / 1024 / 1024;
    if (fileSizeMB > maxSize) {
      alert(`Video must be smaller than ${maxSize}MB!`);
      return;
    }

    onUpload(file);
  };

  handleClick = () => {
    this.fileInputRef.current?.click();
  };

  render() {
    const { isDragging } = this.state;
    const {
      accept = 'video/*',
      icon,
      title = 'Upload Video',
      description = 'Click to browse or drag and drop your video file',
      helpText = 'Supported formats: MP4, MOV, AVI, WebM (Max 500MB)',
      style,
    } = this.props;

    return (
      <Box
        onClick={this.handleClick}
        onDragEnter={this.handleDragEnter}
        onDragOver={this.handleDragOver}
        onDragLeave={this.handleDragLeave}
        onDrop={this.handleDrop}
        sx={{
          border: isDragging ? '2px dashed #38aae1' : '2px dashed #e2e8f0',
          borderRadius: 2,
          p: 3,
          textAlign: 'center',
          bgcolor: isDragging ? 'rgba(56, 170, 225, 0.1)' : '#f8fafc',
          cursor: 'pointer',
          transition: 'all 0.3s ease',
          '&:hover': {
            borderColor: '#38aae1',
            bgcolor: 'rgba(56, 170, 225, 0.05)',
          },
          ...style,
        }}
      >
        <input
          ref={this.fileInputRef}
          type="file"
          accept={accept}
          style={{ display: 'none' }}
          onChange={this.handleFileInput}
        />
        {icon || <CloudUpload sx={{ fontSize: 48, color: '#64748b', mb: 1 }} />}
        <Typography variant="body2" sx={{ color: '#64748b', mb: 1, fontWeight: 600 }}>
          {title}
        </Typography>
        <Typography variant="body2" sx={{ color: '#64748b', mb: 1 }}>
          {description}
        </Typography>
        <Typography variant="caption" sx={{ color: '#94a3b8' }}>
          {helpText}
        </Typography>
      </Box>
    );
  }
}
