import React, { Component, ReactNode } from 'react';
import { Box } from '@mui/material';

interface VideoPlayerErrorBoundaryProps {
  children: ReactNode;
  fallback?: ReactNode;
}

interface VideoPlayerErrorBoundaryState {
  hasError: boolean;
  errorMessage: string;
}

export class VideoPlayerErrorBoundary extends Component<
  VideoPlayerErrorBoundaryProps,
  VideoPlayerErrorBoundaryState
> {
  constructor(props: VideoPlayerErrorBoundaryProps) {
    super(props);
    this.state = {
      hasError: false,
      errorMessage: '',
    };
  }

  static getDerivedStateFromError(error: Error): VideoPlayerErrorBoundaryState {
    // Update state so the next render will show the fallback UI
    return {
      hasError: true,
      errorMessage: error.message || 'Unknown error',
    };
  }

  componentDidCatch(error: Error, errorInfo: React.ErrorInfo) {
    // Log error but don't crash the app
    if (error.message && error.message.includes('provider destroyed')) {
      console.warn('VideoPlayerErrorBoundary: Provider destroyed error suppressed');
    } else {
      console.error('VideoPlayerErrorBoundary caught error:', error, errorInfo);
    }
  }

  render() {
    if (this.state.hasError) {
      // Return custom fallback or default
      if (this.props.fallback) {
        return this.props.fallback;
      }

      return (
        <Box
          sx={{
            width: '100%',
            aspectRatio: '16/9',
            borderRadius: 2,
            overflow: 'hidden',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            bgcolor: '#f8fafc',
            color: '#64748b',
          }}
        >
          <p>Video player unavailable</p>
        </Box>
      );
    }

    return this.props.children;
  }
}
