import React, { Component } from 'react';
import {
  Box,
  Card,
  CardContent,
  Typography,
  Button,
  CircularProgress,
} from '@mui/material';
import {
  EventAvailable,
  Add,
  AccessTime,
} from '@mui/icons-material';
import FullCalendar from '@fullcalendar/react';
import dayGridPlugin from '@fullcalendar/daygrid';
import timeGridPlugin from '@fullcalendar/timegrid';
import interactionPlugin from '@fullcalendar/interaction';
import listPlugin from '@fullcalendar/list';

export class Timetable extends Component {
  constructor(props) {
    super(props);
    this.calendarRef = React.createRef();
    this.state = {
      lessons: [],
      isLoading: true,
      isDataLoaded: false,
    };
  }

  componentDidMount() {
    // Simulate data loading
    this.loadLessons();
  }

  loadLessons = () => {
    this.setState({ isLoading: true });
    
    // Simulate API call with timeout matching the progress duration (3 seconds)
    setTimeout(() => {
      // Simulate empty data - change this to add actual lessons
      // Example with data:
      // const sampleLessons = [
      //   {
      //     id: '1',
      //     title: 'Introduction to Algebra',
      //     start: '2025-10-15T10:00:00',
      //     end: '2025-10-15T11:00:00',
      //     instructor: 'Dr. Sarah Johnson',
      //     students: 25,
      //     type: 'class',
      //     backgroundColor: '#38aae1',
      //     borderColor: '#2c8fc9',
      //     textColor: '#ffffff',
      //   },
      // ];
      
      this.setState({
        lessons: [], // Empty to show empty state
        isLoading: false,
        isDataLoaded: true,
      });
    }, 3100);
  };

  handleDateClick = (arg) => {
    console.log('Date clicked:', arg.dateStr);
    // Add logic to open dialog for creating new lesson
  };

  handleEventClick = (clickInfo) => {
    console.log('Lesson clicked:', clickInfo.event.title);
    // Add logic to show lesson details or edit lesson
  };

  handleAddLesson = () => {
    console.log('Add lesson clicked');
    // Add logic to open create lesson dialog
  };

  renderLoadingState = () => {
    return (
      <Box
        sx={{
          display: 'flex',
          flexDirection: 'column',
          alignItems: 'center',
          justifyContent: 'center',
          minHeight: 500,
          textAlign: 'center',
          px: 3,
        }}
      >
        <CircularProgress
          size={80}
          thickness={4}
          sx={{
            color: '#38aae1',
            mb: 3,
          }}
        />
        <Box
          sx={{
            display: 'flex',
            alignItems: 'center',
            gap: 1.5,
            mb: 1,
          }}
        >
          <AccessTime
            sx={{
              fontSize: 28,
              color: '#38aae1',
            }}
          />
          <Typography
            variant="h5"
            sx={{
              color: '#25476a',
              fontWeight: 600,
            }}
          >
            Fetching lessons...
          </Typography>
        </Box>
        <Typography
          variant="body2"
          sx={{
            color: '#64748b',
            maxWidth: 350,
          }}
        >
          Please wait while we load your complete schedule
        </Typography>
      </Box>
    );
  };

  renderEmptyState = () => {
    return (
      <Box
        sx={{
          display: 'flex',
          flexDirection: 'column',
          alignItems: 'center',
          justifyContent: 'center',
          minHeight: 500,
          textAlign: 'center',
          px: 3,
        }}
      >
        <Box
          sx={{
            width: 120,
            height: 120,
            borderRadius: '50%',
            backgroundColor: '#e3f2fd',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            mb: 3,
          }}
        >
          <EventAvailable
            sx={{
              fontSize: 64,
              color: '#38aae1',
            }}
          />
        </Box>
        
        <Typography
          variant="h5"
          sx={{
            color: '#25476a',
            fontWeight: 700,
            mb: 2,
          }}
        >
          No lessons scheduled
        </Typography>
        
        <Typography
          variant="body1"
          sx={{
            color: '#64748b',
            mb: 3,
            lineHeight: 1.6,
            maxWidth: 400,
          }}
        >
          There are no lessons scheduled. Click "Add Lesson" to create your first lesson.
        </Typography>
        
        <Button
          variant="contained"
          startIcon={<Add />}
          onClick={this.handleAddLesson}
          sx={{
            backgroundColor: '#38aae1',
            color: '#ffffff',
            textTransform: 'none',
            fontWeight: 600,
            px: 4,
            py: 1.5,
            borderRadius: 2,
            boxShadow: '0 4px 12px rgba(56, 170, 225, 0.3)',
            '&:hover': {
              backgroundColor: '#2c8fc9',
              boxShadow: '0 6px 16px rgba(56, 170, 225, 0.4)',
            },
          }}
        >
          Add Lesson
        </Button>
      </Box>
    );
  };

  renderCalendar = () => {
    const { lessons } = this.state;

    return (
      <Box
        sx={{
          '& .fc': {
            fontFamily: '"Inter", "Roboto", "Helvetica", "Arial", sans-serif',
          },
          '& .fc-toolbar-title': {
            color: '#25476a',
            fontWeight: 700,
            fontSize: '1.5rem',
          },
          '& .fc-button': {
            backgroundColor: '#38aae1',
            borderColor: '#38aae1',
            textTransform: 'none',
            fontWeight: 600,
            '&:hover': {
              backgroundColor: '#2c8fc9',
              borderColor: '#2c8fc9',
            },
            '&:focus': {
              boxShadow: '0 0 0 0.2rem rgba(56, 170, 225, 0.5)',
            },
            '&.fc-button-active': {
              backgroundColor: '#25476a',
              borderColor: '#25476a',
            },
          },
          '& .fc-col-header-cell': {
            backgroundColor: '#f8fafc',
            borderColor: '#e2e8f0',
            padding: '12px 0',
          },
          '& .fc-col-header-cell-cushion': {
            color: '#64748b',
            fontWeight: 600,
            fontSize: '0.875rem',
          },
          '& .fc-daygrid-day': {
            '&:hover': {
              backgroundColor: '#f8fafc',
            },
          },
          '& .fc-daygrid-day-number': {
            color: '#25476a',
            fontWeight: 500,
            padding: '8px',
          },
          '& .fc-daygrid-day.fc-day-today': {
            backgroundColor: '#e3f2fd !important',
          },
          '& .fc-daygrid-day.fc-day-today .fc-daygrid-day-number': {
            color: '#38aae1',
            fontWeight: 700,
          },
          '& .fc-event': {
            borderRadius: '4px',
            padding: '2px 4px',
            fontSize: '0.875rem',
            fontWeight: 500,
            cursor: 'pointer',
          },
          '& .fc-daygrid-event-dot': {
            borderColor: '#38aae1',
          },
          '& .fc-list-event:hover td': {
            backgroundColor: '#f8fafc',
          },
          '& .fc-list-day-cushion': {
            backgroundColor: '#25476a',
            color: '#ffffff',
          },
          '& .fc-timegrid-slot': {
            height: '3em',
          },
          '& .fc-timegrid-slot-label': {
            color: '#64748b',
            fontSize: '0.875rem',
          },
        }}
      >
        <FullCalendar
          ref={this.calendarRef}
          plugins={[dayGridPlugin, timeGridPlugin, interactionPlugin, listPlugin]}
          initialView="dayGridMonth"
          headerToolbar={{
            left: 'prev,next today',
            center: 'title',
            right: 'dayGridMonth,timeGridWeek,timeGridDay,listWeek',
          }}
          buttonText={{
            today: 'Today',
            month: 'Month',
            week: 'Week',
            day: 'Day',
            list: 'Agenda',
          }}
          events={lessons}
          dateClick={this.handleDateClick}
          eventClick={this.handleEventClick}
          editable={true}
          selectable={true}
          selectMirror={true}
          dayMaxEvents={true}
          weekends={true}
          height="auto"
          contentHeight={600}
        />
      </Box>
    );
  };

  render() {
    const { isLoading, isDataLoaded, lessons } = this.state;

    return (
      <Box sx={{ p: 4, backgroundColor: '#f8fafc', minHeight: '100vh' }}>
        <Box
          sx={{
            display: 'flex',
            justifyContent: 'space-between',
            alignItems: 'center',
            mb: 3,
          }}
        >
          <Box>
            <Typography
              variant="h4"
              sx={{
                color: '#25476a',
                fontWeight: 700,
                mb: 0.5,
              }}
            >
              Timetable
            </Typography>
            <Typography
              variant="body2"
              sx={{
                color: '#64748b',
              }}
            >
              Manage and view all scheduled lessons
            </Typography>
          </Box>

          {isDataLoaded && lessons.length > 0 && (
            <Button
              variant="contained"
              startIcon={<Add />}
              onClick={this.handleAddLesson}
              sx={{
                backgroundColor: '#38aae1',
                color: '#ffffff',
                textTransform: 'none',
                fontWeight: 600,
                px: 3,
                py: 1.5,
                borderRadius: 2,
                boxShadow: '0 4px 12px rgba(56, 170, 225, 0.3)',
                '&:hover': {
                  backgroundColor: '#2c8fc9',
                  boxShadow: '0 6px 16px rgba(56, 170, 225, 0.4)',
                },
              }}
            >
              Add Lesson
            </Button>
          )}
        </Box>

        <Card
          sx={{
            boxShadow: '0 4px 20px rgba(37, 71, 106, 0.08)',
            borderRadius: 3,
            border: '1px solid rgba(37, 71, 106, 0.08)',
            overflow: 'hidden',
          }}
        >
          <Box
            sx={{
              background: 'linear-gradient(135deg, #38aae1 0%, #2c8fc9 100%)',
              p: 2.5,
              borderBottom: '1px solid rgba(255,255,255,0.2)',
            }}
          >
            <Typography
              variant="h6"
              sx={{
                color: '#ffffff',
                fontWeight: 600,
                display: 'flex',
                alignItems: 'center',
                gap: 1,
              }}
            >
              <EventAvailable sx={{ fontSize: 24 }} />
              Full Schedule Calendar
            </Typography>
            <Typography
              variant="body2"
              sx={{
                color: 'rgba(255,255,255,0.9)',
                mt: 0.5,
              }}
            >
              View and manage all lessons across all dates
            </Typography>
          </Box>
          <CardContent sx={{ p: 3 }}>
            {isLoading && this.renderLoadingState()}
            {!isLoading && isDataLoaded && lessons.length === 0 && this.renderEmptyState()}
            {!isLoading && isDataLoaded && lessons.length > 0 && this.renderCalendar()}
          </CardContent>
        </Card>
      </Box>
    );
  }
}
