import React, { useState } from 'react';
import {
  Box,
  Typography,
  Button,
  Paper,
  Grid,
  Card,
  CardContent,
  TextField,
  InputAdornment,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Chip,
  Avatar,
  IconButton,
  Menu,
  MenuItem,
  Pagination,
  Select,
  FormControl,
  InputLabel,
} from '@mui/material';
import {
  ArrowBack,
  Add,
  Search,
  MoreVert,
  People,
  AccessTime,
  AttachMoney,
  Star,
  TrendingUp,
  FilterList,
  Schedule,
} from '@mui/icons-material';
import { AddMentorDialog } from './AddMentorDialog.jsx';
import { AvailabilityScheduleDialog } from './AvailabilityScheduleDialog.jsx';
import { AssignMentorsToCourseDialog } from './AssignMentorsToCourseDialog.jsx';
import { ViewScheduleDialog } from './ViewScheduleDialog.jsx';

const mockMentors = [
  {
    id: '1',
    name: 'Sarah Johnson',
    specialization: 'Mathematics',
    status: 'Active',
    gender: 'Female',
    ratePerHour: 70,
    rating: 4.5,
    totalHours: 245,
    students: 18,
    monthlyEarnings: 2240,
    subjects: ['Algebra', 'Geometry', 'Calculus'],
  },
  {
    id: '2',
    name: 'Michael Chen',
    specialization: 'Science',
    status: 'Active',
    gender: 'Male',
    ratePerHour: 75,
    rating: 4.7,
    totalHours: 189,
    students: 14,
    monthlyEarnings: 2100,
    subjects: ['Physics', 'Chemistry', 'Biology'],
  },
  {
    id: '3',
    name: 'Emily Rodriguez',
    specialization: 'English',
    status: 'Active',
    gender: 'Female',
    ratePerHour: 65,
    rating: 4.8,
    totalHours: 156,
    students: 12,
    monthlyEarnings: 1560,
    subjects: ['Writing', 'Literature', 'Grammar'],
  },
  {
    id: '4',
    name: 'David Kim',
    specialization: 'Mathematics',
    status: 'Inactive',
    gender: 'Male',
    ratePerHour: 80,
    rating: 4.6,
    totalHours: 98,
    students: 8,
    monthlyEarnings: 0,
    subjects: ['Statistics', 'Calculus'],
  },
  {
    id: '5',
    name: 'Lisa Thompson',
    specialization: 'Science',
    status: 'Active',
    gender: 'Female',
    ratePerHour: 72,
    rating: 4.9,
    totalHours: 134,
    students: 15,
    monthlyEarnings: 1890,
    subjects: ['Biology', 'Environmental Science'],
  },
];

// Mock classes with courses for self-paced learning
const mockClasses = [
  {
    id: 1,
    name: 'Grade 3 Mathematics',
    coursesCount: 3,
    students: 45,
    passType: 'Meeting',
    status: 'Published',
    courses: [
      {
        id: '1',
        title: 'Introduction to Algebra',
        description: 'Learn the basics of algebraic expressions and equations.',
        category: 'Mathematics',
        supervisors: [],
      },
      {
        id: '2',
        title: 'Geometry Fundamentals',
        description: 'Explore shapes, angles, and spatial relationships.',
        category: 'Mathematics',
        supervisors: [],
      },
      {
        id: '3',
        title: 'Basic Statistics',
        description: 'Introduction to data collection and analysis.',
        category: 'Mathematics',
        supervisors: [],
      },
    ],
  },
  {
    id: 2,
    name: 'Grade 4 Science',
    coursesCount: 2,
    students: 32,
    passType: 'Learning',
    status: 'Published',
    courses: [
      {
        id: '4',
        title: 'Introduction to Physics',
        description: 'Discover the fundamentals of motion, energy, and forces.',
        category: 'Science',
        supervisors: [],
      },
      {
        id: '5',
        title: 'Life Sciences',
        description: 'Study living organisms and their environments.',
        category: 'Science',
        supervisors: [],
      },
    ],
  },
  {
    id: 3,
    name: 'Grade 5 English',
    coursesCount: 4,
    students: 58,
    passType: 'Meeting',
    status: 'Draft',
    courses: [
      {
        id: '6',
        title: 'Creative Writing Workshop',
        description: 'Enhance your writing skills through guided exercises and feedback.',
        category: 'English',
        supervisors: [],
      },
      {
        id: '7',
        title: 'Literature Analysis',
        description: 'Learn to analyze and interpret literary works.',
        category: 'English',
        supervisors: [],
      },
      {
        id: '8',
        title: 'Grammar Mastery',
        description: 'Master the rules of English grammar and punctuation.',
        category: 'English',
        supervisors: [],
      },
      {
        id: '9',
        title: 'Public Speaking',
        description: 'Develop confidence in oral communication.',
        category: 'English',
        supervisors: [],
      },
    ],
  },
];

export function TechnicalMentors() {
  const [mentors, setMentors] = useState(mockMentors);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('All Status');
  const [specializationFilter, setSpecializationFilter] = useState('All Specializations');
  const [currentPage, setCurrentPage] = useState(1);
  const [anchorEl, setAnchorEl] = useState(null);
  const [selectedMentor, setSelectedMentor] = useState(null);
  const [openAddMentor, setOpenAddMentor] = useState(false);
  const [openAvailabilityDialog, setOpenAvailabilityDialog] = useState(false);
  const [mentorForSchedule, setMentorForSchedule] = useState(null);
  const [openAssignCourseDialog, setOpenAssignCourseDialog] = useState(false);
  const [classes, setClasses] = useState(mockClasses);
  const [openViewScheduleDialog, setOpenViewScheduleDialog] = useState(false);
  const [mentorForViewSchedule, setMentorForViewSchedule] = useState(null);
  
  const itemsPerPage = 5;

  const handleMenuClick = (event, mentor) => {
    setAnchorEl(event.currentTarget);
    setSelectedMentor(mentor);
  };

  const handleMenuClose = () => {
    setAnchorEl(null);
    setSelectedMentor(null);
  };

  // Filter mentors
  const filteredMentors = mentors.filter(mentor => {
    const matchesSearch = mentor.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         mentor.specialization.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesStatus = statusFilter === 'All Status' || mentor.status === statusFilter;
    const matchesSpecialization = specializationFilter === 'All Specializations' || 
                                  mentor.specialization === specializationFilter;
    
    return matchesSearch && matchesStatus && matchesSpecialization;
  });

  // Pagination
  const totalPages = Math.ceil(filteredMentors.length / itemsPerPage);
  const startIndex = (currentPage - 1) * itemsPerPage;
  const paginatedMentors = filteredMentors.slice(startIndex, startIndex + itemsPerPage);

  const handlePageChange = (event, value) => {
    setCurrentPage(value);
  };

  const handleAddMentor = (mentorData) => {
    const newMentor = {
      id: Date.now().toString(),
      name: mentorData.name,
      specialization: 'Programming', // Default specialization
      status: 'Active',
      gender: 'Male', // Could be determined from form data
      ratePerHour: 65,
      rating: 0,
      totalHours: 0,
      students: 0,
      monthlyEarnings: 0,
      subjects: [],
    };
    setMentors(prev => [...prev, newMentor]);
  };

  const handleAssignToCourse = (classId, courseIds, mentorIds) => {
    // Update multiple courses with new supervisors in the appropriate class
    setClasses(prev => prev.map(classItem => {
      if (classItem.id === classId) {
        return {
          ...classItem,
          courses: classItem.courses.map(course => 
            courseIds.includes(course.id)
              ? { ...course, supervisors: [...new Set([...(course.supervisors || []), ...mentorIds])] }
              : course
          )
        };
      }
      return classItem;
    }));
  };

  const handleSetAvailability = (mentor) => {
    setMentorForSchedule(mentor);
    setOpenAvailabilityDialog(true);
    handleMenuClose();
  };

  const handleViewSchedule = (mentor) => {
    setMentorForViewSchedule(mentor);
    setOpenViewScheduleDialog(true);
    handleMenuClose();
  };



  return (
    <Box sx={{ p: 4, bgcolor: '#f8fafc', minHeight: '100vh' }}>
      {/* Header */}
      <Paper sx={{ p: 4, mb: 4, borderRadius: 3, boxShadow: '0 4px 20px rgba(0,0,0,0.08)' }}>
        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
          <Box>
            <Typography variant="h3" sx={{ fontWeight: 700, color: '#25476a', mb: 1 }}>
              Technical Mentor Management
            </Typography>
            <Typography variant="body1" color="text.secondary" sx={{ fontSize: '1.1rem' }}>
              Manage technical mentors who assist students in self-paced courses
            </Typography>
          </Box>
          <Box sx={{ display: 'flex', gap: 2 }}>
            <Button
              variant="outlined"
              startIcon={<Schedule />}
              onClick={() => setOpenAssignCourseDialog(true)}
              sx={{ 
                borderColor: '#feb139',
                color: '#feb139',
                '&:hover': { 
                  borderColor: '#feb139',
                  bgcolor: 'rgba(254, 177, 57, 0.1)',
                },
                px: 3,
                py: 1.5,
                borderRadius: 2,
                fontWeight: 600,
              }}
            >
              Assign to Courses
            </Button>
            <Button
              variant="contained"
              startIcon={<Add />}
              onClick={() => setOpenAddMentor(true)}
              sx={{ 
                bgcolor: '#38aae1', 
                '&:hover': { bgcolor: '#2e8bc4' },
                px: 3,
                py: 1.5,
                borderRadius: 2,
                boxShadow: '0 4px 12px rgba(56, 170, 225, 0.3)',
              }}
            >
              Add Mentor
            </Button>
          </Box>
        </Box>
      </Paper>

      {/* Filters and Search */}
      <Paper sx={{ p: 3, mb: 3, borderRadius: 3, boxShadow: '0 4px 20px rgba(0,0,0,0.08)' }}>
        <Grid container spacing={3} alignItems="center">
          <Grid item xs={12} md={4}>
            <TextField
              placeholder="Search mentors..."
              fullWidth
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              InputProps={{
                startAdornment: (
                  <InputAdornment position="start">
                    <Search />
                  </InputAdornment>
                ),
              }}
            />
          </Grid>
          <Grid item xs={12} md={3}>
            <FormControl fullWidth>
              <InputLabel>Status</InputLabel>
              <Select
                value={statusFilter}
                label="Status"
                onChange={(e) => setStatusFilter(e.target.value)}
              >
                <MenuItem value="All Status">All Status</MenuItem>
                <MenuItem value="Active">Active</MenuItem>
                <MenuItem value="Inactive">Inactive</MenuItem>
              </Select>
            </FormControl>
          </Grid>
          <Grid item xs={12} md={3}>
            <FormControl fullWidth>
              <InputLabel>Specialization</InputLabel>
              <Select
                value={specializationFilter}
                label="Specialization"
                onChange={(e) => setSpecializationFilter(e.target.value)}
              >
                <MenuItem value="All Specializations">All Specializations</MenuItem>
                <MenuItem value="Mathematics">Mathematics</MenuItem>
                <MenuItem value="Science">Science</MenuItem>
                <MenuItem value="English">English</MenuItem>
              </Select>
            </FormControl>
          </Grid>
          <Grid item xs={12} md={2}>
            <Typography variant="body2" color="text.secondary">
              {filteredMentors.length} mentors found
            </Typography>
          </Grid>
        </Grid>
      </Paper>

      {/* Mentors Table */}
      <TableContainer component={Paper} sx={{ borderRadius: 3, boxShadow: '0 4px 20px rgba(0,0,0,0.08)' }}>
        <Table>
          <TableHead sx={{ bgcolor: '#f8fafc' }}>
            <TableRow>
              <TableCell sx={{ fontWeight: 'bold', color: '#25476a' }}>Mentor</TableCell>
              <TableCell sx={{ fontWeight: 'bold', color: '#25476a' }}>Status</TableCell>
              <TableCell sx={{ fontWeight: 'bold', color: '#25476a' }}>Rating</TableCell>
              <TableCell sx={{ fontWeight: 'bold', color: '#25476a' }}>Students</TableCell>
              <TableCell sx={{ fontWeight: 'bold', color: '#25476a' }}>Actions</TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {paginatedMentors.map((mentor) => (
              <TableRow key={mentor.id} hover>
                <TableCell>
                  <Box sx={{ display: 'flex', alignItems: 'center' }}>
                    <Avatar 
                      sx={{ 
                        bgcolor: mentor.gender === 'Female' ? '#e91e63' : '#2196f3',
                        mr: 2,
                        width: 40,
                        height: 40,
                      }}
                    >
                      {mentor.name.charAt(0)}
                    </Avatar>
                    <Box>
                      <Typography variant="body1" sx={{ fontWeight: 600 }}>
                        {mentor.name}
                      </Typography>
                      <Typography variant="body2" color="text.secondary">
                        {mentor.specialization}
                      </Typography>
                    </Box>
                  </Box>
                </TableCell>
                <TableCell>
                  <Chip
                    label={mentor.status}
                    size="small"
                    color={mentor.status === 'Active' ? 'success' : 'default'}
                    variant={mentor.status === 'Active' ? 'filled' : 'outlined'}
                  />
                  <Chip
                    label={mentor.gender}
                    size="small"
                    color={mentor.gender === 'Female' ? 'secondary' : 'primary'}
                    variant="outlined"
                    sx={{ ml: 1 }}
                  />
                </TableCell>
                <TableCell>
                  <Box sx={{ display: 'flex', alignItems: 'center' }}>
                    <Star sx={{ color: '#ff9800', fontSize: 16, mr: 0.5 }} />
                    <Typography variant="body1" sx={{ fontWeight: 600 }}>
                      {mentor.rating}/5.0
                    </Typography>
                  </Box>
                </TableCell>
                <TableCell>
                  <Typography variant="body1">
                    {mentor.students}
                  </Typography>
                </TableCell>
                <TableCell>
                  <IconButton
                    size="small"
                    onClick={(e) => handleMenuClick(e, mentor)}
                  >
                    <MoreVert />
                  </IconButton>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </TableContainer>

      {/* Pagination */}
      <Box sx={{ display: 'flex', justifyContent: 'center', mt: 4 }}>
        <Pagination
          count={totalPages}
          page={currentPage}
          onChange={handlePageChange}
          color="primary"
          size="large"
        />
      </Box>

      {/* Action Menu */}
      <Menu
        anchorEl={anchorEl}
        open={Boolean(anchorEl)}
        onClose={handleMenuClose}
      >
        <MenuItem onClick={handleMenuClose}>
          <People sx={{ mr: 1, fontSize: 20 }} />
          View Details
        </MenuItem>
        <MenuItem onClick={handleMenuClose}>
          <Star sx={{ mr: 1, fontSize: 20 }} />
          Edit Mentor
        </MenuItem>
        <MenuItem onClick={() => handleSetAvailability(selectedMentor)}>
          <Schedule sx={{ mr: 1, fontSize: 20 }} />
          Set Availability
        </MenuItem>
        <MenuItem onClick={() => handleViewSchedule(selectedMentor)}>
          <AccessTime sx={{ mr: 1, fontSize: 20 }} />
          View Schedule
        </MenuItem>
        <MenuItem onClick={handleMenuClose}>
          <TrendingUp sx={{ mr: 1, fontSize: 20 }} />
          Enroll to Self Paced Course
        </MenuItem>
        <MenuItem onClick={handleMenuClose} sx={{ color: '#ef4444' }}>
          <FilterList sx={{ mr: 1, fontSize: 20 }} />
          Deactivate
        </MenuItem>
      </Menu>

      {/* Add Mentor Dialog */}
      <AddMentorDialog
        open={openAddMentor}
        onClose={() => setOpenAddMentor(false)}
        onSubmit={handleAddMentor}
      />

      {/* Availability Schedule Dialog */}
      <AvailabilityScheduleDialog
        open={openAvailabilityDialog}
        onClose={() => {
          setOpenAvailabilityDialog(false);
          setMentorForSchedule(null);
        }}
        mentor={mentorForSchedule}
      />

      {/* Assign Mentors to Course Dialog */}
      <AssignMentorsToCourseDialog
        open={openAssignCourseDialog}
        onClose={() => setOpenAssignCourseDialog(false)}
        onAssign={handleAssignToCourse}
        mentors={mentors}
        classes={classes}
      />

      {/* View Schedule Dialog */}
      <ViewScheduleDialog
        open={openViewScheduleDialog}
        onClose={() => {
          setOpenViewScheduleDialog(false);
          setMentorForViewSchedule(null);
        }}
        mentor={mentorForViewSchedule}
      />
    </Box>
  );
}