import React, { Component } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  Box,
  Typography,
  TextField,
  Grid,
  IconButton,
  Tabs,
  Tab,
  FormControl,
  Select,
  MenuItem,
  Chip,
  Card,
  CardContent,
  Avatar,
  Collapse,
  RadioGroup,
  FormControlLabel,
  Radio,
} from '@mui/material';
import {
  Close,
  Delete,
  Add,
  Save,
  CheckCircle,
  PlayCircleOutline,
  Assignment,
  SportsEsports,
  Edit,
  VideoLibrary,
  MenuBook,
  School,
  ExpandMore,
  ExpandLess,
  CloudUpload,
} from '@mui/icons-material';
import { VideoUploadDragger } from './VideoUploadDragger';
import { ActivityVideoUploader } from './ActivityVideoUploader';
import { VidstackPlayer } from './VidstackPlayer';

interface SessionData {
  id: string;
  title: string;
  duration: number;
  description: string;
  objectives: string[];
  introVideo: { title: string; duration: number; url: string };
  assignmentVideo: { title: string; duration: number; url: string };
  activityVideos: { title: string; duration: number; url: string }[];
  resources: { name: string; type: 'pdf' | 'doc' | 'ppt' | 'youtube'; url: string }[];
  isCompleted: boolean;
}

interface LessonPlan {
  id: string;
  title: string;
  duration: number;
  description: string;
  objectives: string[];
  introVideo: { title: string; duration: number; url: string };
  assignmentVideo: { title: string; duration: number; url: string };
  activityVideos: { title: string; duration: number; url: string }[];
  resources: { name: string; type: 'pdf' | 'doc' | 'ppt' | 'youtube'; url: string }[];
}

interface SingleSessionEditorProps {
  open: boolean;
  lesson: LessonPlan;
  sessionNumber: number;
  onSave: (lesson: LessonPlan) => void;
  onClose: () => void;
}

interface SingleSessionEditorState {
  session: SessionData;
  activeTab: number;
  newObjective: string;
  newResource: { name: string; type: 'pdf' | 'doc' | 'ppt' | 'youtube'; url: string };
  resourceInputMethod: 'upload' | 'url';
  expandedSections: { [key: string]: boolean };
}

const convertLessonToSession = (lesson: LessonPlan, sessionNumber: number): SessionData => ({
  id: `session-${sessionNumber}`,
  title: lesson.title || `Session ${sessionNumber}`,
  duration: lesson.duration || 60,
  description: lesson.description || `Content for session ${sessionNumber}`,
  objectives: lesson.objectives || [],
  introVideo: lesson.introVideo || { title: '', duration: 0, url: '' },
  assignmentVideo: lesson.assignmentVideo || { title: '', duration: 0, url: '' },
  activityVideos: lesson.activityVideos || [],
  resources: lesson.resources || [],
  isCompleted: false
});

export class SingleSessionEditor extends Component<SingleSessionEditorProps, SingleSessionEditorState> {
  constructor(props: SingleSessionEditorProps) {
    super(props);
    
    this.state = {
      session: convertLessonToSession(props.lesson, props.sessionNumber),
      activeTab: 0,
      newObjective: '',
      newResource: {
        name: '',
        type: 'pdf',
        url: ''
      },
      resourceInputMethod: 'url',
      expandedSections: {
        details: true,
        outcomes: true,
        intro: true,
        assignment: true,
        activities: true
      }
    };
  }

  componentDidUpdate(prevProps: SingleSessionEditorProps) {
    if (prevProps.lesson !== this.props.lesson && this.props.open) {
      this.setState({ 
        session: convertLessonToSession(this.props.lesson, this.props.sessionNumber)
      });
    }
  }

  updateSession = (updates: Partial<SessionData>) => {
    this.setState({
      session: { ...this.state.session, ...updates }
    });
  };

  handleSave = () => {
    const { session } = this.state;
    const updatedLesson: LessonPlan = {
      id: this.props.lesson.id,
      title: session.title,
      duration: session.duration,
      description: session.description,
      objectives: session.objectives,
      introVideo: session.introVideo,
      assignmentVideo: session.assignmentVideo,
      activityVideos: session.activityVideos,
      resources: session.resources
    };
    this.props.onSave(updatedLesson);
    this.props.onClose();
  };

  addObjective = () => {
    const { newObjective, session } = this.state;
    
    if (newObjective.trim()) {
      this.updateSession({
        objectives: [...session.objectives, newObjective]
      });
      this.setState({ newObjective: '' });
    }
  };

  removeObjective = (index: number) => {
    const { session } = this.state;
    this.updateSession({
      objectives: session.objectives.filter((_, i) => i !== index)
    });
  };

  addActivityVideo = () => {
    const { session } = this.state;
    this.updateSession({
      activityVideos: [...session.activityVideos, { title: '', duration: 0, url: '' }]
    });
  };

  removeActivityVideo = (index: number) => {
    const { session } = this.state;
    this.updateSession({
      activityVideos: session.activityVideos.filter((_, i) => i !== index)
    });
  };

  updateActivityVideo = (index: number, field: string, value: any) => {
    const { session } = this.state;
    const newActivities = [...session.activityVideos];
    newActivities[index] = { ...newActivities[index], [field]: value };
    this.updateSession({
      activityVideos: newActivities
    });
  };

  handleResourceFileUpload = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (file) {
      // In a real application, you would upload the file to a server
      // For now, we'll create a local URL
      const fileUrl = URL.createObjectURL(file);
      this.setState({
        newResource: {
          ...this.state.newResource,
          url: fileUrl,
          name: this.state.newResource.name || file.name
        }
      });
    }
  };

  addResource = () => {
    const { newResource, session } = this.state;
    
    if (newResource.name.trim()) {
      this.updateSession({
        resources: [...session.resources, { ...newResource }]
      });
      this.setState({
        newResource: {
          name: '',
          type: 'pdf',
          url: ''
        },
        resourceInputMethod: 'url'
      });
    }
  };

  removeResource = (index: number) => {
    const { session } = this.state;
    this.updateSession({
      resources: session.resources.filter((_, i) => i !== index)
    });
  };

  toggleComplete = () => {
    this.updateSession({
      isCompleted: !this.state.session.isCompleted
    });
  };

  toggleSection = (section: string) => {
    this.setState({
      expandedSections: {
        ...this.state.expandedSections,
        [section]: !this.state.expandedSections[section]
      }
    });
  };

  render() {
    const { open, onClose, sessionNumber } = this.props;
    const { session, activeTab, newObjective, newResource, expandedSections } = this.state;

    return (
      <Dialog 
        open={open} 
        onClose={onClose} 
        maxWidth={false}
        fullWidth
        PaperProps={{
          sx: { 
            borderRadius: 4,
            maxWidth: '90vw',
            maxHeight: '95vh',
            width: '90vw',
            height: '95vh',
            background: 'linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%)',
          }
        }}
      >
        {/* Enhanced Header */}
        <DialogTitle sx={{ 
          p: 0, 
          background: 'linear-gradient(135deg, #25476a 0%, #1e3a52 100%)',
          color: 'white',
          position: 'relative',
          overflow: 'hidden'
        }}>
          <Box sx={{
            position: 'absolute',
            top: 0,
            right: 0,
            width: 150,
            height: 150,
            background: 'radial-gradient(circle, rgba(56, 170, 225, 0.3) 0%, transparent 70%)',
            borderRadius: '50%',
            transform: 'translate(50%, -50%)'
          }} />
          
          <Box sx={{ 
            display: 'flex', 
            alignItems: 'center', 
            justifyContent: 'space-between',
            p: 4,
            position: 'relative'
          }}>
            <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
              <Avatar sx={{ bgcolor: '#38aae1', width: 48, height: 48 }}>
                <School />
              </Avatar>
              <Box>
                <Typography variant="h4" sx={{ fontWeight: 700, mb: 0.5 }}>
                  Edit Session {sessionNumber}
                </Typography>
                <Typography variant="body1" sx={{ opacity: 0.9 }}>
                  {session.title || `Session ${sessionNumber}`}
                </Typography>
              </Box>
            </Box>
            
            <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
              <Chip 
                icon={session.isCompleted ? <CheckCircle /> : <Edit />}
                label={session.isCompleted ? 'Complete' : 'In Progress'} 
                sx={{
                  bgcolor: session.isCompleted ? 'rgba(76, 175, 80, 0.2)' : 'rgba(255,255,255,0.2)',
                  color: 'white',
                  border: '1px solid rgba(255,255,255,0.3)',
                  fontWeight: 600
                }}
              />
              <IconButton 
                onClick={onClose} 
                sx={{ 
                  color: 'white', 
                  bgcolor: 'rgba(255,255,255,0.1)',
                  '&:hover': { bgcolor: 'rgba(255,255,255,0.2)' }
                }}
              >
                <Close />
              </IconButton>
            </Box>
          </Box>
        </DialogTitle>

        <DialogContent sx={{ p: 0, display: 'flex', flexDirection: 'column', height: '100%' }}>
          {/* Enhanced Tabs */}
          <Box sx={{ borderBottom: '1px solid #e2e8f0', bgcolor: 'white', boxShadow: '0 2px 8px rgba(0,0,0,0.05)' }}>
            <Tabs 
              value={activeTab} 
              onChange={(e, newValue) => this.setState({ activeTab: newValue })}
              sx={{ 
                px: 4,
                '& .MuiTab-root': {
                  textTransform: 'none',
                  fontWeight: 600,
                  fontSize: '1rem',
                  color: '#64748b',
                  minHeight: 60,
                  '&.Mui-selected': {
                    color: '#38aae1',
                  }
                },
                '& .MuiTabs-indicator': {
                  backgroundColor: '#38aae1',
                  height: 3,
                  borderRadius: '3px 3px 0 0'
                }
              }}
            >
              <Tab 
                label="Overview" 
                icon={<Edit />} 
                iconPosition="start"
                sx={{ mr: 2 }}
              />
              <Tab 
                label="Videos" 
                icon={<VideoLibrary />} 
                iconPosition="start"
                sx={{ mr: 2 }}
              />
              <Tab 
                label="Resources" 
                icon={<MenuBook />} 
                iconPosition="start"
              />
            </Tabs>
          </Box>

          {/* Tab Content */}
          <Box sx={{ flex: 1, p: 4, overflow: 'auto' }}>
            {/* Overview Tab */}
            {activeTab === 0 && (
              <Box sx={{ maxWidth: 800 }}>
                {/* Session Details Card */}
                <Card sx={{ mb: 4, borderRadius: 3, overflow: 'hidden' }}>
                  <Box 
                    onClick={() => this.toggleSection('details')}
                    sx={{ 
                      p: 3, 
                      bgcolor: '#f8fafc', 
                      borderBottom: '1px solid #e2e8f0',
                      cursor: 'pointer',
                      display: 'flex',
                      alignItems: 'center',
                      justifyContent: 'space-between',
                      '&:hover': { bgcolor: '#f1f5f9' }
                    }}
                  >
                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                      <Avatar sx={{ bgcolor: '#38aae1', width: 40, height: 40 }}>
                        <Edit />
                      </Avatar>
                      <Box>
                        <Typography variant="h6" sx={{ fontWeight: 700, color: '#25476a' }}>
                          Lesson Topic & Details
                        </Typography>
                        <Typography variant="body2" sx={{ color: '#64748b' }}>
                          Basic lesson information and structure
                        </Typography>
                      </Box>
                    </Box>
                    {expandedSections.details ? <ExpandLess /> : <ExpandMore />}
                  </Box>
                  
                  <Collapse in={expandedSections.details}>
                    <CardContent sx={{ p: 4 }}>
                      <Grid container spacing={3}>
                        <Grid item xs={12}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                            Lesson Topic
                          </Typography>
                          <TextField
                            fullWidth
                            value={session.title}
                            onChange={(e) => this.updateSession({ title: e.target.value })}
                            placeholder="Enter lesson topic"
                            size="small"
                            sx={{
                              '& .MuiOutlinedInput-root': {
                                bgcolor: 'white',
                                borderRadius: 2,
                              }
                            }}
                          />
                        </Grid>
                        <Grid item xs={12}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                            Lesson Sub Topic
                          </Typography>
                          <TextField
                            fullWidth
                            multiline
                            rows={3}
                            value={session.description}
                            onChange={(e) => this.updateSession({ description: e.target.value })}
                            placeholder="Describe the specific sub topics covered in this lesson"
                            size="small"
                            sx={{
                              '& .MuiOutlinedInput-root': {
                                bgcolor: 'white',
                                borderRadius: 2,
                              }
                            }}
                          />
                        </Grid>
                        <Grid item xs={12} md={4}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                            Duration (minutes)
                          </Typography>
                          <TextField
                            fullWidth
                            type="number"
                            value={session.duration}
                            onChange={(e) => this.updateSession({ duration: parseInt(e.target.value) || 0 })}
                            size="small"
                            sx={{
                              '& .MuiOutlinedInput-root': {
                                bgcolor: 'white',
                                borderRadius: 2,
                              }
                            }}
                          />
                        </Grid>
                      </Grid>
                    </CardContent>
                  </Collapse>
                </Card>

                {/* Desired Outcomes Card */}
                <Card sx={{ borderRadius: 3, overflow: 'hidden' }}>
                  <Box 
                    onClick={() => this.toggleSection('outcomes')}
                    sx={{ 
                      p: 3, 
                      bgcolor: '#f8fafc', 
                      borderBottom: '1px solid #e2e8f0',
                      cursor: 'pointer',
                      display: 'flex',
                      alignItems: 'center',
                      justifyContent: 'space-between',
                      '&:hover': { bgcolor: '#f1f5f9' }
                    }}
                  >
                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                      <Avatar sx={{ bgcolor: '#4caf50', width: 40, height: 40 }}>
                        <CheckCircle />
                      </Avatar>
                      <Box>
                        <Typography variant="h6" sx={{ fontWeight: 700, color: '#25476a' }}>
                          Desired Outcomes
                        </Typography>
                        <Typography variant="body2" sx={{ color: '#64748b' }}>
                          Learning goals and objectives for this session
                        </Typography>
                      </Box>
                    </Box>
                    {expandedSections.outcomes ? <ExpandLess /> : <ExpandMore />}
                  </Box>
                  
                  <Collapse in={expandedSections.outcomes}>
                    <CardContent sx={{ p: 4 }}>
                      <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 3 }}>
                        <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                          Desired Outcomes ({session.objectives.length})
                        </Typography>
                        <Button
                          startIcon={<Add />}
                          variant="contained"
                          size="small"
                          onClick={this.addObjective}
                          disabled={!newObjective.trim()}
                          sx={{ 
                            bgcolor: '#4caf50',
                            textTransform: 'none',
                            borderRadius: 2,
                            '&:hover': { bgcolor: '#45a049' }
                          }}
                        >
                          Add Outcome
                        </Button>
                      </Box>
                      
                      <TextField
                        fullWidth
                        placeholder="Enter desired learning outcome for this session"
                        value={newObjective}
                        onChange={(e) => this.setState({ newObjective: e.target.value })}
                        onKeyPress={(e) => {
                          if (e.key === 'Enter') {
                            this.addObjective();
                          }
                        }}
                        size="small"
                        sx={{ 
                          mb: 3,
                          '& .MuiOutlinedInput-root': {
                            bgcolor: 'white',
                            borderRadius: 2,
                          }
                        }}
                      />

                      {session.objectives.map((objective, index) => (
                        <Card 
                          key={index}
                          sx={{ 
                            mb: 2,
                            border: '1px solid #e2e8f0',
                            borderRadius: 2,
                            '&:hover': { boxShadow: '0 4px 12px rgba(0,0,0,0.1)' }
                          }}
                        >
                          <CardContent sx={{ 
                            display: 'flex', 
                            alignItems: 'center', 
                            justifyContent: 'space-between',
                            p: 2,
                            '&:last-child': { pb: 2 }
                          }}>
                            <Typography sx={{ flex: 1 }}>{objective}</Typography>
                            <IconButton 
                              size="small" 
                              onClick={() => this.removeObjective(index)}
                              sx={{ color: '#ef4444' }}
                            >
                              <Delete fontSize="small" />
                            </IconButton>
                          </CardContent>
                        </Card>
                      ))}
                    </CardContent>
                  </Collapse>
                </Card>
              </Box>
            )}

            {/* Videos Tab */}
            {activeTab === 1 && (
              <Box sx={{ maxWidth: 800 }}>
                {/* Introduction Video Card */}
                <Card sx={{ mb: 4, borderRadius: 3, overflow: 'hidden' }}>
                  <Box 
                    onClick={() => this.toggleSection('intro')}
                    sx={{ 
                      p: 3, 
                      bgcolor: '#f8fafc', 
                      borderBottom: '1px solid #e2e8f0',
                      cursor: 'pointer',
                      display: 'flex',
                      alignItems: 'center',
                      justifyContent: 'space-between',
                      '&:hover': { bgcolor: '#f1f5f9' }
                    }}
                  >
                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                      <Avatar sx={{ bgcolor: '#38aae1', width: 40, height: 40 }}>
                        <PlayCircleOutline />
                      </Avatar>
                      <Box>
                        <Typography variant="h6" sx={{ fontWeight: 700, color: '#25476a' }}>
                          Introduction Video
                        </Typography>
                        <Typography variant="body2" sx={{ color: '#64748b' }}>
                          Welcome and introduction content
                        </Typography>
                      </Box>
                    </Box>
                    {expandedSections.intro ? <ExpandLess /> : <ExpandMore />}
                  </Box>
                  
                  <Collapse in={expandedSections.intro}>
                    <CardContent sx={{ p: 4 }}>
                      <Grid container spacing={3}>
                        <Grid item xs={12} md={8}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                            Video Title
                          </Typography>
                          <TextField
                            fullWidth
                            value={session.introVideo.title}
                            onChange={(e) => this.updateSession({
                              introVideo: { ...session.introVideo, title: e.target.value }
                            })}
                            placeholder="Introduction video title"
                            size="small"
                            sx={{ '& .MuiOutlinedInput-root': { borderRadius: 2, bgcolor: 'white' } }}
                          />
                        </Grid>
                        <Grid item xs={12} md={4}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                            Duration (minutes)
                          </Typography>
                          <TextField
                            fullWidth
                            type="number"
                            value={session.introVideo.duration}
                            onChange={(e) => this.updateSession({
                              introVideo: { ...session.introVideo, duration: parseInt(e.target.value) || 0 }
                            })}
                            size="small"
                            sx={{ '& .MuiOutlinedInput-root': { borderRadius: 2, bgcolor: 'white' } }}
                          />
                        </Grid>
                        <Grid item xs={12}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                            Upload Video
                          </Typography>
                          
                          {/* Video Preview */}
                          {session.introVideo.url && (
                            <Box sx={{ mb: 3 }}>
                              <Typography variant="body2" sx={{ mb: 2, fontWeight: 600, color: '#25476a' }}>
                                Video Preview
                              </Typography>
                              <VidstackPlayer
                                src={session.introVideo.url}
                                title={session.introVideo.title}
                                aspectRatio="16/9"
                              />
                              <Typography variant="body2" sx={{ color: '#38aae1', mt: 1, fontWeight: 600 }}>
                                Current video: {session.introVideo.url}
                              </Typography>
                            </Box>
                          )}
                          
                          <VideoUploadDragger
                            accept="video/*"
                            maxSize={100}
                            onUpload={(file) => {
                              this.updateSession({
                                introVideo: { ...session.introVideo, url: file.name }
                              });
                            }}
                            icon={<PlayCircleOutline sx={{ fontSize: 48, color: '#64748b', mb: 1 }} />}
                            title={session.introVideo.url ? "Click to replace introduction video" : "Click to upload introduction video"}
                            description="Drag and drop or click to browse"
                            helpText="MP4, AVI, MOV up to 100MB"
                          />
                        </Grid>
                      </Grid>
                    </CardContent>
                  </Collapse>
                </Card>

                {/* Assignment Video Card */}
                <Card sx={{ mb: 4, borderRadius: 3, overflow: 'hidden' }}>
                  <Box 
                    onClick={() => this.toggleSection('assignment')}
                    sx={{ 
                      p: 3, 
                      bgcolor: '#f8fafc', 
                      borderBottom: '1px solid #e2e8f0',
                      cursor: 'pointer',
                      display: 'flex',
                      alignItems: 'center',
                      justifyContent: 'space-between',
                      '&:hover': { bgcolor: '#f1f5f9' }
                    }}
                  >
                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                      <Avatar sx={{ bgcolor: '#ff9800', width: 40, height: 40 }}>
                        <Assignment />
                      </Avatar>
                      <Box>
                        <Typography variant="h6" sx={{ fontWeight: 700, color: '#25476a' }}>
                          Assignment Video
                        </Typography>
                        <Typography variant="body2" sx={{ color: '#64748b' }}>
                          Instructions and assignments content
                        </Typography>
                      </Box>
                    </Box>
                    {expandedSections.assignment ? <ExpandLess /> : <ExpandMore />}
                  </Box>
                  
                  <Collapse in={expandedSections.assignment}>
                    <CardContent sx={{ p: 4 }}>
                      <Grid container spacing={3}>
                        <Grid item xs={12} md={8}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                            Video Title
                          </Typography>
                          <TextField
                            fullWidth
                            value={session.assignmentVideo.title}
                            onChange={(e) => this.updateSession({
                              assignmentVideo: { ...session.assignmentVideo, title: e.target.value }
                            })}
                            placeholder="Assignment video title"
                            size="small"
                            sx={{ '& .MuiOutlinedInput-root': { borderRadius: 2, bgcolor: 'white' } }}
                          />
                        </Grid>
                        <Grid item xs={12} md={4}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                            Duration (minutes)
                          </Typography>
                          <TextField
                            fullWidth
                            type="number"
                            value={session.assignmentVideo.duration}
                            onChange={(e) => this.updateSession({
                              assignmentVideo: { ...session.assignmentVideo, duration: parseInt(e.target.value) || 0 }
                            })}
                            size="small"
                            sx={{ '& .MuiOutlinedInput-root': { borderRadius: 2, bgcolor: 'white' } }}
                          />
                        </Grid>
                        <Grid item xs={12}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                            Upload Video
                          </Typography>
                          
                          {/* Video Preview */}
                          {session.assignmentVideo.url && (
                            <Box sx={{ mb: 3 }}>
                              <Typography variant="body2" sx={{ mb: 2, fontWeight: 600, color: '#25476a' }}>
                                Video Preview
                              </Typography>
                              <VidstackPlayer
                                src={session.assignmentVideo.url}
                                title={session.assignmentVideo.title}
                                aspectRatio="16/9"
                              />
                              <Typography variant="body2" sx={{ color: '#ff9800', mt: 1, fontWeight: 600 }}>
                                Current video: {session.assignmentVideo.url}
                              </Typography>
                            </Box>
                          )}
                          
                          <VideoUploadDragger
                            accept="video/*"
                            maxSize={100}
                            onUpload={(file) => {
                              this.updateSession({
                                assignmentVideo: { ...session.assignmentVideo, url: file.name }
                              });
                            }}
                            icon={<Assignment sx={{ fontSize: 48, color: '#64748b', mb: 1 }} />}
                            title={session.assignmentVideo.url ? "Click to replace assignment video" : "Click to upload assignment video"}
                            description="Drag and drop or click to browse"
                            helpText="MP4, AVI, MOV up to 100MB"
                          />
                        </Grid>
                      </Grid>
                    </CardContent>
                  </Collapse>
                </Card>

                {/* Activity Videos Card */}
                <Card sx={{ borderRadius: 3, overflow: 'hidden' }}>
                  <Box 
                    onClick={() => this.toggleSection('activities')}
                    sx={{ 
                      p: 3, 
                      bgcolor: '#f8fafc', 
                      borderBottom: '1px solid #e2e8f0',
                      cursor: 'pointer',
                      display: 'flex',
                      alignItems: 'center',
                      justifyContent: 'space-between',
                      '&:hover': { bgcolor: '#f1f5f9' }
                    }}
                  >
                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                      <Avatar sx={{ bgcolor: '#9c27b0', width: 40, height: 40 }}>
                        <SportsEsports />
                      </Avatar>
                      <Box>
                        <Typography variant="h6" sx={{ fontWeight: 700, color: '#25476a' }}>
                          Activity Videos ({session.activityVideos.length})
                        </Typography>
                        <Typography variant="body2" sx={{ color: '#64748b' }}>
                          Interactive learning activities and exercises
                        </Typography>
                      </Box>
                    </Box>
                    {expandedSections.activities ? <ExpandLess /> : <ExpandMore />}
                  </Box>
                  
                  <Collapse in={expandedSections.activities}>
                    <CardContent sx={{ p: 4 }}>
                      <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 3 }}>
                        <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                          Activity Videos
                        </Typography>
                        <Button
                          startIcon={<Add />}
                          variant="contained"
                          onClick={this.addActivityVideo}
                          sx={{ 
                            bgcolor: '#9c27b0',
                            '&:hover': { bgcolor: '#7b1fa2' },
                            textTransform: 'none',
                            borderRadius: 2
                          }}
                        >
                          Add Activity Video
                        </Button>
                      </Box>
                      
                      {session.activityVideos.map((activity, index) => (
                        <Card key={index} sx={{ mb: 3, border: '1px solid #e2e8f0', borderRadius: 2 }}>
                          <CardContent sx={{ p: 3 }}>
                            <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 2 }}>
                              <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                                Activity Video {index + 1}
                              </Typography>
                              <IconButton 
                                size="small" 
                                onClick={() => this.removeActivityVideo(index)}
                                sx={{ color: '#ef4444' }}
                              >
                                <Delete />
                              </IconButton>
                            </Box>
                            
                            <Grid container spacing={3}>
                              <Grid item xs={12} md={8}>
                                <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                  Video Title
                                </Typography>
                                <TextField
                                  fullWidth
                                  value={activity.title}
                                  onChange={(e) => this.updateActivityVideo(index, 'title', e.target.value)}
                                  placeholder="Activity video title"
                                  size="small"
                                  sx={{ '& .MuiOutlinedInput-root': { borderRadius: 2 } }}
                                />
                              </Grid>
                              <Grid item xs={12} md={4}>
                                <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                  Duration (minutes)
                                </Typography>
                                <TextField
                                  fullWidth
                                  type="number"
                                  value={activity.duration}
                                  onChange={(e) => this.updateActivityVideo(index, 'duration', parseInt(e.target.value) || 0)}
                                  size="small"
                                  sx={{ '& .MuiOutlinedInput-root': { borderRadius: 2 } }}
                                />
                              </Grid>
                              <Grid item xs={12}>
                                <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                  Upload Video
                                </Typography>
                                <ActivityVideoUploader
                                  activityIndex={index}
                                  currentUrl={activity.url}
                                  currentTitle={activity.title}
                                  onUpload={(file) => {
                                    const videoUrl = URL.createObjectURL(file);
                                    this.updateActivityVideo(index, 'url', videoUrl);
                                    if (!activity.title) {
                                      this.updateActivityVideo(index, 'title', file.name);
                                    }
                                  }}
                                />
                              </Grid>
                              <Grid item xs={12}>
                                <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                  Or Enter Video URL
                                </Typography>
                                <TextField
                                  fullWidth
                                  value={activity.url}
                                  onChange={(e) => this.updateActivityVideo(index, 'url', e.target.value)}
                                  placeholder="Video file path or URL"
                                  size="small"
                                  sx={{ '& .MuiOutlinedInput-root': { borderRadius: 2, bgcolor: 'white' } }}
                                />
                              </Grid>
                            </Grid>
                          </CardContent>
                        </Card>
                      ))}
                    </CardContent>
                  </Collapse>
                </Card>
              </Box>
            )}

            {/* Resources Tab */}
            {activeTab === 2 && (
              <Box sx={{ maxWidth: 800 }}>
                <Card sx={{ borderRadius: 3, overflow: 'hidden' }}>
                  <CardContent sx={{ p: 4 }}>
                    <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 3 }}>
                      <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                        Learning Resources ({session.resources.length})
                      </Typography>
                      <Button
                        startIcon={<Add />}
                        variant="contained"
                        onClick={this.addResource}
                        disabled={!newResource.name.trim()}
                        sx={{ 
                          bgcolor: '#9c27b0',
                          '&:hover': { bgcolor: '#7b1fa2' },
                          textTransform: 'none',
                          borderRadius: 2
                        }}
                      >
                        Add Resource
                      </Button>
                    </Box>

                    {/* Add New Resource */}
                    <Card sx={{ mb: 4, border: '1px solid #e2e8f0', borderRadius: 2, bgcolor: '#f8fafc' }}>
                      <CardContent sx={{ p: 3 }}>
                        <Typography variant="h6" sx={{ mb: 2, fontWeight: 600, color: '#25476a' }}>
                          Add New Resource
                        </Typography>
                        <Grid container spacing={3}>
                          <Grid item xs={12}>
                            <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                              Resource Input Method
                            </Typography>
                            <FormControl component="fieldset">
                              <RadioGroup
                                row
                                value={this.state.resourceInputMethod}
                                onChange={(e) => {
                                  const method = e.target.value as 'upload' | 'url';
                                  this.setState({ 
                                    resourceInputMethod: method,
                                    newResource: { 
                                      ...newResource, 
                                      url: '',
                                      type: method === 'upload' ? 'pdf' : 'youtube'
                                    }
                                  });
                                }}
                              >
                                <FormControlLabel 
                                  value="url" 
                                  control={<Radio sx={{ color: '#38aae1', '&.Mui-checked': { color: '#38aae1' } }} />} 
                                  label="URL" 
                                />
                                <FormControlLabel 
                                  value="upload" 
                                  control={<Radio sx={{ color: '#38aae1', '&.Mui-checked': { color: '#38aae1' } }} />} 
                                  label="Upload File" 
                                />
                              </RadioGroup>
                            </FormControl>
                          </Grid>
                          <Grid item xs={12} md={6}>
                            <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                              Resource Name
                            </Typography>
                            <TextField
                              fullWidth
                              value={newResource.name}
                              onChange={(e) => this.setState({
                                newResource: { ...newResource, name: e.target.value }
                              })}
                              placeholder="Resource name"
                              size="small"
                              sx={{ '& .MuiOutlinedInput-root': { borderRadius: 2, bgcolor: 'white' } }}
                            />
                          </Grid>
                          <Grid item xs={12} md={6}>
                            <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                              Resource Type
                            </Typography>
                            <FormControl fullWidth size="small">
                              <Select
                                value={newResource.type}
                                onChange={(e) => this.setState({
                                  newResource: { ...newResource, type: e.target.value as 'pdf' | 'doc' | 'ppt' | 'youtube' }
                                })}
                                sx={{ borderRadius: 2, bgcolor: 'white' }}
                              >
                                {this.state.resourceInputMethod === 'upload' ? (
                                  <>
                                    <MenuItem value="pdf">PDF Document</MenuItem>
                                    <MenuItem value="doc">Word Document</MenuItem>
                                    <MenuItem value="ppt">PowerPoint Presentation</MenuItem>
                                  </>
                                ) : (
                                  <>
                                    <MenuItem value="youtube">YouTube Video</MenuItem>
                                    <MenuItem value="pdf">PDF Link</MenuItem>
                                    <MenuItem value="doc">Document Link</MenuItem>
                                    <MenuItem value="ppt">Presentation Link</MenuItem>
                                  </>
                                )}
                              </Select>
                            </FormControl>
                          </Grid>
                          {this.state.resourceInputMethod === 'url' ? (
                            <Grid item xs={12}>
                              <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                {newResource.type === 'youtube' ? 'YouTube URL' : 'Resource URL'}
                              </Typography>
                              <TextField
                                fullWidth
                                value={newResource.url}
                                onChange={(e) => this.setState({
                                  newResource: { ...newResource, url: e.target.value }
                                })}
                                placeholder={
                                  newResource.type === 'youtube' 
                                    ? 'Enter YouTube URL (e.g., https://www.youtube.com/watch?v=...)' 
                                    : 'Enter resource URL (e.g., https://example.com/document.pdf)'
                                }
                                size="small"
                                sx={{ '& .MuiOutlinedInput-root': { borderRadius: 2, bgcolor: 'white' } }}
                              />
                            </Grid>
                          ) : (
                            <Grid item xs={12}>
                              <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                Upload File
                              </Typography>
                              <Box sx={{ 
                                border: '2px dashed #38aae1', 
                                borderRadius: 2, 
                                p: 3, 
                                textAlign: 'center',
                                bgcolor: 'white',
                                cursor: 'pointer',
                                '&:hover': {
                                  bgcolor: '#f8fafc',
                                  borderColor: '#25476a'
                                }
                              }}>
                                <input
                                  accept=".pdf,.doc,.docx,.ppt,.pptx"
                                  style={{ display: 'none' }}
                                  id="resource-file-upload-single"
                                  type="file"
                                  onChange={this.handleResourceFileUpload}
                                />
                                <label htmlFor="resource-file-upload-single" style={{ cursor: 'pointer', width: '100%' }}>
                                  <CloudUpload sx={{ fontSize: 48, color: '#38aae1', mb: 1 }} />
                                  <Typography variant="body1" sx={{ color: '#25476a', fontWeight: 600, mb: 0.5 }}>
                                    {newResource.url ? 'File Selected' : 'Click to Upload File'}
                                  </Typography>
                                  <Typography variant="body2" sx={{ color: '#64748b' }}>
                                    {newResource.url ? newResource.name || 'File uploaded' : 'PDF, DOC, DOCX, PPT, PPTX files supported'}
                                  </Typography>
                                </label>
                              </Box>
                            </Grid>
                          )}
                        </Grid>
                      </CardContent>
                    </Card>

                    {/* Existing Resources */}
                    {session.resources.map((resource, index) => (
                      <Card 
                        key={index}
                        sx={{ 
                          mb: 2,
                          border: '1px solid #e2e8f0',
                          borderRadius: 2,
                          '&:hover': { boxShadow: '0 4px 12px rgba(0,0,0,0.1)' }
                        }}
                      >
                        <CardContent sx={{ 
                          display: 'flex', 
                          alignItems: 'center', 
                          justifyContent: 'space-between',
                          p: 2,
                          '&:last-child': { pb: 2 }
                        }}>
                          <Box>
                            <Typography variant="body1" sx={{ fontWeight: 600, color: '#25476a' }}>
                              {resource.name}
                            </Typography>
                            <Chip 
                              label={resource.type.toUpperCase()}
                              size="small"
                              sx={{ mt: 0.5, fontSize: '0.75rem' }}
                            />
                          </Box>
                          <IconButton 
                            size="small" 
                            onClick={() => this.removeResource(index)}
                            sx={{ color: '#ef4444' }}
                          >
                            <Delete fontSize="small" />
                          </IconButton>
                        </CardContent>
                      </Card>
                    ))}
                  </CardContent>
                </Card>
              </Box>
            )}
          </Box>
        </DialogContent>

        <DialogActions sx={{ 
          p: 4, 
          bgcolor: 'white', 
          borderTop: '1px solid #e2e8f0',
          display: 'flex',
          justifyContent: 'space-between',
          alignItems: 'center',
          boxShadow: '0 -4px 20px rgba(0,0,0,0.05)'
        }}>
          <Button
            onClick={this.toggleComplete}
            variant={session.isCompleted ? "outlined" : "contained"}
            startIcon={session.isCompleted ? <Close /> : <CheckCircle />}
            sx={{
              textTransform: 'none',
              bgcolor: session.isCompleted ? 'transparent' : '#4caf50',
              color: session.isCompleted ? '#4caf50' : 'white',
              borderColor: '#4caf50',
              borderRadius: 2,
              px: 3,
              '&:hover': {
                bgcolor: session.isCompleted ? 'rgba(76, 175, 80, 0.1)' : '#45a049',
              }
            }}
          >
            {session.isCompleted ? 'Mark Incomplete' : 'Mark Complete'}
          </Button>
          
          <Box sx={{ display: 'flex', gap: 2 }}>
            <Button 
              onClick={onClose}
              variant="outlined"
              sx={{ 
                color: '#64748b',
                borderColor: '#e2e8f0',
                textTransform: 'none',
                borderRadius: 2,
                px: 3,
                '&:hover': { 
                  bgcolor: '#f8fafc',
                  borderColor: '#d1d5db' 
                }
              }}
            >
              Cancel
            </Button>
            <Button 
              onClick={this.handleSave}
              variant="contained"
              startIcon={<Save />}
              sx={{
                background: 'linear-gradient(135deg, #38aae1 0%, #2563eb 100%)',
                textTransform: 'none',
                fontWeight: 600,
                borderRadius: 2,
                px: 4,
                boxShadow: '0 4px 20px rgba(56, 170, 225, 0.3)',
                '&:hover': {
                  background: 'linear-gradient(135deg, #2e8bc4 0%, #1d4ed8 100%)',
                  boxShadow: '0 6px 25px rgba(56, 170, 225, 0.4)',
                },
              }}
            >
              Save Session
            </Button>
          </Box>
        </DialogActions>
      </Dialog>
    );
  }
}