import React, { useState } from 'react';
import {
  Box,
  Typography,
  Button,
  Paper,
  Grid,
  Card,
  CardContent,
  TextField,
  Tabs,
  Tab,
  InputAdornment,
  Divider,
} from '@mui/material';
import {
  ArrowBack,
  Payment,
  Quiz,
  Home,
  Save,
  CalendarMonth,
  EventAvailable,
  Visibility,
  AccessTime,
  CheckCircle,
  Edit,
} from '@mui/icons-material';
import {
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Chip,
} from '@mui/material';
import { AvailabilityScheduleDialog } from './AvailabilityScheduleDialog';

interface TabPanelProps {
  children?: React.ReactNode;
  index: number;
  value: number;
}

interface Mentor {
  id: string;
  name: string;
  specialization: string;
  status: string;
  gender: string;
  ratePerHour: number;
  rating: number;
  totalHours: number;
  students: number;
  monthlyEarnings: number;
  subjects: string[];
}

function TabPanel(props: TabPanelProps) {
  const { children, value, index, ...other } = props;

  return (
    <div
      role="tabpanel"
      hidden={value !== index}
      id={`settings-tabpanel-${index}`}
      aria-labelledby={`settings-tab-${index}`}
      {...other}
    >
      {value === index && (
        <Box sx={{ py: 3 }}>
          {children}
        </Box>
      )}
    </div>
  );
}

export function Settings() {
  const [selectedTab, setSelectedTab] = useState(0);
  const [paymentRates, setPaymentRates] = useState({
    technicalMentor: 564.64,
    assistantMentor: 500,
    onlineTrainer: 500,
    digiUnziPay: 750.00,
  });
  const [thresholds, setThresholds] = useState({
    classClaiming: 100,
    advanceRequest: 50,
  });
  const [centerManagement, setCenterManagement] = useState({
    basePrice: 24000,
    multiChildPrice: 40000,
    maxChildren: 5,
  });
  
  // Schedule dialog states
  const [openAvailabilityDialog, setOpenAvailabilityDialog] = useState(false);
  
  // Mock current user as a mentor (in a real app, this would come from auth context)
  const currentMentor: Mentor = {
    id: '1',
    name: 'Current User',
    specialization: 'Technical Mentor',
    status: 'Active',
    gender: 'Male',
    ratePerHour: 564.64,
    rating: 4.8,
    totalHours: 150,
    students: 25,
    monthlyEarnings: 84696,
    subjects: ['Python', 'JavaScript', 'React'],
  };

  // Mock schedule data - In a real app, this would come from your backend
  const getMentorSchedule = (mentorId: string | undefined) => {
    return {
      duration: 60,
      availability: {
        Monday: ['09:00', '10:00', '14:00'],
        Tuesday: ['09:00', '10:00', '14:00', '15:00'],
        Wednesday: ['10:00', '11:00', '15:00'],
        Thursday: ['09:00', '14:00', '15:00', '16:00'],
        Friday: ['09:00', '10:00', '13:00', '14:00'],
        Saturday: [],
        Sunday: [],
      },
      totalSlotsPerWeek: 17,
      activeDays: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri'],
    };
  };

  const schedule = getMentorSchedule(currentMentor.id);

  const DAYS_OF_WEEK = [
    { short: 'Mon', full: 'Monday' },
    { short: 'Tue', full: 'Tuesday' },
    { short: 'Wed', full: 'Wednesday' },
    { short: 'Thu', full: 'Thursday' },
    { short: 'Fri', full: 'Friday' },
    { short: 'Sat', full: 'Saturday' },
    { short: 'Sun', full: 'Sunday' },
  ];

  const getTotalSlots = (): number => {
    return Object.values(schedule.availability).reduce(
      (total, slots) => total + slots.length, 
      0
    );
  };

  const getActiveDaysCount = (): number => {
    return Object.values(schedule.availability).filter(
      slots => slots.length > 0
    ).length;
  };

  const formatTimeSlots = (slots: string[]): string => {
    if (slots.length === 0) return 'No slots available';
    return slots.join(', ');
  };

  const getTimeRanges = (slots: string[]): string => {
    if (slots.length === 0) return '-';
    if (slots.length === 1) return slots[0];
    
    const sortedSlots = [...slots].sort();
    const ranges: string[] = [];
    let rangeStart = sortedSlots[0];
    let rangeEnd = sortedSlots[0];

    for (let i = 1; i < sortedSlots.length; i++) {
      const currentTime = sortedSlots[i];
      const prevTime = sortedSlots[i - 1];
      
      const currentHour = parseInt(currentTime.split(':')[0]);
      const prevHour = parseInt(prevTime.split(':')[0]);
      
      if (currentHour === prevHour + 1) {
        rangeEnd = currentTime;
      } else {
        ranges.push(rangeStart === rangeEnd ? rangeStart : `${rangeStart} - ${rangeEnd}`);
        rangeStart = currentTime;
        rangeEnd = currentTime;
      }
    }
    
    ranges.push(rangeStart === rangeEnd ? rangeStart : `${rangeStart} - ${rangeEnd}`);
    return ranges.join(', ');
  };

  const handleTabChange = (event: React.SyntheticEvent, newValue: number) => {
    setSelectedTab(newValue);
  };

  const handlePaymentRateChange = (field: string, value: number) => {
    setPaymentRates({ ...paymentRates, [field]: value });
  };

  const handleThresholdChange = (field: string, value: number) => {
    setThresholds({ ...thresholds, [field]: value });
  };

  const handleCenterChange = (field: string, value: number) => {
    setCenterManagement({ ...centerManagement, [field]: value });
  };

  return (
    <Box sx={{ p: 3 }}>
      {/* Header */}
      <Box sx={{ mb: 4 }}>
        <Typography variant="h4" sx={{ fontWeight: 600, color: '#25476a', mb: 1 }}>
          Platform Settings
        </Typography>
        <Typography variant="body1" color="text.secondary">
          Configure payment rates, thresholds, and system settings
        </Typography>
      </Box>

      {/* Settings Tabs */}
      <Paper sx={{ borderRadius: 3, boxShadow: '0 4px 20px rgba(0,0,0,0.08)' }}>
        <Box sx={{ borderBottom: 1, borderColor: 'divider' }}>
          <Tabs
            value={selectedTab}
            onChange={handleTabChange}
            sx={{
              '& .MuiTab-root': {
                textTransform: 'none',
                fontWeight: 600,
                fontSize: '1rem',
                minHeight: 64,
              }
            }}
          >
            <Tab 
              icon={<Payment />} 
              label="Payment Settings" 
              iconPosition="start"
              sx={{ px: 4 }}
            />
            <Tab 
              icon={<Quiz />} 
              label="Quarky Settings" 
              iconPosition="start"
              sx={{ px: 4 }}
            />
            <Tab 
              icon={<Home />} 
              label="Curriculum" 
              iconPosition="start"
              sx={{ px: 4 }}
            />
            <Tab 
              icon={<CalendarMonth />} 
              label="Schedule & Availability" 
              iconPosition="start"
              sx={{ px: 4 }}
            />
          </Tabs>
        </Box>

        <TabPanel value={selectedTab} index={0}>
          <Box sx={{ p: 4 }}>
            {/* Payment Rates & Compensation */}
            <Card sx={{ mb: 4, borderRadius: 3, boxShadow: '0 2px 8px rgba(0,0,0,0.06)' }}>
              <CardContent sx={{ p: 4 }}>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 3 }}>
                  <Payment sx={{ mr: 2, color: '#38aae1' }} />
                  <Box>
                    <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                      Payment Rates & Compensation
                    </Typography>
                    <Typography variant="body2" color="text.secondary">
                      Configure hourly rates for different roles
                    </Typography>
                  </Box>
                </Box>

                <Grid container spacing={4}>
                  <Grid item xs={12} md={6}>
                    <Typography variant="subtitle1" sx={{ mb: 2, fontWeight: 600 }}>
                      Technical Mentor Rate
                    </Typography>
                    <TextField
                      fullWidth
                      value={paymentRates.technicalMentor}
                      onChange={(e) => handlePaymentRateChange('technicalMentor', parseFloat(e.target.value))}
                      InputProps={{
                        startAdornment: <InputAdornment position="start">KES</InputAdornment>,
                      }}
                      sx={{ mb: 1 }}
                    />
                    <Typography variant="caption" color="text.secondary">
                      Per hour
                    </Typography>
                  </Grid>

                  <Grid item xs={12} md={6}>
                    <Typography variant="subtitle1" sx={{ mb: 2, fontWeight: 600 }}>
                      Assistant Mentor Rate
                    </Typography>
                    <TextField
                      fullWidth
                      value={paymentRates.assistantMentor}
                      onChange={(e) => handlePaymentRateChange('assistantMentor', parseFloat(e.target.value))}
                      InputProps={{
                        startAdornment: <InputAdornment position="start">KES</InputAdornment>,
                      }}
                      sx={{ mb: 1 }}
                    />
                    <Typography variant="caption" color="text.secondary">
                      Per hour
                    </Typography>
                  </Grid>

                  <Grid item xs={12} md={6}>
                    <Typography variant="subtitle1" sx={{ mb: 2, fontWeight: 600 }}>
                      Online Trainer Rate
                    </Typography>
                    <TextField
                      fullWidth
                      value={paymentRates.onlineTrainer}
                      onChange={(e) => handlePaymentRateChange('onlineTrainer', parseFloat(e.target.value))}
                      InputProps={{
                        startAdornment: <InputAdornment position="start">KES</InputAdornment>,
                      }}
                      sx={{ mb: 1 }}
                    />
                    <Typography variant="caption" color="text.secondary">
                      Per hour
                    </Typography>
                  </Grid>

                  <Grid item xs={12} md={6}>
                    <Typography variant="subtitle1" sx={{ mb: 2, fontWeight: 600 }}>
                      DigiUnzi Pay Rate
                    </Typography>
                    <TextField
                      fullWidth
                      value={paymentRates.digiUnziPay}
                      onChange={(e) => handlePaymentRateChange('digiUnziPay', parseFloat(e.target.value))}
                      InputProps={{
                        startAdornment: <InputAdornment position="start">KES</InputAdornment>,
                      }}
                      sx={{ mb: 1 }}
                    />
                    <Typography variant="caption" color="text.secondary">
                      Per hour
                    </Typography>
                  </Grid>
                </Grid>

                <Box sx={{ mt: 4, display: 'flex', justifyContent: 'flex-end' }}>
                  <Button
                    variant="contained"
                    startIcon={<Save />}
                    sx={{ 
                      bgcolor: '#38aae1', 
                      '&:hover': { bgcolor: '#2e8bc4' },
                      px: 3,
                    }}
                  >
                    Save Payment Rates
                  </Button>
                </Box>
              </CardContent>
            </Card>

            {/* Performance Thresholds */}
            <Card sx={{ borderRadius: 3, boxShadow: '0 2px 8px rgba(0,0,0,0.06)' }}>
              <CardContent sx={{ p: 4 }}>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 3 }}>
                  <Quiz sx={{ mr: 2, color: '#feb139' }} />
                  <Box>
                    <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                      Performance Thresholds
                    </Typography>
                    <Typography variant="body2" color="text.secondary">
                      Set minimum completion thresholds
                    </Typography>
                  </Box>
                </Box>

                <Grid container spacing={4}>
                  <Grid item xs={12} md={6}>
                    <Typography variant="subtitle1" sx={{ mb: 2, fontWeight: 600 }}>
                      Class Claiming Threshold
                    </Typography>
                    <TextField
                      fullWidth
                      value={thresholds.classClaiming}
                      onChange={(e) => handleThresholdChange('classClaiming', parseInt(e.target.value))}
                      InputProps={{
                        endAdornment: <InputAdornment position="end">%</InputAdornment>,
                      }}
                      sx={{ mb: 1 }}
                    />
                    <Typography variant="caption" color="text.secondary">
                      Minimum completion to claim payment
                    </Typography>
                  </Grid>

                  <Grid item xs={12} md={6}>
                    <Typography variant="subtitle1" sx={{ mb: 2, fontWeight: 600 }}>
                      Advance Request Threshold
                    </Typography>
                    <TextField
                      fullWidth
                      value={thresholds.advanceRequest}
                      onChange={(e) => handleThresholdChange('advanceRequest', parseInt(e.target.value))}
                      InputProps={{
                        endAdornment: <InputAdornment position="end">%</InputAdornment>,
                      }}
                      sx={{ mb: 1 }}
                    />
                    <Typography variant="caption" color="text.secondary">
                      Minimum completion to request advance
                    </Typography>
                  </Grid>
                </Grid>

                <Box sx={{ mt: 4, display: 'flex', justifyContent: 'flex-end' }}>
                  <Button
                    variant="outlined"
                    startIcon={<Save />}
                    sx={{ 
                      borderColor: '#feb139',
                      color: '#feb139',
                      '&:hover': { 
                        borderColor: '#e09c2c',
                        color: '#e09c2c',
                        bgcolor: 'rgba(254, 177, 57, 0.04)'
                      },
                      px: 3,
                    }}
                  >
                    Update
                  </Button>
                </Box>
              </CardContent>
            </Card>
          </Box>
        </TabPanel>

        <TabPanel value={selectedTab} index={1}>
          <Box sx={{ p: 4 }}>
            <Card sx={{ borderRadius: 3, boxShadow: '0 2px 8px rgba(0,0,0,0.06)' }}>
              <CardContent sx={{ p: 4 }}>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 3 }}>
                  <Home sx={{ mr: 2, color: '#4caf50' }} />
                  <Box>
                    <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                      Center Management
                    </Typography>
                    <Typography variant="body2" color="text.secondary">
                      Configure house center limits and pricing
                    </Typography>
                  </Box>
                </Box>

                <Grid container spacing={4}>
                  <Grid item xs={12} md={6}>
                    <Typography variant="subtitle1" sx={{ mb: 2, fontWeight: 600 }}>
                      Base House Center Price
                    </Typography>
                    <TextField
                      fullWidth
                      value={centerManagement.basePrice}
                      onChange={(e) => handleCenterChange('basePrice', parseInt(e.target.value))}
                      InputProps={{
                        startAdornment: <InputAdornment position="start">KES</InputAdornment>,
                      }}
                      sx={{ mb: 1 }}
                    />
                  </Grid>

                  <Grid item xs={12} md={6}>
                    <Typography variant="subtitle1" sx={{ mb: 2, fontWeight: 600 }}>
                      Multi-Child House Center Price
                    </Typography>
                    <TextField
                      fullWidth
                      value={centerManagement.multiChildPrice}
                      onChange={(e) => handleCenterChange('multiChildPrice', parseInt(e.target.value))}
                      InputProps={{
                        startAdornment: <InputAdornment position="start">KES</InputAdornment>,
                      }}
                      sx={{ mb: 1 }}
                    />
                    <Typography variant="caption" color="text.secondary">
                      For centers with more than 1 child
                    </Typography>
                  </Grid>

                  <Grid item xs={12} md={6}>
                    <Typography variant="subtitle1" sx={{ mb: 2, fontWeight: 600 }}>
                      Maximum Children per House Center
                    </Typography>
                    <TextField
                      fullWidth
                      value={centerManagement.maxChildren}
                      onChange={(e) => handleCenterChange('maxChildren', parseInt(e.target.value))}
                      sx={{ mb: 1 }}
                    />
                  </Grid>
                </Grid>

                <Box sx={{ mt: 4, display: 'flex', justifyContent: 'flex-end' }}>
                  <Button
                    variant="outlined"
                    startIcon={<Save />}
                    sx={{ 
                      borderColor: '#4caf50',
                      color: '#4caf50',
                      '&:hover': { 
                        borderColor: '#45a049',
                        color: '#45a049',
                        bgcolor: 'rgba(76, 175, 80, 0.04)'
                      },
                      px: 3,
                    }}
                  >
                    Update
                  </Button>
                </Box>
              </CardContent>
            </Card>
          </Box>
        </TabPanel>

        <TabPanel value={selectedTab} index={2}>
          <Box sx={{ p: 4, textAlign: 'center' }}>
            <Typography variant="h6" color="text.secondary">
              Curriculum settings will be configured here
            </Typography>
            <Typography variant="body2" color="text.secondary" sx={{ mt: 2 }}>
              This section is under development
            </Typography>
          </Box>
        </TabPanel>

        <TabPanel value={selectedTab} index={3}>
          <Box sx={{ p: 4 }}>
            {/* Header with Configure Button */}
            <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 3 }}>
              <Box sx={{ display: 'flex', alignItems: 'center' }}>
                <CalendarMonth sx={{ color: '#38aae1', mr: 2, fontSize: 32 }} />
                <Box>
                  <Typography variant="h5" sx={{ fontWeight: 600, color: '#25476a' }}>
                    Availability Schedule
                  </Typography>
                  <Typography variant="body2" sx={{ color: '#64748b', mt: 0.5 }}>
                    {currentMentor.name}'s consultation availability schedule
                  </Typography>
                </Box>
              </Box>
              <Button
                variant="contained"
                startIcon={<Edit />}
                onClick={() => setOpenAvailabilityDialog(true)}
                sx={{
                  backgroundColor: '#38aae1',
                  px: 3,
                  py: 1.5,
                  '&:hover': {
                    backgroundColor: '#2e8bc4',
                  },
                }}
              >
                Configure Availability
              </Button>
            </Box>

            {/* Schedule Summary */}
            <Grid container spacing={3} sx={{ mb: 4 }}>
              <Grid item xs={12} md={3}>
                <Paper 
                  sx={{ 
                    p: 3, 
                    borderRadius: 2, 
                    backgroundColor: '#f0f9ff',
                    border: '1px solid #bae6fd',
                  }}
                >
                  <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                    <AccessTime sx={{ color: '#38aae1', fontSize: 20, mr: 1 }} />
                    <Typography variant="body2" sx={{ color: '#64748b' }}>
                      Session Duration
                    </Typography>
                  </Box>
                  <Typography variant="h4" sx={{ fontWeight: 700, color: '#25476a' }}>
                    {schedule.duration} min
                  </Typography>
                </Paper>
              </Grid>

              <Grid item xs={12} md={3}>
                <Paper 
                  sx={{ 
                    p: 3, 
                    borderRadius: 2, 
                    backgroundColor: '#fff7ed',
                    border: '1px solid #fed7aa',
                  }}
                >
                  <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                    <EventAvailable sx={{ color: '#feb139', fontSize: 20, mr: 1 }} />
                    <Typography variant="body2" sx={{ color: '#64748b' }}>
                      Total Slots
                    </Typography>
                  </Box>
                  <Typography variant="h4" sx={{ fontWeight: 700, color: '#25476a' }}>
                    {getTotalSlots()}
                  </Typography>
                  <Typography variant="caption" sx={{ color: '#64748b' }}>
                    per week
                  </Typography>
                </Paper>
              </Grid>

              <Grid item xs={12} md={3}>
                <Paper 
                  sx={{ 
                    p: 3, 
                    borderRadius: 2, 
                    backgroundColor: '#f0fdf4',
                    border: '1px solid #bbf7d0',
                  }}
                >
                  <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                    <CheckCircle sx={{ color: '#22c55e', fontSize: 20, mr: 1 }} />
                    <Typography variant="body2" sx={{ color: '#64748b' }}>
                      Active Days
                    </Typography>
                  </Box>
                  <Typography variant="h4" sx={{ fontWeight: 700, color: '#25476a' }}>
                    {getActiveDaysCount()}
                  </Typography>
                  <Typography variant="caption" sx={{ color: '#64748b' }}>
                    days per week
                  </Typography>
                </Paper>
              </Grid>

              <Grid item xs={12} md={3}>
                <Paper 
                  sx={{ 
                    p: 3, 
                    borderRadius: 2, 
                    backgroundColor: '#faf5ff',
                    border: '1px solid #e9d5ff',
                  }}
                >
                  <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                    <AccessTime sx={{ color: '#a855f7', fontSize: 20, mr: 1 }} />
                    <Typography variant="body2" sx={{ color: '#64748b' }}>
                      Total Hours
                    </Typography>
                  </Box>
                  <Typography variant="h4" sx={{ fontWeight: 700, color: '#25476a' }}>
                    {(getTotalSlots() * schedule.duration) / 60}h
                  </Typography>
                  <Typography variant="caption" sx={{ color: '#64748b' }}>
                    per week
                  </Typography>
                </Paper>
              </Grid>
            </Grid>

            <Divider sx={{ mb: 3 }} />

            {/* Weekly Schedule Table */}
            <Box sx={{ mb: 3 }}>
              <Typography variant="h6" sx={{ mb: 2, color: '#25476a', fontWeight: 600 }}>
                Weekly Availability
              </Typography>
              
              <TableContainer component={Paper} sx={{ borderRadius: 2, border: '1px solid #e2e8f0' }}>
                <Table>
                  <TableHead sx={{ backgroundColor: '#f8fafc' }}>
                    <TableRow>
                      <TableCell sx={{ fontWeight: 600, color: '#25476a', width: '120px' }}>
                        Day
                      </TableCell>
                      <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>
                        Available Time Slots
                      </TableCell>
                      <TableCell sx={{ fontWeight: 600, color: '#25476a', width: '100px', textAlign: 'center' }}>
                        Slots
                      </TableCell>
                      <TableCell sx={{ fontWeight: 600, color: '#25476a', width: '120px', textAlign: 'center' }}>
                        Status
                      </TableCell>
                    </TableRow>
                  </TableHead>
                  <TableBody>
                    {DAYS_OF_WEEK.map((day) => {
                      const daySlots = schedule.availability[day.full as keyof typeof schedule.availability] || [];
                      const hasSlots = daySlots.length > 0;
                      
                      return (
                        <TableRow 
                          key={day.full}
                          sx={{ 
                            '&:hover': { backgroundColor: '#f8fafc' },
                            backgroundColor: hasSlots ? 'white' : '#fafbfc',
                          }}
                        >
                          <TableCell>
                            <Box sx={{ display: 'flex', alignItems: 'center' }}>
                              <Box
                                sx={{
                                  width: 8,
                                  height: 8,
                                  borderRadius: '50%',
                                  backgroundColor: hasSlots ? '#22c55e' : '#e2e8f0',
                                  mr: 2,
                                }}
                              />
                              <Typography variant="body1" sx={{ fontWeight: 600, color: '#25476a' }}>
                                {day.full}
                              </Typography>
                            </Box>
                          </TableCell>
                          <TableCell>
                            <Typography 
                              variant="body2" 
                              sx={{ 
                                color: hasSlots ? '#25476a' : '#94a3b8',
                                fontFamily: 'monospace',
                              }}
                            >
                              {formatTimeSlots(daySlots)}
                            </Typography>
                          </TableCell>
                          <TableCell sx={{ textAlign: 'center' }}>
                            <Chip
                              label={daySlots.length}
                              size="small"
                              sx={{
                                backgroundColor: hasSlots ? '#dbeafe' : '#f1f5f9',
                                color: hasSlots ? '#1e40af' : '#64748b',
                                fontWeight: 600,
                                minWidth: 45,
                              }}
                            />
                          </TableCell>
                          <TableCell sx={{ textAlign: 'center' }}>
                            <Chip
                              label={hasSlots ? 'Available' : 'Off'}
                              size="small"
                              color={hasSlots ? 'success' : 'default'}
                              variant={hasSlots ? 'filled' : 'outlined'}
                              sx={{
                                fontWeight: 500,
                                minWidth: 85,
                              }}
                            />
                          </TableCell>
                        </TableRow>
                      );
                    })}
                  </TableBody>
                </Table>
              </TableContainer>
            </Box>

            {/* Time Range Summary */}
            <Paper 
              sx={{ 
                p: 3, 
                borderRadius: 2,
                backgroundColor: '#f8fafc',
                border: '1px solid #e2e8f0',
              }}
            >
              <Typography variant="h6" sx={{ mb: 2, color: '#25476a', fontWeight: 600 }}>
                Daily Time Ranges
              </Typography>
              <Grid container spacing={2}>
                {DAYS_OF_WEEK.map((day) => {
                  const daySlots = schedule.availability[day.full as keyof typeof schedule.availability] || [];
                  if (daySlots.length === 0) return null;

                  return (
                    <Grid item xs={12} sm={6} md={4} key={day.full}>
                      <Box
                        sx={{
                          p: 2,
                          borderRadius: 1.5,
                          backgroundColor: 'white',
                          border: '1px solid #e2e8f0',
                        }}
                      >
                        <Typography 
                          variant="body2" 
                          sx={{ 
                            fontWeight: 600, 
                            color: '#25476a',
                            mb: 0.5,
                          }}
                        >
                          {day.short}
                        </Typography>
                        <Typography 
                          variant="body2" 
                          sx={{ 
                            color: '#64748b',
                            fontFamily: 'monospace',
                            fontSize: '0.85rem',
                          }}
                        >
                          {getTimeRanges(daySlots)}
                        </Typography>
                      </Box>
                    </Grid>
                  );
                })}
              </Grid>
            </Paper>
          </Box>
        </TabPanel>
      </Paper>

      {/* Availability Schedule Dialog */}
      <AvailabilityScheduleDialog
        open={openAvailabilityDialog}
        onClose={() => setOpenAvailabilityDialog(false)}
        mentor={currentMentor}
      />
    </Box>
  );
}