import React, { Component } from 'react';
import {
  Box,
  Typography,
  Button,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Chip,
  IconButton,
  Menu,
  MenuItem,
  TextField,
  InputAdornment,
} from '@mui/material';
import {
  Add,
  Search,
  MoreVert,
  Visibility,
  Edit,
  Delete,
} from '@mui/icons-material';
import { CreateClassDialog } from './CreateClassDialog';
import { ClassView } from './ClassView';

interface Class {
  id: number;
  name: string;
  coursesCount: number;
  revenue: number;
  students: number;
  passType: string;
  status: 'Published' | 'Draft';
}

const mockClasses: Class[] = [
  {
    id: 1,
    name: 'Grade 3 Mathematics',
    coursesCount: 3,
    revenue: 1700,
    students: 45,
    passType: 'Meeting',
    status: 'Published',
  },
  {
    id: 2,
    name: 'Grade 4 Science',
    coursesCount: 2,
    revenue: 1200,
    students: 32,
    passType: 'Learning',
    status: 'Published',
  },
  {
    id: 3,
    name: 'Grade 5 English',
    coursesCount: 4,
    revenue: 2100,
    students: 58,
    passType: 'Meeting',
    status: 'Draft',
  },
];

interface SelfPacedLearningState {
  classes: Class[];
  searchTerm: string;
  openCreateDialog: boolean;
  openEditDialog: boolean;
  anchorEl: HTMLElement | null;
  selectedClass: Class | null;
  editingClass: Class | null;
  viewingClass: Class | null;
}

export class SelfPacedLearning extends Component<{}, SelfPacedLearningState> {
  constructor(props: {}) {
    super(props);
    this.state = {
      classes: mockClasses,
      searchTerm: '',
      openCreateDialog: false,
      openEditDialog: false,
      anchorEl: null,
      selectedClass: null,
      editingClass: null,
      viewingClass: null,
    };
  }

  handleMenuClick = (event: React.MouseEvent<HTMLElement>, classItem: Class) => {
    this.setState({
      anchorEl: event.currentTarget,
      selectedClass: classItem
    });
  };

  handleMenuClose = () => {
    this.setState({
      anchorEl: null,
      selectedClass: null
    });
  };

  handleViewClass = (classItem: Class) => {
    this.setState({ viewingClass: classItem });
    this.handleMenuClose();
  };

  handleBackToList = () => {
    this.setState({ viewingClass: null });
  };

  handleEditClass = (classItem: Class) => {
    this.setState({
      editingClass: classItem,
      openEditDialog: true
    });
    this.handleMenuClose();
  };

  handleCreateClass = (newClass: Omit<Class, 'id'>) => {
    const newId = Math.max(...this.state.classes.map(c => c.id)) + 1;
    this.setState({
      classes: [...this.state.classes, { ...newClass, id: newId }],
      openCreateDialog: false
    });
  };

  handleUpdateClass = (updatedClass: Class) => {
    this.setState({
      classes: this.state.classes.map(c => c.id === updatedClass.id ? updatedClass : c),
      openEditDialog: false,
      editingClass: null
    });
  };

  getStatusColor = (status: string) => {
    return status === 'Published' ? 'success' : 'warning';
  };

  getPassTypeColor = (passType: string) => {
    return passType === 'Learning' ? 'primary' : 'secondary';
  };

  render() {
    const {
      classes,
      searchTerm,
      openCreateDialog,
      openEditDialog,
      anchorEl,
      selectedClass,
      editingClass,
      viewingClass
    } = this.state;

    const filteredClasses = classes.filter(cls =>
      cls.name.toLowerCase().includes(searchTerm.toLowerCase())
    );

    if (viewingClass) {
      return <ClassView classData={viewingClass} onBack={this.handleBackToList} />;
    }

    return (
      <Box sx={{ p: 4, bgcolor: '#f8fafc', minHeight: '100vh' }}>
        {/* Header */}
        <Paper sx={{ p: 4, mb: 4, borderRadius: 3, boxShadow: '0 4px 20px rgba(0,0,0,0.08)' }}>
          <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
            <Box>
              <Typography variant="h3" sx={{ fontWeight: 700, color: '#25476a', mb: 1 }}>
                Self-Paced Learning
              </Typography>
              <Typography variant="body1" sx={{ color: '#64748b' }}>
                Manage and monitor self-paced learning classes
              </Typography>
            </Box>
            <Button
              variant="contained"
              startIcon={<Add />}
              onClick={() => this.setState({ openCreateDialog: true })}
              sx={{
                bgcolor: '#38aae1',
                '&:hover': { bgcolor: '#2e8bc4' },
                borderRadius: 2,
                px: 3,
                py: 1.5,
                textTransform: 'none',
                fontWeight: 600,
              }}
            >
              Create New Class
            </Button>
          </Box>
        </Paper>

        {/* Search and Filters */}
        <Paper sx={{ p: 3, mb: 4, borderRadius: 3, boxShadow: '0 4px 20px rgba(0,0,0,0.08)' }}>
          <Box sx={{ display: 'flex', gap: 2, alignItems: 'center' }}>
            <TextField
              placeholder="Search classes..."
              value={searchTerm}
              onChange={(e) => this.setState({ searchTerm: e.target.value })}
              sx={{ flex: 1 }}
              InputProps={{
                startAdornment: (
                  <InputAdornment position="start">
                    <Search sx={{ color: '#64748b' }} />
                  </InputAdornment>
                ),
              }}
            />
          </Box>
        </Paper>

        {/* Classes Table */}
        <Paper sx={{ borderRadius: 3, boxShadow: '0 4px 20px rgba(0,0,0,0.08)', overflow: 'hidden' }}>
          <TableContainer>
            <Table>
              <TableHead>
                <TableRow sx={{ bgcolor: '#f8fafc' }}>
                  <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Class Name</TableCell>
                  <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Courses</TableCell>
                  <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Revenue (KES)</TableCell>
                  <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Students</TableCell>
                  <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Pass Type</TableCell>
                  <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Status</TableCell>
                  <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Actions</TableCell>
                </TableRow>
              </TableHead>
              <TableBody>
                {filteredClasses.map((classItem) => (
                  <TableRow
                    key={classItem.id}
                    sx={{
                      '&:hover': { bgcolor: '#f8fafc' },
                      borderBottom: '1px solid #e2e8f0',
                    }}
                  >
                    <TableCell>
                      <Typography variant="subtitle1" sx={{ fontWeight: 600, color: '#25476a' }}>
                        {classItem.name}
                      </Typography>
                    </TableCell>
                    <TableCell>
                      <Typography variant="body2" sx={{ color: '#64748b' }}>
                        {classItem.coursesCount} courses
                      </Typography>
                    </TableCell>
                    <TableCell>
                      <Typography variant="body2" sx={{ fontWeight: 600, color: '#25476a' }}>
                        {classItem.revenue.toLocaleString()}
                      </Typography>
                    </TableCell>
                    <TableCell>
                      <Typography variant="body2" sx={{ color: '#64748b' }}>
                        {classItem.students}
                      </Typography>
                    </TableCell>
                    <TableCell>
                      <Chip
                        label={classItem.passType}
                        color={this.getPassTypeColor(classItem.passType) as any}
                        size="small"
                        sx={{ fontWeight: 600 }}
                      />
                    </TableCell>
                    <TableCell>
                      <Chip
                        label={classItem.status}
                        color={this.getStatusColor(classItem.status) as any}
                        size="small"
                        sx={{ fontWeight: 600 }}
                      />
                    </TableCell>
                    <TableCell>
                      <IconButton
                        onClick={(e) => this.handleMenuClick(e, classItem)}
                        sx={{ color: '#64748b' }}
                      >
                        <MoreVert />
                      </IconButton>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </TableContainer>
        </Paper>

        {/* Action Menu */}
        <Menu
          anchorEl={anchorEl}
          open={Boolean(anchorEl)}
          onClose={this.handleMenuClose}
          PaperProps={{
            sx: { minWidth: 160, borderRadius: 2 }
          }}
        >
          <MenuItem onClick={() => selectedClass && this.handleViewClass(selectedClass)}>
            <Visibility sx={{ mr: 1, fontSize: 20 }} />
            View Details
          </MenuItem>
          <MenuItem onClick={() => selectedClass && this.handleEditClass(selectedClass)}>
            <Edit sx={{ mr: 1, fontSize: 20 }} />
            Edit Class
          </MenuItem>
          <MenuItem sx={{ color: '#f44336' }}>
            <Delete sx={{ mr: 1, fontSize: 20 }} />
            Delete Class
          </MenuItem>
        </Menu>

        {/* Create Class Dialog */}
        <CreateClassDialog
          open={openCreateDialog}
          onClose={() => this.setState({ openCreateDialog: false })}
          onSave={this.handleCreateClass}
        />

        {/* Edit Class Dialog */}
        <CreateClassDialog
          open={openEditDialog}
          onClose={() => this.setState({ openEditDialog: false, editingClass: null })}
          onSave={this.handleUpdateClass}
          editingClass={editingClass}
        />
      </Box>
    );
  }
}