import React, { Component } from 'react';
import {
  Box,
  Typography,
  Card,
  CardContent,
  Grid,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Button,
  IconButton,
  Paper,
  Drawer,
  List,
  ListItem,
  ListItemButton,
  ListItemIcon,
  ListItemText,
  MenuItem,
  Select,
  FormControl,
  Chip,
  LinearProgress,
  Avatar,
} from '@mui/material';
import {
  Dashboard,
  School,
  People,
  Logout,
  ArrowForward,
  Description,
  Receipt,
  FolderOpen,
  Assessment,
  MoreVert,
  Add,
  TrendingUp,
  PlayCircle,
  CalendarMonth,
  Person,
  Schedule,
  CheckCircle,
  AccessTime,
  Error as ErrorIcon,
  Assignment,
  Article,
} from '@mui/icons-material';

const drawerWidth = 240;

interface LearnerReport {
  id: string;
  learnerName: string;
  registrationNumber: string;
  class: string;
  course: string;
  technicalMentor: string;
  progress: number;
  status: 'active' | 'completed' | 'pending';
}

interface Invoice {
  id: string;
  invoiceNumber: string;
  learnerName: string;
  amount: number;
  dueDate: string;
  status: 'paid' | 'pending' | 'overdue';
  courseName: string;
}

interface Project {
  id: string;
  title: string;
  learnerName: string;
  course: string;
  submittedDate: string;
  grade: string;
  status: 'submitted' | 'graded' | 'pending';
}

interface CourseProgress {
  id: string;
  learnerName: string;
  courseName: string;
  currentLesson: string;
  lessonsCompleted: number;
  totalLessons: number;
  progress: number;
  lastActive: string;
  status: 'on-track' | 'behind' | 'ahead';
}

interface MentorRequest {
  id: string;
  learnerName: string;
  mentorName: string;
  requestDate: string;
  scheduledDate: string;
  duration: number;
  topic: string;
  status: 'scheduled' | 'completed' | 'pending' | 'cancelled';
}

interface ParentDashboardState {
  selectedPage: string;
  selectedMonth: string;
  selectedYear: string;
  learnerReports: LearnerReport[];
  invoices: Invoice[];
  projects: Project[];
  courseProgress: CourseProgress[];
  mentorRequests: MentorRequest[];
}

export class ParentDashboard extends Component<{}, ParentDashboardState> {
  constructor(props: {}) {
    super(props);
    this.state = {
      selectedPage: 'dashboard',
      selectedMonth: 'April',
      selectedYear: '2024',
      learnerReports: [
        {
          id: '1',
          learnerName: 'Sarah Johnson',
          registrationNumber: 'DU2024001',
          class: 'Web Development Bootcamp',
          course: 'Full Stack Development',
          technicalMentor: 'Michael Chen',
          progress: 78,
          status: 'active',
        },
        {
          id: '2',
          learnerName: 'David Martinez',
          registrationNumber: 'DU2024002',
          class: 'Data Science Track',
          course: 'Python for Data Science',
          technicalMentor: 'Emily Rodriguez',
          progress: 92,
          status: 'active',
        },
        {
          id: '3',
          learnerName: 'Emma Wilson',
          registrationNumber: 'DU2024003',
          class: 'UI/UX Design',
          course: 'Advanced Design Principles',
          technicalMentor: 'James Wilson',
          progress: 100,
          status: 'completed',
        },
      ],
      invoices: [
        {
          id: '1',
          invoiceNumber: 'INV-2024-001',
          learnerName: 'Sarah Johnson',
          amount: 2500,
          dueDate: '2024-04-30',
          status: 'paid',
          courseName: 'Full Stack Development',
        },
        {
          id: '2',
          invoiceNumber: 'INV-2024-002',
          learnerName: 'David Martinez',
          amount: 3500,
          dueDate: '2024-05-15',
          status: 'pending',
          courseName: 'Python for Data Science',
        },
        {
          id: '3',
          invoiceNumber: 'INV-2024-003',
          learnerName: 'Emma Wilson',
          amount: 2800,
          dueDate: '2024-04-10',
          status: 'overdue',
          courseName: 'Advanced Design Principles',
        },
      ],
      projects: [
        {
          id: '1',
          title: 'E-commerce Website Project',
          learnerName: 'Sarah Johnson',
          course: 'Full Stack Development',
          submittedDate: '2024-04-15',
          grade: 'A',
          status: 'graded',
        },
        {
          id: '2',
          title: 'Data Analysis Dashboard',
          learnerName: 'David Martinez',
          course: 'Python for Data Science',
          submittedDate: '2024-04-18',
          grade: 'A+',
          status: 'graded',
        },
        {
          id: '3',
          title: 'Mobile App UI Design',
          learnerName: 'Emma Wilson',
          course: 'Advanced Design Principles',
          submittedDate: '2024-04-20',
          grade: 'Pending',
          status: 'submitted',
        },
      ],
      courseProgress: [
        {
          id: '1',
          learnerName: 'Sarah Johnson',
          courseName: 'Full Stack Development',
          currentLesson: 'Lesson 8: Advanced React Patterns',
          lessonsCompleted: 7,
          totalLessons: 12,
          progress: 58,
          lastActive: '2024-04-18',
          status: 'on-track',
        },
        {
          id: '2',
          learnerName: 'David Martinez',
          courseName: 'Python for Data Science',
          currentLesson: 'Lesson 11: Machine Learning Basics',
          lessonsCompleted: 10,
          totalLessons: 12,
          progress: 83,
          lastActive: '2024-04-19',
          status: 'ahead',
        },
        {
          id: '3',
          learnerName: 'Emma Wilson',
          courseName: 'Advanced Design Principles',
          currentLesson: 'Course Completed',
          lessonsCompleted: 12,
          totalLessons: 12,
          progress: 100,
          lastActive: '2024-04-17',
          status: 'on-track',
        },
      ],
      mentorRequests: [
        {
          id: '1',
          learnerName: 'Sarah Johnson',
          mentorName: 'Michael Chen',
          requestDate: '2024-04-15',
          scheduledDate: '2024-04-22 14:00',
          duration: 60,
          topic: 'React State Management',
          status: 'scheduled',
        },
        {
          id: '2',
          learnerName: 'David Martinez',
          mentorName: 'Emily Rodriguez',
          requestDate: '2024-04-16',
          scheduledDate: '2024-04-20 10:00',
          duration: 90,
          topic: 'Advanced Python Techniques',
          status: 'completed',
        },
        {
          id: '3',
          learnerName: 'Sarah Johnson',
          mentorName: 'Michael Chen',
          requestDate: '2024-04-18',
          scheduledDate: 'TBD',
          duration: 60,
          topic: 'API Integration Help',
          status: 'pending',
        },
        {
          id: '4',
          learnerName: 'David Martinez',
          mentorName: 'Emily Rodriguez',
          requestDate: '2024-04-19',
          scheduledDate: '2024-04-25 15:30',
          duration: 60,
          topic: 'Data Visualization Project Review',
          status: 'scheduled',
        },
      ],
    };
  }

  handlePageChange = (page: string) => {
    this.setState({ selectedPage: page });
  };

  handleMonthChange = (event: any) => {
    this.setState({ selectedMonth: event.target.value });
  };

  handleYearChange = (event: any) => {
    this.setState({ selectedYear: event.target.value });
  };

  getStatusColor = (status: string) => {
    switch (status) {
      case 'active':
      case 'paid':
      case 'graded':
        return { bg: '#dcfce7', color: '#166534' };
      case 'pending':
      case 'submitted':
        return { bg: '#fef3c7', color: '#92400e' };
      case 'completed':
        return { bg: '#e0f2fe', color: '#0284c7' };
      case 'overdue':
        return { bg: '#fee2e2', color: '#991b1b' };
      default:
        return { bg: '#f3f4f6', color: '#374151' };
    }
  };

  render() {
    const { selectedPage, selectedMonth, selectedYear, learnerReports, invoices, projects, courseProgress, mentorRequests } = this.state;

    const menuItems = [
      { text: 'Dashboard', icon: <Dashboard />, key: 'dashboard' },
      { text: 'Classes', icon: <School />, key: 'classes' },
      { text: 'Learners', icon: <People />, key: 'learners' },
    ];

    const months = [
      'January', 'February', 'March', 'April', 'May', 'June',
      'July', 'August', 'September', 'October', 'November', 'December'
    ];

    const years = ['2023', '2024', '2025'];

    const totalClasses = 3;
    const totalProjects = projects.length;
    const totalReports = learnerReports.length;
    const totalInvoices = invoices.length;
    const activeCourses = courseProgress.filter(c => c.progress < 100).length;
    const upcomingMentorSessions = mentorRequests.filter(r => r.status === 'scheduled').length;

    return (
      <Box sx={{ display: 'flex', minHeight: '100vh', bgcolor: '#f8fafc' }}>
        {/* Sidebar */}
        <Drawer
          variant="permanent"
          sx={{
            width: drawerWidth,
            flexShrink: 0,
            '& .MuiDrawer-paper': {
              width: drawerWidth,
              boxSizing: 'border-box',
              backgroundColor: '#ffffff',
              borderRight: '1px solid #e2e8f0',
            },
          }}
        >
          {/* Logo */}
          <Box sx={{ p: 3, borderBottom: '1px solid #e2e8f0' }}>
            <Typography
              variant="h5"
              sx={{
                color: '#25476a',
                fontWeight: 700,
                fontSize: '1.5rem',
                letterSpacing: '-0.02em',
              }}
            >
              digifunzi
            </Typography>
          </Box>

          {/* Menu Items */}
          <List sx={{ pt: 2, px: 2, flexGrow: 1 }}>
            {menuItems.map((item) => (
              <ListItem key={item.key} disablePadding sx={{ mb: 1 }}>
                <ListItemButton
                  onClick={() => this.handlePageChange(item.key)}
                  sx={{
                    borderRadius: 2,
                    backgroundColor: selectedPage === item.key ? '#38aae1' : 'transparent',
                    color: selectedPage === item.key ? '#ffffff' : '#64748b',
                    '&:hover': {
                      backgroundColor: selectedPage === item.key ? '#2c8fc9' : 'rgba(56, 170, 225, 0.1)',
                    },
                  }}
                >
                  <ListItemIcon sx={{ color: selectedPage === item.key ? '#ffffff' : '#64748b', minWidth: '40px' }}>
                    {item.icon}
                  </ListItemIcon>
                  <ListItemText
                    primary={item.text}
                    sx={{
                      '& .MuiListItemText-primary': {
                        fontWeight: selectedPage === item.key ? 600 : 500,
                      },
                    }}
                  />
                </ListItemButton>
              </ListItem>
            ))}
          </List>

          {/* Logout Button */}
          <Box sx={{ p: 2, borderTop: '1px solid #e2e8f0' }}>
            <ListItemButton
              sx={{
                borderRadius: 2,
                color: '#64748b',
                '&:hover': {
                  backgroundColor: 'rgba(239, 68, 68, 0.1)',
                  color: '#ef4444',
                },
              }}
            >
              <ListItemIcon sx={{ color: 'inherit', minWidth: '40px' }}>
                <Logout />
              </ListItemIcon>
              <ListItemText primary="Logout" />
            </ListItemButton>
          </Box>
        </Drawer>

        {/* Main Content */}
        <Box component="main" sx={{ flexGrow: 1, p: 4 }}>
          {/* Summary Cards */}
          <Grid container spacing={3} sx={{ mb: 4 }}>
            {/* Classes Card */}
            <Grid item xs={12} sm={6} lg={3}>
              <Card
                sx={{
                  borderRadius: 3,
                  border: '1px solid #e2e8f0',
                  boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
                  height: '100%',
                  transition: 'all 0.3s',
                  '&:hover': {
                    boxShadow: '0 8px 16px rgba(56, 170, 225, 0.15)',
                    transform: 'translateY(-4px)',
                  },
                }}
              >
                <CardContent sx={{ p: 3 }}>
                  <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                    <Box sx={{ p: 1.5, bgcolor: 'rgba(56, 170, 225, 0.1)', borderRadius: 2 }}>
                      <School sx={{ color: '#38aae1', fontSize: 28 }} />
                    </Box>
                    <Chip
                      label={`${totalClasses} Enrolled`}
                      size="small"
                      sx={{
                        bgcolor: '#dbeafe',
                        color: '#1e40af',
                        fontWeight: 600,
                        fontSize: '0.75rem',
                      }}
                    />
                  </Box>
                  <Typography variant="body2" sx={{ color: '#64748b', mb: 1, fontWeight: 500 }}>
                    Classes
                  </Typography>
                  <Typography variant="h3" sx={{ color: '#25476a', fontWeight: 700 }}>
                    {totalClasses}
                  </Typography>
                </CardContent>
              </Card>
            </Grid>

            {/* Class Projects Card */}
            <Grid item xs={12} sm={6} lg={3}>
              <Card
                sx={{
                  borderRadius: 3,
                  border: '1px solid #e2e8f0',
                  boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
                  height: '100%',
                  transition: 'all 0.3s',
                  '&:hover': {
                    boxShadow: '0 8px 16px rgba(254, 177, 57, 0.15)',
                    transform: 'translateY(-4px)',
                  },
                }}
              >
                <CardContent sx={{ p: 3 }}>
                  <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                    <Box sx={{ p: 1.5, bgcolor: 'rgba(254, 177, 57, 0.1)', borderRadius: 2 }}>
                      <Assignment sx={{ color: '#feb139', fontSize: 28 }} />
                    </Box>
                    <Chip
                      label={`${projects.filter(p => p.status === 'graded').length} Graded`}
                      size="small"
                      sx={{
                        bgcolor: '#dcfce7',
                        color: '#166534',
                        fontWeight: 600,
                        fontSize: '0.75rem',
                      }}
                    />
                  </Box>
                  <Typography variant="body2" sx={{ color: '#64748b', mb: 1, fontWeight: 500 }}>
                    Class Projects
                  </Typography>
                  <Typography variant="h3" sx={{ color: '#25476a', fontWeight: 700 }}>
                    {totalProjects}
                  </Typography>
                </CardContent>
              </Card>
            </Grid>

            {/* Reports Card */}
            <Grid item xs={12} sm={6} lg={3}>
              <Card
                sx={{
                  borderRadius: 3,
                  border: '1px solid #e2e8f0',
                  boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
                  height: '100%',
                  transition: 'all 0.3s',
                  '&:hover': {
                    boxShadow: '0 8px 16px rgba(139, 92, 246, 0.15)',
                    transform: 'translateY(-4px)',
                  },
                }}
              >
                <CardContent sx={{ p: 3 }}>
                  <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                    <Box sx={{ p: 1.5, bgcolor: 'rgba(139, 92, 246, 0.1)', borderRadius: 2 }}>
                      <Article sx={{ color: '#8b5cf6', fontSize: 28 }} />
                    </Box>
                    <Chip
                      label="New Available"
                      size="small"
                      sx={{
                        bgcolor: '#e9d5ff',
                        color: '#6b21a8',
                        fontWeight: 600,
                        fontSize: '0.75rem',
                      }}
                    />
                  </Box>
                  <Typography variant="body2" sx={{ color: '#64748b', mb: 1, fontWeight: 500 }}>
                    Reports
                  </Typography>
                  <Typography variant="h3" sx={{ color: '#25476a', fontWeight: 700 }}>
                    {totalReports}
                  </Typography>
                </CardContent>
              </Card>
            </Grid>

            {/* Invoices Card */}
            <Grid item xs={12} sm={6} lg={3}>
              <Card
                sx={{
                  borderRadius: 3,
                  border: '1px solid #e2e8f0',
                  boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
                  height: '100%',
                  transition: 'all 0.3s',
                  '&:hover': {
                    boxShadow: '0 8px 16px rgba(34, 197, 94, 0.15)',
                    transform: 'translateY(-4px)',
                  },
                }}
              >
                <CardContent sx={{ p: 3 }}>
                  <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                    <Box sx={{ p: 1.5, bgcolor: 'rgba(34, 197, 94, 0.1)', borderRadius: 2 }}>
                      <Receipt sx={{ color: '#22c55e', fontSize: 28 }} />
                    </Box>
                    <Chip
                      label={`${invoices.filter(i => i.status === 'paid').length} Paid`}
                      size="small"
                      sx={{
                        bgcolor: '#dcfce7',
                        color: '#166534',
                        fontWeight: 600,
                        fontSize: '0.75rem',
                      }}
                    />
                  </Box>
                  <Typography variant="body2" sx={{ color: '#64748b', mb: 1, fontWeight: 500 }}>
                    Invoices
                  </Typography>
                  <Typography variant="h3" sx={{ color: '#25476a', fontWeight: 700 }}>
                    {totalInvoices}
                  </Typography>
                </CardContent>
              </Card>
            </Grid>
          </Grid>

          {/* Course Progress Section */}
          <Card
            sx={{
              borderRadius: 3,
              border: '1px solid #e2e8f0',
              boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
              mb: 3,
            }}
          >
            <CardContent sx={{ p: 3 }}>
              <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
                <Box>
                  <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 700, mb: 0.5 }}>
                    Self-Paced Course Progress
                  </Typography>
                  <Typography variant="body2" sx={{ color: '#64748b' }}>
                    Track your learners' course progress and current lessons
                  </Typography>
                </Box>
              </Box>
              <Grid container spacing={3}>
                {courseProgress.map((course) => {
                  const statusColor = course.status === 'ahead' ? '#22c55e' : course.status === 'on-track' ? '#38aae1' : '#f59e0b';
                  const statusBg = course.status === 'ahead' ? '#dcfce7' : course.status === 'on-track' ? '#dbeafe' : '#fef3c7';
                  
                  return (
                    <Grid item xs={12} md={6} lg={4} key={course.id}>
                      <Card
                        sx={{
                          p: 3,
                          bgcolor: '#ffffff',
                          border: '1px solid #e2e8f0',
                          borderRadius: 2,
                          height: '100%',
                          transition: 'all 0.2s',
                          '&:hover': {
                            boxShadow: '0 4px 12px rgba(0, 0, 0, 0.08)',
                            transform: 'translateY(-2px)',
                          },
                        }}
                      >
                        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                          <Avatar
                            sx={{
                              width: 44,
                              height: 44,
                              bgcolor: '#38aae1',
                              fontSize: '1rem',
                              fontWeight: 700,
                            }}
                          >
                            {course.learnerName.split(' ').map(n => n[0]).join('')}
                          </Avatar>
                          <Chip
                            label={course.status.replace('-', ' ').toUpperCase()}
                            size="small"
                            sx={{
                              bgcolor: statusBg,
                              color: statusColor,
                              fontWeight: 600,
                              fontSize: '0.7rem',
                            }}
                          />
                        </Box>
                        <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 700, mb: 0.5 }}>
                          {course.learnerName}
                        </Typography>
                        <Typography variant="body2" sx={{ color: '#64748b', mb: 2 }}>
                          {course.courseName}
                        </Typography>
                        <Box sx={{ mb: 2 }}>
                          <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 1 }}>
                            <Typography variant="body2" sx={{ color: '#64748b', fontSize: '0.875rem' }}>
                              {course.currentLesson}
                            </Typography>
                            <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 700 }}>
                              {course.progress}%
                            </Typography>
                          </Box>
                          <LinearProgress
                            variant="determinate"
                            value={course.progress}
                            sx={{
                              height: 8,
                              borderRadius: 4,
                              bgcolor: '#e2e8f0',
                              '& .MuiLinearProgress-bar': {
                                bgcolor: course.progress === 100 ? '#22c55e' : '#38aae1',
                                borderRadius: 4,
                              },
                            }}
                          />
                        </Box>
                        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', pt: 2, borderTop: '1px solid #f1f5f9' }}>
                          <Typography variant="caption" sx={{ color: '#94a3b8', display: 'flex', alignItems: 'center' }}>
                            <AccessTime sx={{ fontSize: 14, mr: 0.5 }} />
                            Last active: {new Date(course.lastActive).toLocaleDateString()}
                          </Typography>
                          <Typography variant="caption" sx={{ color: '#64748b', fontWeight: 600 }}>
                            {course.lessonsCompleted}/{course.totalLessons} lessons
                          </Typography>
                        </Box>
                      </Card>
                    </Grid>
                  );
                })}
              </Grid>
            </CardContent>
          </Card>

          {/* Technical Mentor Hours Section */}
          <Card
            sx={{
              borderRadius: 3,
              border: '1px solid #e2e8f0',
              boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
              mb: 3,
            }}
          >
            <CardContent sx={{ p: 3 }}>
              <Box sx={{ mb: 3 }}>
                <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 700, mb: 0.5 }}>
                  Technical Mentor Hour Requests
                </Typography>
                <Typography variant="body2" sx={{ color: '#64748b' }}>
                  View and manage technical mentor session requests
                </Typography>
              </Box>
              <TableContainer>
                <Table>
                  <TableHead>
                    <TableRow sx={{ bgcolor: '#f8fafc' }}>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Learner
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Mentor
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Topic
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Requested
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Scheduled
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Duration
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Status
                      </TableCell>
                    </TableRow>
                  </TableHead>
                  <TableBody>
                    {mentorRequests.map((request) => {
                      const statusColors = {
                        scheduled: { bg: '#dbeafe', color: '#1e40af', icon: <Schedule sx={{ fontSize: 16 }} /> },
                        completed: { bg: '#dcfce7', color: '#166534', icon: <CheckCircle sx={{ fontSize: 16 }} /> },
                        pending: { bg: '#fef3c7', color: '#92400e', icon: <AccessTime sx={{ fontSize: 16 }} /> },
                        cancelled: { bg: '#fee2e2', color: '#991b1b', icon: <ErrorIcon sx={{ fontSize: 16 }} /> },
                      };
                      const statusColor = statusColors[request.status] || statusColors.pending;

                      return (
                        <TableRow key={request.id} sx={{ '&:hover': { bgcolor: '#f8fafc' } }}>
                          <TableCell sx={{ border: 'none' }}>
                            <Box sx={{ display: 'flex', alignItems: 'center' }}>
                              <Avatar
                                sx={{
                                  width: 32,
                                  height: 32,
                                  bgcolor: '#38aae1',
                                  mr: 1.5,
                                  fontSize: '0.75rem',
                                  fontWeight: 600,
                                }}
                              >
                                {request.learnerName.split(' ').map(n => n[0]).join('')}
                              </Avatar>
                              <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                                {request.learnerName}
                              </Typography>
                            </Box>
                          </TableCell>
                          <TableCell sx={{ border: 'none', color: '#64748b' }}>
                            <Box sx={{ display: 'flex', alignItems: 'center' }}>
                              <Person sx={{ fontSize: 18, mr: 0.5, color: '#38aae1' }} />
                              {request.mentorName}
                            </Box>
                          </TableCell>
                          <TableCell sx={{ border: 'none', color: '#25476a', fontWeight: 500 }}>
                            {request.topic}
                          </TableCell>
                          <TableCell sx={{ border: 'none', color: '#64748b' }}>
                            {new Date(request.requestDate).toLocaleDateString()}
                          </TableCell>
                          <TableCell sx={{ border: 'none', color: '#64748b', fontWeight: 500 }}>
                            {request.scheduledDate}
                          </TableCell>
                          <TableCell sx={{ border: 'none', color: '#64748b' }}>
                            {request.duration} min
                          </TableCell>
                          <TableCell sx={{ border: 'none' }}>
                            <Chip
                              icon={statusColor.icon}
                              label={request.status.toUpperCase()}
                              size="small"
                              sx={{
                                bgcolor: statusColor.bg,
                                color: statusColor.color,
                                fontWeight: 600,
                                fontSize: '0.75rem',
                              }}
                            />
                          </TableCell>
                        </TableRow>
                      );
                    })}
                  </TableBody>
                </Table>
              </TableContainer>
            </CardContent>
          </Card>

          {/* Invoices Section */}
          <Card
            sx={{
              borderRadius: 3,
              border: '1px solid #e2e8f0',
              boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
              mb: 3,
            }}
          >
            <CardContent sx={{ p: 3 }}>
              <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
                <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 700 }}>
                  Invoices
                </Typography>
                <Button
                  endIcon={<ArrowForward sx={{ fontSize: 16 }} />}
                  sx={{
                    color: '#38aae1',
                    textTransform: 'none',
                    fontSize: '0.875rem',
                    fontWeight: 600,
                    '&:hover': {
                      backgroundColor: 'transparent',
                      textDecoration: 'underline',
                    },
                  }}
                >
                  Show more invoices
                </Button>
              </Box>
              <Grid container spacing={2}>
                {invoices.map((invoice) => {
                  const statusColor = this.getStatusColor(invoice.status);
                  return (
                    <Grid item xs={12} md={6} lg={4} key={invoice.id}>
                      <Card
                        sx={{
                          p: 2.5,
                          bgcolor: '#ffffff',
                          border: '1px solid #e2e8f0',
                          borderRadius: 2,
                          transition: 'all 0.2s',
                          '&:hover': {
                            boxShadow: '0 4px 12px rgba(0, 0, 0, 0.08)',
                            transform: 'translateY(-2px)',
                          },
                        }}
                      >
                        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                          <Box>
                            <Typography variant="body2" sx={{ color: '#64748b', mb: 0.5 }}>
                              {invoice.invoiceNumber}
                            </Typography>
                            <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 700 }}>
                              ${invoice.amount.toLocaleString()}
                            </Typography>
                          </Box>
                          <Chip
                            label={invoice.status.toUpperCase()}
                            size="small"
                            sx={{
                              bgcolor: statusColor.bg,
                              color: statusColor.color,
                              fontWeight: 600,
                              fontSize: '0.75rem',
                              height: '24px',
                            }}
                          />
                        </Box>
                        <Box sx={{ mb: 1.5 }}>
                          <Typography variant="body2" sx={{ color: '#64748b', mb: 0.5 }}>
                            <strong>Learner:</strong> {invoice.learnerName}
                          </Typography>
                          <Typography variant="body2" sx={{ color: '#64748b', mb: 0.5 }}>
                            <strong>Course:</strong> {invoice.courseName}
                          </Typography>
                          <Typography variant="body2" sx={{ color: '#64748b' }}>
                            <strong>Due Date:</strong> {new Date(invoice.dueDate).toLocaleDateString()}
                          </Typography>
                        </Box>
                        <Button
                          fullWidth
                          variant="outlined"
                          size="small"
                          sx={{
                            borderColor: '#38aae1',
                            color: '#38aae1',
                            textTransform: 'none',
                            fontWeight: 600,
                            '&:hover': {
                              borderColor: '#2c8fc9',
                              backgroundColor: 'rgba(56, 170, 225, 0.05)',
                            },
                          }}
                        >
                          View Details
                        </Button>
                      </Card>
                    </Grid>
                  );
                })}
              </Grid>
            </CardContent>
          </Card>

          {/* Class Projects Section */}
          <Card
            sx={{
              borderRadius: 3,
              border: '1px solid #e2e8f0',
              boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
              mb: 3,
            }}
          >
            <CardContent sx={{ p: 3 }}>
              <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
                <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 700 }}>
                  Class Projects
                </Typography>
                <Box sx={{ display: 'flex', gap: 2, alignItems: 'center' }}>
                  <FormControl size="small" sx={{ minWidth: 120 }}>
                    <Select
                      value={selectedMonth}
                      onChange={this.handleMonthChange}
                      sx={{
                        borderRadius: 2,
                        bgcolor: '#ffffff',
                        '& .MuiOutlinedInput-notchedOutline': {
                          borderColor: '#e2e8f0',
                        },
                      }}
                    >
                      {months.map((month) => (
                        <MenuItem key={month} value={month}>
                          {month}
                        </MenuItem>
                      ))}
                    </Select>
                  </FormControl>
                  <FormControl size="small" sx={{ minWidth: 100 }}>
                    <Select
                      value={selectedYear}
                      onChange={this.handleYearChange}
                      sx={{
                        borderRadius: 2,
                        bgcolor: '#ffffff',
                        '& .MuiOutlinedInput-notchedOutline': {
                          borderColor: '#e2e8f0',
                        },
                      }}
                    >
                      {years.map((year) => (
                        <MenuItem key={year} value={year}>
                          {year}
                        </MenuItem>
                      ))}
                    </Select>
                  </FormControl>
                  <Button
                    endIcon={<Add />}
                    sx={{
                      color: '#38aae1',
                      textTransform: 'none',
                      fontSize: '0.875rem',
                      fontWeight: 600,
                      '&:hover': {
                        backgroundColor: 'rgba(56, 170, 225, 0.1)',
                      },
                    }}
                  >
                    Show more projects
                  </Button>
                </Box>
              </Box>
              <TableContainer>
                <Table>
                  <TableHead>
                    <TableRow sx={{ bgcolor: '#f8fafc' }}>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Project Title
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Learner
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Course
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Submitted
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Grade
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Status
                      </TableCell>
                    </TableRow>
                  </TableHead>
                  <TableBody>
                    {projects.map((project) => {
                      const statusColor = this.getStatusColor(project.status);
                      return (
                        <TableRow key={project.id} sx={{ '&:hover': { bgcolor: '#f8fafc' } }}>
                          <TableCell sx={{ border: 'none', color: '#25476a', fontWeight: 600 }}>
                            {project.title}
                          </TableCell>
                          <TableCell sx={{ border: 'none', color: '#64748b' }}>
                            {project.learnerName}
                          </TableCell>
                          <TableCell sx={{ border: 'none', color: '#64748b' }}>
                            {project.course}
                          </TableCell>
                          <TableCell sx={{ border: 'none', color: '#64748b' }}>
                            {new Date(project.submittedDate).toLocaleDateString()}
                          </TableCell>
                          <TableCell sx={{ border: 'none' }}>
                            <Chip
                              label={project.grade}
                              size="small"
                              sx={{
                                bgcolor: project.grade === 'Pending' ? '#fef3c7' : '#dcfce7',
                                color: project.grade === 'Pending' ? '#92400e' : '#166534',
                                fontWeight: 600,
                                fontSize: '0.75rem',
                              }}
                            />
                          </TableCell>
                          <TableCell sx={{ border: 'none' }}>
                            <Chip
                              label={project.status.charAt(0).toUpperCase() + project.status.slice(1)}
                              size="small"
                              sx={{
                                bgcolor: statusColor.bg,
                                color: statusColor.color,
                                fontWeight: 600,
                                fontSize: '0.75rem',
                              }}
                            />
                          </TableCell>
                        </TableRow>
                      );
                    })}
                  </TableBody>
                </Table>
              </TableContainer>
            </CardContent>
          </Card>

          {/* Learner Reports Section */}
          <Card
            sx={{
              borderRadius: 3,
              border: '1px solid #e2e8f0',
              boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
            }}
          >
            <CardContent sx={{ p: 3 }}>
              <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 700, mb: 3 }}>
                Learner Reports
              </Typography>
              <TableContainer>
                <Table>
                  <TableHead>
                    <TableRow sx={{ bgcolor: '#f8fafc' }}>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Learner Name
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Registration Number
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Course
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Technical Mentor
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Progress
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Status
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Action
                      </TableCell>
                    </TableRow>
                  </TableHead>
                  <TableBody>
                    {learnerReports.length === 0 ? (
                      <TableRow>
                        <TableCell colSpan={7} sx={{ border: 'none', py: 6 }}>
                          <Box sx={{ textAlign: 'center', color: '#94a3b8' }}>
                            <People sx={{ fontSize: 64, color: '#cbd5e1', mb: 2 }} />
                            <Typography variant="body1">No data</Typography>
                          </Box>
                        </TableCell>
                      </TableRow>
                    ) : (
                      learnerReports.map((report) => {
                        const statusColor = this.getStatusColor(report.status);
                        return (
                          <TableRow key={report.id} sx={{ '&:hover': { bgcolor: '#f8fafc' } }}>
                            <TableCell sx={{ border: 'none' }}>
                              <Box sx={{ display: 'flex', alignItems: 'center' }}>
                                <Avatar
                                  sx={{
                                    width: 36,
                                    height: 36,
                                    bgcolor: '#38aae1',
                                    mr: 1.5,
                                    fontSize: '0.875rem',
                                    fontWeight: 600,
                                  }}
                                >
                                  {report.learnerName.split(' ').map(n => n[0]).join('')}
                                </Avatar>
                                <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                                  {report.learnerName}
                                </Typography>
                              </Box>
                            </TableCell>
                            <TableCell sx={{ border: 'none', color: '#64748b' }}>
                              {report.registrationNumber}
                            </TableCell>
                            <TableCell sx={{ border: 'none', color: '#64748b' }}>
                              <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 500 }}>
                                {report.course}
                              </Typography>
                              <Typography variant="caption" sx={{ color: '#94a3b8' }}>
                                {report.class}
                              </Typography>
                            </TableCell>
                            <TableCell sx={{ border: 'none', color: '#64748b' }}>
                              {report.technicalMentor}
                            </TableCell>
                            <TableCell sx={{ border: 'none' }}>
                              <Box sx={{ minWidth: 120 }}>
                                <Box sx={{ display: 'flex', alignItems: 'center', mb: 0.5 }}>
                                  <TrendingUp sx={{ fontSize: 16, color: '#38aae1', mr: 0.5 }} />
                                  <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                                    {report.progress}%
                                  </Typography>
                                </Box>
                                <LinearProgress
                                  variant="determinate"
                                  value={report.progress}
                                  sx={{
                                    height: 6,
                                    borderRadius: 3,
                                    bgcolor: '#e2e8f0',
                                    '& .MuiLinearProgress-bar': {
                                      bgcolor: report.progress === 100 ? '#4caf50' : '#38aae1',
                                      borderRadius: 3,
                                    },
                                  }}
                                />
                              </Box>
                            </TableCell>
                            <TableCell sx={{ border: 'none' }}>
                              <Chip
                                label={report.status.charAt(0).toUpperCase() + report.status.slice(1)}
                                size="small"
                                sx={{
                                  bgcolor: statusColor.bg,
                                  color: statusColor.color,
                                  fontWeight: 600,
                                  fontSize: '0.75rem',
                                }}
                              />
                            </TableCell>
                            <TableCell sx={{ border: 'none' }}>
                              <IconButton size="small" sx={{ color: '#64748b' }}>
                                <MoreVert />
                              </IconButton>
                            </TableCell>
                          </TableRow>
                        );
                      })
                    )}
                  </TableBody>
                </Table>
              </TableContainer>
            </CardContent>
          </Card>
        </Box>
      </Box>
    );
  }
}
