import React, { Component } from 'react';
import {
  Box,
  Typography,
  Card,
  CardContent,
  Grid,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Button,
  IconButton,
  Drawer,
  List,
  ListItem,
  ListItemButton,
  ListItemIcon,
  ListItemText,
  Chip,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Divider,
  Tab,
  Tabs,
} from '@mui/material';
import {
  Dashboard,
  School,
  People,
  Logout,
  Receipt,
  Payment,
  Download,
  Visibility,
  CreditCard,
  AccountBalance,
  Description,
  CheckCircle,
  Warning,
  Error as ErrorIcon,
} from '@mui/icons-material';

const drawerWidth = 240;

interface Invoice {
  id: string;
  invoiceNumber: string;
  learnerName: string;
  amount: number;
  dueDate: string;
  issueDate: string;
  status: 'paid' | 'pending' | 'overdue' | 'partially-paid';
  courseName: string;
  items: InvoiceItem[];
  paidAmount?: number;
}

interface InvoiceItem {
  description: string;
  quantity: number;
  unitPrice: number;
  total: number;
}

interface ParentBillingState {
  selectedPage: string;
  selectedTab: number;
  invoices: Invoice[];
  selectedInvoice: Invoice | null;
  invoiceDialogOpen: boolean;
}

export class ParentBilling extends Component<{}, ParentBillingState> {
  constructor(props: {}) {
    super(props);
    this.state = {
      selectedPage: 'billing',
      selectedTab: 0,
      selectedInvoice: null,
      invoiceDialogOpen: false,
      invoices: [
        {
          id: '1',
          invoiceNumber: 'INV-2024-001',
          learnerName: 'Sarah Johnson',
          amount: 2500,
          dueDate: '2024-05-30',
          issueDate: '2024-04-30',
          status: 'paid',
          courseName: 'Full Stack Development',
          items: [
            { description: 'Course Fee - Full Stack Development', quantity: 1, unitPrice: 2000, total: 2000 },
            { description: 'Learning Materials', quantity: 1, unitPrice: 300, total: 300 },
            { description: 'Platform Access (3 months)', quantity: 1, unitPrice: 200, total: 200 },
          ],
          paidAmount: 2500,
        },
        {
          id: '2',
          invoiceNumber: 'INV-2024-002',
          learnerName: 'David Martinez',
          amount: 3500,
          dueDate: '2024-05-15',
          issueDate: '2024-04-15',
          status: 'pending',
          courseName: 'Python for Data Science',
          items: [
            { description: 'Course Fee - Python for Data Science', quantity: 1, unitPrice: 2800, total: 2800 },
            { description: 'Learning Materials', quantity: 1, unitPrice: 400, total: 400 },
            { description: 'Platform Access (3 months)', quantity: 1, unitPrice: 300, total: 300 },
          ],
        },
        {
          id: '3',
          invoiceNumber: 'INV-2024-003',
          learnerName: 'Emma Wilson',
          amount: 2800,
          dueDate: '2024-04-10',
          issueDate: '2024-03-10',
          status: 'overdue',
          courseName: 'Advanced Design Principles',
          items: [
            { description: 'Course Fee - Advanced Design Principles', quantity: 1, unitPrice: 2200, total: 2200 },
            { description: 'Learning Materials', quantity: 1, unitPrice: 350, total: 350 },
            { description: 'Platform Access (3 months)', quantity: 1, unitPrice: 250, total: 250 },
          ],
        },
        {
          id: '4',
          invoiceNumber: 'INV-2024-004',
          learnerName: 'Sarah Johnson',
          amount: 1500,
          dueDate: '2024-05-20',
          issueDate: '2024-04-20',
          status: 'partially-paid',
          courseName: 'Advanced JavaScript',
          items: [
            { description: 'Course Fee - Advanced JavaScript', quantity: 1, unitPrice: 1200, total: 1200 },
            { description: 'Learning Materials', quantity: 1, unitPrice: 300, total: 300 },
          ],
          paidAmount: 800,
        },
      ],
    };
  }

  handlePageChange = (page: string) => {
    this.setState({ selectedPage: page });
  };

  handleTabChange = (event: any, newValue: number) => {
    this.setState({ selectedTab: newValue });
  };

  handleViewInvoice = (invoice: Invoice) => {
    this.setState({ selectedInvoice: invoice, invoiceDialogOpen: true });
  };

  handleCloseInvoiceDialog = () => {
    this.setState({ invoiceDialogOpen: false });
  };

  getStatusColor = (status: string) => {
    switch (status) {
      case 'paid':
        return { bg: '#dcfce7', color: '#166534', icon: <CheckCircle sx={{ fontSize: 16 }} /> };
      case 'pending':
        return { bg: '#fef3c7', color: '#92400e', icon: <Warning sx={{ fontSize: 16 }} /> };
      case 'overdue':
        return { bg: '#fee2e2', color: '#991b1b', icon: <ErrorIcon sx={{ fontSize: 16 }} /> };
      case 'partially-paid':
        return { bg: '#dbeafe', color: '#1e40af', icon: <Payment sx={{ fontSize: 16 }} /> };
      default:
        return { bg: '#f3f4f6', color: '#374151', icon: null };
    }
  };

  getFilteredInvoices = () => {
    const { selectedTab, invoices } = this.state;
    switch (selectedTab) {
      case 0: // All
        return invoices;
      case 1: // Paid
        return invoices.filter(inv => inv.status === 'paid');
      case 2: // Pending
        return invoices.filter(inv => inv.status === 'pending' || inv.status === 'partially-paid');
      case 3: // Overdue
        return invoices.filter(inv => inv.status === 'overdue');
      default:
        return invoices;
    }
  };

  render() {
    const { selectedPage, selectedTab, invoices, selectedInvoice, invoiceDialogOpen } = this.state;

    const menuItems = [
      { text: 'Dashboard', icon: <Dashboard />, key: 'dashboard' },
      { text: 'Billing', icon: <Receipt />, key: 'billing' },
      { text: 'Learners', icon: <People />, key: 'learners' },
    ];

    const totalAmount = invoices.reduce((sum, inv) => sum + inv.amount, 0);
    const paidAmount = invoices.filter(inv => inv.status === 'paid').reduce((sum, inv) => sum + inv.amount, 0);
    const pendingAmount = invoices.filter(inv => inv.status === 'pending').reduce((sum, inv) => sum + inv.amount, 0);
    const overdueAmount = invoices.filter(inv => inv.status === 'overdue').reduce((sum, inv) => sum + inv.amount, 0);

    const filteredInvoices = this.getFilteredInvoices();

    return (
      <Box sx={{ display: 'flex', minHeight: '100vh', bgcolor: '#f8fafc' }}>
        {/* Sidebar */}
        <Drawer
          variant="permanent"
          sx={{
            width: drawerWidth,
            flexShrink: 0,
            '& .MuiDrawer-paper': {
              width: drawerWidth,
              boxSizing: 'border-box',
              backgroundColor: '#ffffff',
              borderRight: '1px solid #e2e8f0',
            },
          }}
        >
          <Box sx={{ p: 3, borderBottom: '1px solid #e2e8f0' }}>
            <Typography variant="h5" sx={{ color: '#25476a', fontWeight: 700, fontSize: '1.5rem' }}>
              digifunzi
            </Typography>
          </Box>

          <List sx={{ pt: 2, px: 2, flexGrow: 1 }}>
            {menuItems.map((item) => (
              <ListItem key={item.key} disablePadding sx={{ mb: 1 }}>
                <ListItemButton
                  onClick={() => this.handlePageChange(item.key)}
                  sx={{
                    borderRadius: 2,
                    backgroundColor: selectedPage === item.key ? '#38aae1' : 'transparent',
                    color: selectedPage === item.key ? '#ffffff' : '#64748b',
                    '&:hover': {
                      backgroundColor: selectedPage === item.key ? '#2c8fc9' : 'rgba(56, 170, 225, 0.1)',
                    },
                  }}
                >
                  <ListItemIcon sx={{ color: selectedPage === item.key ? '#ffffff' : '#64748b', minWidth: '40px' }}>
                    {item.icon}
                  </ListItemIcon>
                  <ListItemText primary={item.text} />
                </ListItemButton>
              </ListItem>
            ))}
          </List>

          <Box sx={{ p: 2, borderTop: '1px solid #e2e8f0' }}>
            <ListItemButton
              sx={{
                borderRadius: 2,
                color: '#64748b',
                '&:hover': {
                  backgroundColor: 'rgba(239, 68, 68, 0.1)',
                  color: '#ef4444',
                },
              }}
            >
              <ListItemIcon sx={{ color: 'inherit', minWidth: '40px' }}>
                <Logout />
              </ListItemIcon>
              <ListItemText primary="Logout" />
            </ListItemButton>
          </Box>
        </Drawer>

        {/* Main Content */}
        <Box component="main" sx={{ flexGrow: 1, p: 4 }}>
          {/* Header */}
          <Box sx={{ mb: 4 }}>
            <Typography variant="h4" sx={{ color: '#25476a', fontWeight: 700, mb: 1 }}>
              Billing & Invoices
            </Typography>
            <Typography variant="body1" sx={{ color: '#64748b' }}>
              Manage payments and view invoice history
            </Typography>
          </Box>

          {/* Summary Cards */}
          <Grid container spacing={3} sx={{ mb: 4 }}>
            <Grid item xs={12} sm={6} lg={3}>
              <Card sx={{ borderRadius: 3, border: '1px solid #e2e8f0', boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)' }}>
                <CardContent sx={{ p: 3 }}>
                  <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                    <Box sx={{ p: 1.5, bgcolor: '#e0f2fe', borderRadius: 2 }}>
                      <Receipt sx={{ color: '#38aae1', fontSize: 24 }} />
                    </Box>
                  </Box>
                  <Typography variant="body2" sx={{ color: '#64748b', mb: 1 }}>
                    Total Billed
                  </Typography>
                  <Typography variant="h4" sx={{ color: '#25476a', fontWeight: 700 }}>
                    ${totalAmount.toLocaleString()}
                  </Typography>
                </CardContent>
              </Card>
            </Grid>

            <Grid item xs={12} sm={6} lg={3}>
              <Card sx={{ borderRadius: 3, border: '1px solid #e2e8f0', boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)' }}>
                <CardContent sx={{ p: 3 }}>
                  <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                    <Box sx={{ p: 1.5, bgcolor: '#dcfce7', borderRadius: 2 }}>
                      <CheckCircle sx={{ color: '#16a34a', fontSize: 24 }} />
                    </Box>
                  </Box>
                  <Typography variant="body2" sx={{ color: '#64748b', mb: 1 }}>
                    Paid
                  </Typography>
                  <Typography variant="h4" sx={{ color: '#16a34a', fontWeight: 700 }}>
                    ${paidAmount.toLocaleString()}
                  </Typography>
                </CardContent>
              </Card>
            </Grid>

            <Grid item xs={12} sm={6} lg={3}>
              <Card sx={{ borderRadius: 3, border: '1px solid #e2e8f0', boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)' }}>
                <CardContent sx={{ p: 3 }}>
                  <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                    <Box sx={{ p: 1.5, bgcolor: '#fef3c7', borderRadius: 2 }}>
                      <Warning sx={{ color: '#d97706', fontSize: 24 }} />
                    </Box>
                  </Box>
                  <Typography variant="body2" sx={{ color: '#64748b', mb: 1 }}>
                    Pending
                  </Typography>
                  <Typography variant="h4" sx={{ color: '#d97706', fontWeight: 700 }}>
                    ${pendingAmount.toLocaleString()}
                  </Typography>
                </CardContent>
              </Card>
            </Grid>

            <Grid item xs={12} sm={6} lg={3}>
              <Card sx={{ borderRadius: 3, border: '1px solid #e2e8f0', boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)' }}>
                <CardContent sx={{ p: 3 }}>
                  <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                    <Box sx={{ p: 1.5, bgcolor: '#fee2e2', borderRadius: 2 }}>
                      <ErrorIcon sx={{ color: '#dc2626', fontSize: 24 }} />
                    </Box>
                  </Box>
                  <Typography variant="body2" sx={{ color: '#64748b', mb: 1 }}>
                    Overdue
                  </Typography>
                  <Typography variant="h4" sx={{ color: '#dc2626', fontWeight: 700 }}>
                    ${overdueAmount.toLocaleString()}
                  </Typography>
                </CardContent>
              </Card>
            </Grid>
          </Grid>

          {/* Invoices Table */}
          <Card sx={{ borderRadius: 3, border: '1px solid #e2e8f0', boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)' }}>
            <CardContent sx={{ p: 3 }}>
              <Box sx={{ borderBottom: 1, borderColor: '#e2e8f0', mb: 3 }}>
                <Tabs value={selectedTab} onChange={this.handleTabChange}>
                  <Tab label="All Invoices" />
                  <Tab label="Paid" />
                  <Tab label="Pending" />
                  <Tab label="Overdue" />
                </Tabs>
              </Box>

              <TableContainer>
                <Table>
                  <TableHead>
                    <TableRow sx={{ bgcolor: '#f8fafc' }}>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Invoice Number
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Learner
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Course
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Issue Date
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Due Date
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Amount
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Status
                      </TableCell>
                      <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                        Actions
                      </TableCell>
                    </TableRow>
                  </TableHead>
                  <TableBody>
                    {filteredInvoices.map((invoice) => {
                      const statusColor = this.getStatusColor(invoice.status);
                      return (
                        <TableRow key={invoice.id} sx={{ '&:hover': { bgcolor: '#f8fafc' } }}>
                          <TableCell sx={{ border: 'none', color: '#25476a', fontWeight: 600 }}>
                            {invoice.invoiceNumber}
                          </TableCell>
                          <TableCell sx={{ border: 'none', color: '#64748b' }}>
                            {invoice.learnerName}
                          </TableCell>
                          <TableCell sx={{ border: 'none', color: '#64748b' }}>
                            {invoice.courseName}
                          </TableCell>
                          <TableCell sx={{ border: 'none', color: '#64748b' }}>
                            {new Date(invoice.issueDate).toLocaleDateString()}
                          </TableCell>
                          <TableCell sx={{ border: 'none', color: '#64748b' }}>
                            {new Date(invoice.dueDate).toLocaleDateString()}
                          </TableCell>
                          <TableCell sx={{ border: 'none', color: '#25476a', fontWeight: 600 }}>
                            ${invoice.amount.toLocaleString()}
                          </TableCell>
                          <TableCell sx={{ border: 'none' }}>
                            <Chip
                              icon={statusColor.icon}
                              label={invoice.status.replace('-', ' ').toUpperCase()}
                              size="small"
                              sx={{
                                bgcolor: statusColor.bg,
                                color: statusColor.color,
                                fontWeight: 600,
                                fontSize: '0.75rem',
                              }}
                            />
                          </TableCell>
                          <TableCell sx={{ border: 'none' }}>
                            <Box sx={{ display: 'flex', gap: 1 }}>
                              <IconButton
                                size="small"
                                onClick={() => this.handleViewInvoice(invoice)}
                                sx={{ color: '#38aae1' }}
                              >
                                <Visibility fontSize="small" />
                              </IconButton>
                              <IconButton size="small" sx={{ color: '#64748b' }}>
                                <Download fontSize="small" />
                              </IconButton>
                            </Box>
                          </TableCell>
                        </TableRow>
                      );
                    })}
                  </TableBody>
                </Table>
              </TableContainer>
            </CardContent>
          </Card>
        </Box>

        {/* Invoice Detail Dialog */}
        <Dialog
          open={invoiceDialogOpen}
          onClose={this.handleCloseInvoiceDialog}
          maxWidth="md"
          fullWidth
        >
          {selectedInvoice && (
            <>
              <DialogTitle sx={{ bgcolor: '#f8fafc', borderBottom: '1px solid #e2e8f0' }}>
                <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                  <Box>
                    <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 700 }}>
                      {selectedInvoice.invoiceNumber}
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#64748b' }}>
                      Issued: {new Date(selectedInvoice.issueDate).toLocaleDateString()}
                    </Typography>
                  </Box>
                  <Chip
                    icon={this.getStatusColor(selectedInvoice.status).icon}
                    label={selectedInvoice.status.replace('-', ' ').toUpperCase()}
                    sx={{
                      bgcolor: this.getStatusColor(selectedInvoice.status).bg,
                      color: this.getStatusColor(selectedInvoice.status).color,
                      fontWeight: 600,
                    }}
                  />
                </Box>
              </DialogTitle>
              <DialogContent sx={{ p: 4 }}>
                <Grid container spacing={3} sx={{ mb: 3 }}>
                  <Grid item xs={6}>
                    <Typography variant="body2" sx={{ color: '#64748b', mb: 0.5 }}>
                      Learner Name
                    </Typography>
                    <Typography variant="body1" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {selectedInvoice.learnerName}
                    </Typography>
                  </Grid>
                  <Grid item xs={6}>
                    <Typography variant="body2" sx={{ color: '#64748b', mb: 0.5 }}>
                      Course
                    </Typography>
                    <Typography variant="body1" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {selectedInvoice.courseName}
                    </Typography>
                  </Grid>
                  <Grid item xs={6}>
                    <Typography variant="body2" sx={{ color: '#64748b', mb: 0.5 }}>
                      Due Date
                    </Typography>
                    <Typography variant="body1" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {new Date(selectedInvoice.dueDate).toLocaleDateString()}
                    </Typography>
                  </Grid>
                </Grid>

                <Divider sx={{ my: 3 }} />

                <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 700, mb: 2 }}>
                  Invoice Items
                </Typography>
                <TableContainer>
                  <Table>
                    <TableHead>
                      <TableRow sx={{ bgcolor: '#f8fafc' }}>
                        <TableCell>Description</TableCell>
                        <TableCell align="center">Quantity</TableCell>
                        <TableCell align="right">Unit Price</TableCell>
                        <TableCell align="right">Total</TableCell>
                      </TableRow>
                    </TableHead>
                    <TableBody>
                      {selectedInvoice.items.map((item, index) => (
                        <TableRow key={index}>
                          <TableCell>{item.description}</TableCell>
                          <TableCell align="center">{item.quantity}</TableCell>
                          <TableCell align="right">${item.unitPrice.toLocaleString()}</TableCell>
                          <TableCell align="right">${item.total.toLocaleString()}</TableCell>
                        </TableRow>
                      ))}
                      <TableRow>
                        <TableCell colSpan={3} align="right" sx={{ fontWeight: 700, bgcolor: '#f8fafc' }}>
                          Total Amount:
                        </TableCell>
                        <TableCell align="right" sx={{ fontWeight: 700, bgcolor: '#f8fafc' }}>
                          ${selectedInvoice.amount.toLocaleString()}
                        </TableCell>
                      </TableRow>
                      {selectedInvoice.paidAmount !== undefined && selectedInvoice.paidAmount > 0 && (
                        <>
                          <TableRow>
                            <TableCell colSpan={3} align="right" sx={{ color: '#16a34a', fontWeight: 600 }}>
                              Paid Amount:
                            </TableCell>
                            <TableCell align="right" sx={{ color: '#16a34a', fontWeight: 600 }}>
                              ${selectedInvoice.paidAmount.toLocaleString()}
                            </TableCell>
                          </TableRow>
                          {selectedInvoice.amount - selectedInvoice.paidAmount > 0 && (
                            <TableRow>
                              <TableCell colSpan={3} align="right" sx={{ color: '#dc2626', fontWeight: 600 }}>
                                Balance Due:
                              </TableCell>
                              <TableCell align="right" sx={{ color: '#dc2626', fontWeight: 600 }}>
                                ${(selectedInvoice.amount - selectedInvoice.paidAmount).toLocaleString()}
                              </TableCell>
                            </TableRow>
                          )}
                        </>
                      )}
                    </TableBody>
                  </Table>
                </TableContainer>
              </DialogContent>
              <DialogActions sx={{ p: 3, bgcolor: '#f8fafc' }}>
                <Button onClick={this.handleCloseInvoiceDialog} sx={{ color: '#64748b' }}>
                  Close
                </Button>
                <Button
                  variant="outlined"
                  startIcon={<Download />}
                  sx={{ borderColor: '#38aae1', color: '#38aae1' }}
                >
                  Download PDF
                </Button>
                {selectedInvoice.status !== 'paid' && (
                  <Button
                    variant="contained"
                    startIcon={<Payment />}
                    sx={{ bgcolor: '#38aae1', color: '#ffffff' }}
                  >
                    Pay Now
                  </Button>
                )}
              </DialogActions>
            </>
          )}
        </Dialog>
      </Box>
    );
  }
}
