import React, { useState } from 'react';
import {
  Box,
  Typography,
  Grid,
  Card,
  CardContent,
  Button,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  TextField,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Chip,
  Avatar,
  List,
  ListItem,
  ListItemAvatar,
  ListItemText,
  IconButton,
  Divider,
  Paper,
  Tab,
  Tabs,
} from '@mui/material';
import {
  Add,
  Edit,
  Delete,
  VideoCall,
  Schedule,
  Person,
  CalendarToday,
  AccessTime,
} from '@mui/icons-material';

const mockLessons = [
  {
    id: 1,
    title: 'Introduction to Algebra',
    instructor: 'Dr. Sarah Johnson',
    date: '2024-10-15',
    time: '10:00 AM',
    duration: 60,
    students: 25,
    status: 'Scheduled',
    type: 'Live',
    course: 'Grade 3 Mathematics',
  },
  {
    id: 2,
    title: 'Chemical Reactions Basics',
    instructor: 'Prof. Michael Chen',
    date: '2024-10-16',
    time: '2:00 PM',
    duration: 45,
    students: 18,
    status: 'Scheduled',
    type: 'Live',
    course: 'Grade 4 Science',
  },
  {
    id: 3,
    title: 'Reading Comprehension Strategies',
    instructor: 'Ms. Emily Davis',
    date: '2024-10-14',
    time: '11:00 AM',
    duration: 50,
    students: 22,
    status: 'Completed',
    type: 'Recorded',
    course: 'Grade 5 English',
  },
];

export function LessonScheduler() {
  const [lessons, setLessons] = useState(mockLessons);
  const [openCreateDialog, setOpenCreateDialog] = useState(false);
  const [selectedTab, setSelectedTab] = useState(0);
  const [newLesson, setNewLesson] = useState({
    title: '',
    instructor: '',
    date: '',
    time: '',
    duration: 60,
    course: '',
    type: 'Live',
  });

  const handleCreateLesson = () => {
    const lesson = {
      id: lessons.length + 1,
      ...newLesson,
      students: 0,
      status: 'Scheduled',
    };
    setLessons([...lessons, lesson]);
    setOpenCreateDialog(false);
    setNewLesson({
      title: '',
      instructor: '',
      date: '',
      time: '',
      duration: 60,
      course: '',
      type: 'Live',
    });
  };

  const getStatusColor = (status) => {
    switch (status) {
      case 'Scheduled':
        return 'primary';
      case 'Completed':
        return 'success';
      case 'Cancelled':
        return 'error';
      default:
        return 'default';
    }
  };

  const getTypeColor = (type) => {
    return type === 'Live' ? 'error' : 'warning';
  };

  const upcomingLessons = lessons.filter(lesson => lesson.status === 'Scheduled');
  const completedLessons = lessons.filter(lesson => lesson.status === 'Completed');

  const TabPanel = ({ children, value, index }) => (
    <div hidden={value !== index}>
      {value === index && <Box sx={{ py: 3 }}>{children}</Box>}
    </div>
  );

  return (
    <Box sx={{ p: 3 }}>
      {/* Header */}
      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
        <Box>
          <Typography variant="h4" sx={{ fontWeight: 600, mb: 1 }}>
            Lesson Scheduler
          </Typography>
          <Typography variant="body2" color="text.secondary">
            Create and manage live and recorded lessons for your courses.
          </Typography>
        </Box>
        <Button
          variant="contained"
          startIcon={<Add />}
          onClick={() => setOpenCreateDialog(true)}
          sx={{ bgcolor: '#38aae1', '&:hover': { bgcolor: '#2e8bc4' } }}
        >
          Schedule Lesson
        </Button>
      </Box>

      {/* Quick Stats */}
      <Grid container spacing={3} sx={{ mb: 4 }}>
        <Grid item xs={12} sm={6} md={3}>
          <Card sx={{ bgcolor: '#e3f2fd', border: '1px solid #38aae1' }}>
            <CardContent>
              <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
                <Box>
                  <Typography variant="h6" color="primary">
                    Total Lessons
                  </Typography>
                  <Typography variant="h3" sx={{ fontWeight: 'bold', color: '#38aae1' }}>
                    {lessons.length}
                  </Typography>
                </Box>
                <Schedule sx={{ fontSize: 40, color: '#38aae1', opacity: 0.7 }} />
              </Box>
            </CardContent>
          </Card>
        </Grid>

        <Grid item xs={12} sm={6} md={3}>
          <Card sx={{ bgcolor: '#e8f5e8', border: '1px solid #4caf50' }}>
            <CardContent>
              <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
                <Box>
                  <Typography variant="h6" color="success.main">
                    Upcoming
                  </Typography>
                  <Typography variant="h3" sx={{ fontWeight: 'bold', color: '#4caf50' }}>
                    {upcomingLessons.length}
                  </Typography>
                </Box>
                <CalendarToday sx={{ fontSize: 40, color: '#4caf50', opacity: 0.7 }} />
              </Box>
            </CardContent>
          </Card>
        </Grid>

        <Grid item xs={12} sm={6} md={3}>
          <Card sx={{ bgcolor: '#fff3e0', border: '1px solid #feb139' }}>
            <CardContent>
              <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
                <Box>
                  <Typography variant="h6" sx={{ color: '#feb139' }}>
                    Live Sessions
                  </Typography>
                  <Typography variant="h3" sx={{ fontWeight: 'bold', color: '#feb139' }}>
                    {lessons.filter(l => l.type === 'Live').length}
                  </Typography>
                </Box>
                <VideoCall sx={{ fontSize: 40, color: '#feb139', opacity: 0.7 }} />
              </Box>
            </CardContent>
          </Card>
        </Grid>

        <Grid item xs={12} sm={6} md={3}>
          <Card sx={{ bgcolor: '#f3e5f5', border: '1px solid #9c27b0' }}>
            <CardContent>
              <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
                <Box>
                  <Typography variant="h6" sx={{ color: '#9c27b0' }}>
                    Total Students
                  </Typography>
                  <Typography variant="h3" sx={{ fontWeight: 'bold', color: '#9c27b0' }}>
                    {lessons.reduce((sum, lesson) => sum + lesson.students, 0)}
                  </Typography>
                </Box>
                <Person sx={{ fontSize: 40, color: '#9c27b0', opacity: 0.7 }} />
              </Box>
            </CardContent>
          </Card>
        </Grid>
      </Grid>

      {/* Tabs */}
      <Paper sx={{ mb: 3 }}>
        <Tabs
          value={selectedTab}
          onChange={(e, newValue) => setSelectedTab(newValue)}
          sx={{ borderBottom: 1, borderColor: 'divider' }}
        >
          <Tab label="All Lessons" />
          <Tab label="Upcoming" />
          <Tab label="Completed" />
        </Tabs>

        <TabPanel value={selectedTab} index={0}>
          <List>
            {lessons.map((lesson, index) => (
              <React.Fragment key={lesson.id}>
                <ListItem
                  secondaryAction={
                    <Box>
                      <IconButton edge="end">
                        <Edit />
                      </IconButton>
                      <IconButton edge="end">
                        <Delete />
                      </IconButton>
                    </Box>
                  }
                >
                  <ListItemAvatar>
                    <Avatar sx={{ bgcolor: '#38aae1' }}>
                      {lesson.type === 'Live' ? <VideoCall /> : <Schedule />}
                    </Avatar>
                  </ListItemAvatar>
                  <ListItemText
                    primary={
                      <Box sx={{ display: 'flex', alignItems: 'center', gap: 1, mb: 1 }}>
                        <Typography variant="h6" sx={{ fontWeight: 600 }}>
                          {lesson.title}
                        </Typography>
                        <Chip
                          size="small"
                          label={lesson.status}
                          color={getStatusColor(lesson.status)}
                        />
                        <Chip
                          size="small"
                          label={lesson.type}
                          color={getTypeColor(lesson.type)}
                          variant="outlined"
                        />
                      </Box>
                    }
                    secondary={
                      <Box>
                        <Typography variant="body2" color="text.secondary">
                          Instructor: {lesson.instructor} • Course: {lesson.course}
                        </Typography>
                        <Typography variant="body2" color="text.secondary">
                          📅 {lesson.date} at {lesson.time} • ⏱️ {lesson.duration} min • 👥 {lesson.students} students
                        </Typography>
                      </Box>
                    }
                  />
                </ListItem>
                {index < lessons.length - 1 && <Divider />}
              </React.Fragment>
            ))}
          </List>
        </TabPanel>

        <TabPanel value={selectedTab} index={1}>
          <List>
            {upcomingLessons.map((lesson, index) => (
              <React.Fragment key={lesson.id}>
                <ListItem>
                  <ListItemAvatar>
                    <Avatar sx={{ bgcolor: '#4caf50' }}>
                      {lesson.type === 'Live' ? <VideoCall /> : <Schedule />}
                    </Avatar>
                  </ListItemAvatar>
                  <ListItemText
                    primary={
                      <Box sx={{ display: 'flex', alignItems: 'center', gap: 1, mb: 1 }}>
                        <Typography variant="h6" sx={{ fontWeight: 600 }}>
                          {lesson.title}
                        </Typography>
                        <Chip
                          size="small"
                          label={lesson.type}
                          color={getTypeColor(lesson.type)}
                          variant="outlined"
                        />
                      </Box>
                    }
                    secondary={
                      <Box>
                        <Typography variant="body2" color="text.secondary">
                          Instructor: {lesson.instructor} • Course: {lesson.course}
                        </Typography>
                        <Typography variant="body2" color="text.secondary">
                          📅 {lesson.date} at {lesson.time} • ⏱️ {lesson.duration} min
                        </Typography>
                      </Box>
                    }
                  />
                </ListItem>
                {index < upcomingLessons.length - 1 && <Divider />}
              </React.Fragment>
            ))}
          </List>
        </TabPanel>

        <TabPanel value={selectedTab} index={2}>
          <List>
            {completedLessons.map((lesson, index) => (
              <React.Fragment key={lesson.id}>
                <ListItem>
                  <ListItemAvatar>
                    <Avatar sx={{ bgcolor: '#9e9e9e' }}>
                      {lesson.type === 'Live' ? <VideoCall /> : <Schedule />}
                    </Avatar>
                  </ListItemAvatar>
                  <ListItemText
                    primary={
                      <Box sx={{ display: 'flex', alignItems: 'center', gap: 1, mb: 1 }}>
                        <Typography variant="h6" sx={{ fontWeight: 600 }}>
                          {lesson.title}
                        </Typography>
                        <Chip
                          size="small"
                          label="Completed"
                          color="success"
                        />
                      </Box>
                    }
                    secondary={
                      <Box>
                        <Typography variant="body2" color="text.secondary">
                          Instructor: {lesson.instructor} • Course: {lesson.course}
                        </Typography>
                        <Typography variant="body2" color="text.secondary">
                          📅 {lesson.date} • 👥 {lesson.students} students attended
                        </Typography>
                      </Box>
                    }
                  />
                </ListItem>
                {index < completedLessons.length - 1 && <Divider />}
              </React.Fragment>
            ))}
          </List>
        </TabPanel>
      </Paper>

      {/* Create Lesson Dialog */}
      <Dialog open={openCreateDialog} onClose={() => setOpenCreateDialog(false)} maxWidth="md" fullWidth>
        <DialogTitle>Schedule New Lesson</DialogTitle>
        <DialogContent>
          <Grid container spacing={3} sx={{ mt: 1 }}>
            <Grid item xs={12} md={6}>
              <TextField
                label="Lesson Title"
                fullWidth
                value={newLesson.title}
                onChange={(e) => setNewLesson({ ...newLesson, title: e.target.value })}
              />
            </Grid>
            <Grid item xs={12} md={6}>
              <TextField
                label="Instructor Name"
                fullWidth
                value={newLesson.instructor}
                onChange={(e) => setNewLesson({ ...newLesson, instructor: e.target.value })}
              />
            </Grid>
            <Grid item xs={12} md={6}>
              <FormControl fullWidth>
                <InputLabel>Course</InputLabel>
                <Select
                  value={newLesson.course}
                  onChange={(e) => setNewLesson({ ...newLesson, course: e.target.value })}
                  label="Course"
                >
                  <MenuItem value="Grade 3 Mathematics">Grade 3 Mathematics</MenuItem>
                  <MenuItem value="Grade 4 Science">Grade 4 Science</MenuItem>
                  <MenuItem value="Grade 5 English">Grade 5 English</MenuItem>
                </Select>
              </FormControl>
            </Grid>
            <Grid item xs={12} md={6}>
              <FormControl fullWidth>
                <InputLabel>Lesson Type</InputLabel>
                <Select
                  value={newLesson.type}
                  onChange={(e) => setNewLesson({ ...newLesson, type: e.target.value })}
                  label="Lesson Type"
                >
                  <MenuItem value="Live">Live Session</MenuItem>
                  <MenuItem value="Recorded">Recorded Session</MenuItem>
                </Select>
              </FormControl>
            </Grid>
            <Grid item xs={12} md={6}>
              <TextField
                label="Date"
                type="date"
                fullWidth
                InputLabelProps={{ shrink: true }}
                value={newLesson.date}
                onChange={(e) => setNewLesson({ ...newLesson, date: e.target.value })}
              />
            </Grid>
            <Grid item xs={12} md={6}>
              <TextField
                label="Time"
                type="time"
                fullWidth
                InputLabelProps={{ shrink: true }}
                value={newLesson.time}
                onChange={(e) => setNewLesson({ ...newLesson, time: e.target.value })}
              />
            </Grid>
            <Grid item xs={12}>
              <TextField
                label="Duration (minutes)"
                type="number"
                fullWidth
                value={newLesson.duration}
                onChange={(e) => setNewLesson({ ...newLesson, duration: parseInt(e.target.value) })}
              />
            </Grid>
          </Grid>
        </DialogContent>
        <DialogActions>
          <Button onClick={() => setOpenCreateDialog(false)}>Cancel</Button>
          <Button variant="contained" onClick={handleCreateLesson}>
            Schedule Lesson
          </Button>
        </DialogActions>
      </Dialog>
    </Box>
  );
}