import React, { Component } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  Box,
  Typography,
  TextField,
  Grid,
  List,
  ListItem,
  ListItemButton,
  ListItemText,
  IconButton,
  Tabs,
  Tab,
  Card,
  CardContent,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Divider,
  Chip,
  Badge,
  RadioGroup,
  FormControlLabel,
  Radio,
} from '@mui/material';
import {
  Close,
  Delete,
  Add,
  Save,
  Upload,
  AttachFile,
  CloudUpload,
  Edit,
  CheckCircle,
  PlayCircleOutline,
  Assignment,
  SportsEsports,
} from '@mui/icons-material';
import { ActivityVideoUploader } from './ActivityVideoUploader.jsx';

const createDefaultSession = (sessionNumber) => ({
  id: `session-${sessionNumber}`,
  title: `Session ${sessionNumber}`,
  duration: 60,
  description: `Content for session ${sessionNumber}`,
  objectives: [],
  introVideo: { title: '', duration: 0, url: '' },
  assignmentVideo: { title: '', duration: 0, url: '' },
  assignmentChecklist: [],
  activityVideos: [],
  resources: [],
  isCompleted: false
});

const convertLessonToSession = (lesson, sessionNumber) => ({
  id: `session-${sessionNumber}`,
  title: lesson.title || `Session ${sessionNumber}`,
  duration: lesson.duration || 60,
  description: lesson.description || `Content for session ${sessionNumber}`,
  objectives: lesson.objectives || [],
  introVideo: lesson.introVideo || { title: '', duration: 0, url: '' },
  assignmentVideo: lesson.assignmentVideo || { title: '', duration: 0, url: '' },
  assignmentChecklist: lesson.assignmentChecklist || [],
  activityVideos: lesson.activityVideos || [],
  resources: lesson.resources || [],
  isCompleted: false
});

export class LessonEditor extends Component {
  constructor(props) {
    super(props);
    
    // Initialize 12 sessions - convert the incoming lesson to session 1, create defaults for others
    const sessions = Array.from({ length: 12 }, (_, index) => {
      if (index === 0) {
        return convertLessonToSession(props.lesson, index + 1);
      }
      return createDefaultSession(index + 1);
    });

    this.state = {
      sessions,
      currentSessionId: 'session-1',
      activeTab: 0,
      newObjective: '',
      newChecklistItem: '',
      newResource: {
        name: '',
        type: 'pdf',
        url: ''
      },
      resourceInputMethod: 'url'
    };
  }

  componentDidUpdate(prevProps) {
    if (prevProps.lesson !== this.props.lesson) {
      const sessions = Array.from({ length: 12 }, (_, index) => {
        if (index === 0) {
          return convertLessonToSession(this.props.lesson, index + 1);
        }
        return createDefaultSession(index + 1);
      });
      
      this.setState({ 
        sessions,
        currentSessionId: 'session-1'
      });
    }
  }

  getCurrentSession = () => {
    return this.state.sessions.find(s => s.id === this.state.currentSessionId) || this.state.sessions[0];
  };

  updateCurrentSession = (updates) => {
    this.setState({
      sessions: this.state.sessions.map(session =>
        session.id === this.state.currentSessionId 
          ? { ...session, ...updates }
          : session
      )
    });
  };

  handleSave = () => {
    const mainSession = this.state.sessions[0]; // Use session 1 as the main lesson data
    const updatedLesson = {
      id: this.props.lesson.id,
      title: mainSession.title,
      duration: mainSession.duration,
      description: mainSession.description,
      objectives: mainSession.objectives,
      introVideo: mainSession.introVideo,
      assignmentVideo: mainSession.assignmentVideo,
      activityVideos: mainSession.activityVideos,
      resources: mainSession.resources
    };
    this.props.onSave(updatedLesson);
  };

  handleCancel = () => {
    this.props.onCancel();
  };

  switchToSession = (sessionId) => {
    this.setState({ currentSessionId: sessionId });
  };

  addObjective = () => {
    const { newObjective } = this.state;
    const currentSession = this.getCurrentSession();
    
    if (newObjective.trim()) {
      this.updateCurrentSession({
        objectives: [...currentSession.objectives, newObjective]
      });
      this.setState({ newObjective: '' });
    }
  };

  removeObjective = (index) => {
    const currentSession = this.getCurrentSession();
    this.updateCurrentSession({
      objectives: currentSession.objectives.filter((_, i) => i !== index)
    });
  };

  addChecklistItem = () => {
    const { newChecklistItem } = this.state;
    const currentSession = this.getCurrentSession();
    
    if (newChecklistItem.trim()) {
      this.updateCurrentSession({
        assignmentChecklist: [...currentSession.assignmentChecklist, newChecklistItem]
      });
      this.setState({ newChecklistItem: '' });
    }
  };

  removeChecklistItem = (index) => {
    const currentSession = this.getCurrentSession();
    this.updateCurrentSession({
      assignmentChecklist: currentSession.assignmentChecklist.filter((_, i) => i !== index)
    });
  };

  addActivityVideo = () => {
    const currentSession = this.getCurrentSession();
    this.updateCurrentSession({
      activityVideos: [...currentSession.activityVideos, { title: '', duration: 0, url: '' }]
    });
  };

  removeActivityVideo = (index) => {
    const currentSession = this.getCurrentSession();
    this.updateCurrentSession({
      activityVideos: currentSession.activityVideos.filter((_, i) => i !== index)
    });
  };

  updateActivityVideo = (index, field, value) => {
    const currentSession = this.getCurrentSession();
    const newActivities = [...currentSession.activityVideos];
    newActivities[index] = { ...newActivities[index], [field]: value };
    this.updateCurrentSession({
      activityVideos: newActivities
    });
  };

  handleResourceFileUpload = (event) => {
    const file = event.target.files?.[0];
    if (file) {
      // In a real application, you would upload the file to a server
      // For now, we'll create a local URL
      const fileUrl = URL.createObjectURL(file);
      this.setState({
        newResource: {
          ...this.state.newResource,
          url: fileUrl,
          name: this.state.newResource.name || file.name
        }
      });
    }
  };

  addResource = () => {
    const { newResource } = this.state;
    const currentSession = this.getCurrentSession();
    
    if (newResource.name.trim()) {
      this.updateCurrentSession({
        resources: [...currentSession.resources, { ...newResource }]
      });
      this.setState({
        newResource: {
          name: '',
          type: 'pdf',
          url: ''
        },
        resourceInputMethod: 'url'
      });
    }
  };

  removeResource = (index) => {
    const currentSession = this.getCurrentSession();
    this.updateCurrentSession({
      resources: currentSession.resources.filter((_, i) => i !== index)
    });
  };

  toggleSessionComplete = (sessionId) => {
    this.setState({
      sessions: this.state.sessions.map(session =>
        session.id === sessionId 
          ? { ...session, isCompleted: !session.isCompleted }
          : session
      )
    });
  };

  getSessionProgress = () => {
    const completedSessions = this.state.sessions.filter(s => s.isCompleted).length;
    return Math.round((completedSessions / 12) * 100);
  };

  render() {
    const { sessions, currentSessionId, activeTab, newObjective, newChecklistItem, newResource } = this.state;
    const currentSession = this.getCurrentSession();
    const currentSessionNumber = parseInt(currentSessionId.split('-')[1]);

    return (
      <Box sx={{ p: 3, maxHeight: '100vh', overflow: 'auto' }}>
        {/* Header */}
        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 4 }}>
          <Box>
            <Typography variant="h4" sx={{ color: '#25476a', fontWeight: 700, mb: 1 }}>
              Edit Lesson Plan - All Sessions
            </Typography>
            <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
              <Typography variant="body1" sx={{ color: '#64748b' }}>
                Currently editing: {currentSession.title}
              </Typography>
              <Chip 
                label={`${this.getSessionProgress()}% Complete`} 
                size="small" 
                sx={{ 
                  bgcolor: 'rgba(56, 170, 225, 0.1)', 
                  color: '#38aae1',
                  fontWeight: 600 
                }} 
              />
            </Box>
          </Box>
          <Box sx={{ display: 'flex', gap: 2 }}>
            <Button
              variant="outlined"
              onClick={this.handleCancel}
              sx={{ 
                color: '#64748b',
                borderColor: '#64748b',
                textTransform: 'none',
                '&:hover': { 
                  bgcolor: 'rgba(100, 116, 139, 0.1)',
                  borderColor: '#64748b' 
                }
              }}
            >
              Cancel
            </Button>
            <Button
              variant="contained"
              startIcon={<Save />}
              onClick={this.handleSave}
              sx={{
                backgroundColor: '#38aae1',
                textTransform: 'none',
                fontWeight: 600,
                '&:hover': {
                  backgroundColor: '#2563eb',
                },
              }}
            >
              Save All Sessions
            </Button>
          </Box>
        </Box>

        <Grid container spacing={3}>
          {/* Sessions Sidebar */}
          <Grid item xs={12} md={3}>
            <Card sx={{ height: 'fit-content' }}>
              <CardContent>
                <Typography variant="h6" sx={{ mb: 2, color: '#25476a', fontWeight: 600 }}>
                  Sessions ({sessions.filter(s => s.isCompleted).length}/12)
                </Typography>
                <List dense sx={{ p: 0 }}>
                  {sessions.map((session, index) => (
                    <ListItem key={session.id} disablePadding sx={{ mb: 0.5 }}>
                      <ListItemButton
                        selected={currentSessionId === session.id}
                        onClick={() => this.switchToSession(session.id)}
                        sx={{
                          borderRadius: 2,
                          backgroundColor: currentSessionId === session.id ? '#38aae1' : 'transparent',
                          color: currentSessionId === session.id ? 'white' : '#64748b',
                          pr: 1,
                          '&:hover': {
                            backgroundColor: currentSessionId === session.id ? '#2e8bc4' : 'rgba(56, 170, 225, 0.1)',
                          },
                        }}
                      >
                        <ListItemText 
                          primary={
                            <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
                              <Typography sx={{
                                fontWeight: currentSessionId === session.id ? 600 : 500,
                                fontSize: '0.875rem',
                              }}>
                                Session {index + 1}
                              </Typography>
                              {session.isCompleted && (
                                <CheckCircle sx={{ fontSize: 16, color: currentSessionId === session.id ? 'white' : '#4caf50' }} />
                              )}
                            </Box>
                          }
                          secondary={
                            <Typography sx={{ 
                              fontSize: '0.75rem', 
                              color: currentSessionId === session.id ? 'rgba(255,255,255,0.8)' : '#94a3b8',
                              mt: 0.5
                            }}>
                              {session.title.length > 20 ? `${session.title.substring(0, 20)}...` : session.title}
                            </Typography>
                          }
                        />
                        <IconButton
                          size="small"
                          onClick={(e) => {
                            e.stopPropagation();
                            this.toggleSessionComplete(session.id);
                          }}
                          sx={{ 
                            color: currentSessionId === session.id ? 'white' : '#64748b',
                            ml: 1
                          }}
                        >
                          <Edit fontSize="small" />
                        </IconButton>
                      </ListItemButton>
                    </ListItem>
                  ))}
                </List>
                
                <Divider sx={{ my: 2 }} />
                
                <Button
                  fullWidth
                  size="small"
                  onClick={() => this.toggleSessionComplete(currentSessionId)}
                  variant={currentSession.isCompleted ? "outlined" : "contained"}
                  startIcon={currentSession.isCompleted ? <Close /> : <CheckCircle />}
                  sx={{
                    textTransform: 'none',
                    bgcolor: currentSession.isCompleted ? 'transparent' : '#4caf50',
                    color: currentSession.isCompleted ? '#4caf50' : 'white',
                    borderColor: '#4caf50',
                    '&:hover': {
                      bgcolor: currentSession.isCompleted ? 'rgba(76, 175, 80, 0.1)' : '#45a049',
                    }
                  }}
                >
                  {currentSession.isCompleted ? 'Mark Incomplete' : 'Mark Complete'}
                </Button>
              </CardContent>
            </Card>
          </Grid>

          {/* Main Content */}
          <Grid item xs={12} md={9}>
            <Card>
              {/* Tabs */}
              <Box sx={{ borderBottom: '1px solid #e2e8f0' }}>
                <Tabs 
                  value={activeTab} 
                  onChange={(e, newValue) => this.setState({ activeTab: newValue })}
                  sx={{ 
                    px: 3,
                    '& .MuiTab-root': {
                      textTransform: 'none',
                      fontWeight: 600,
                      fontSize: '1rem',
                      color: '#64748b',
                      '&.Mui-selected': {
                        color: '#38aae1',
                      }
                    }
                  }}
                >
                  <Tab label="Overview" />
                  <Tab label="Videos" />
                  <Tab label="Resources" />
                </Tabs>
              </Box>

              {/* Tab Content */}
              <CardContent sx={{ p: 4 }}>
                {/* Overview Tab */}
                {activeTab === 0 && (
                  <Box>
                    {/* Session Header */}
                    <Box sx={{ mb: 4, p: 3, bgcolor: '#f8fafc', borderRadius: 2, border: '1px solid #e2e8f0' }}>
                      <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 2 }}>
                        <Typography variant="h5" sx={{ fontWeight: 600, color: '#25476a' }}>
                          Session {currentSessionNumber} Configuration
                        </Typography>
                        <Chip 
                          icon={currentSession.isCompleted ? <CheckCircle /> : <Edit />}
                          label={currentSession.isCompleted ? 'Complete' : 'In Progress'} 
                          color={currentSession.isCompleted ? 'success' : 'default'}
                          size="small"
                        />
                      </Box>
                      <Typography variant="body2" sx={{ color: '#64748b' }}>
                        Configure the content and structure for this individual session
                      </Typography>
                    </Box>

                    {/* Session Details */}
                    <Box sx={{ mb: 4 }}>
                      <Typography variant="h6" sx={{ mb: 3, fontWeight: 600, color: '#25476a' }}>
                        Session Details
                      </Typography>
                      
                      <Grid container spacing={3}>
                        <Grid item xs={12} md={8}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                            Session Title
                          </Typography>
                          <TextField
                            fullWidth
                            value={currentSession.title}
                            onChange={(e) => this.updateCurrentSession({ title: e.target.value })}
                            placeholder="Enter session title"
                            size="small"
                          />
                        </Grid>
                        <Grid item xs={12} md={4}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                            Duration (minutes)
                          </Typography>
                          <TextField
                            fullWidth
                            type="number"
                            value={currentSession.duration}
                            onChange={(e) => this.updateCurrentSession({ duration: parseInt(e.target.value) || 0 })}
                            size="small"
                          />
                        </Grid>
                        <Grid item xs={12}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                            Session Description
                          </Typography>
                          <TextField
                            fullWidth
                            multiline
                            rows={3}
                            value={currentSession.description}
                            onChange={(e) => this.updateCurrentSession({ description: e.target.value })}
                            placeholder="Describe what students will learn in this session"
                            size="small"
                          />
                        </Grid>
                      </Grid>
                    </Box>

                    {/* Learning Objectives */}
                    <Box>
                      <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 2 }}>
                        <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                          Learning Objectives
                        </Typography>
                        <Button
                          startIcon={<Add />}
                          variant="outlined"
                          size="small"
                          onClick={this.addObjective}
                          disabled={!newObjective.trim()}
                          sx={{ 
                            borderColor: '#38aae1', 
                            color: '#38aae1',
                            textTransform: 'none'
                          }}
                        >
                          Add Objective
                        </Button>
                      </Box>
                      
                      <TextField
                        fullWidth
                        placeholder="Enter learning objective for this session"
                        value={newObjective}
                        onChange={(e) => this.setState({ newObjective: e.target.value })}
                        onKeyPress={(e) => {
                          if (e.key === 'Enter') {
                            this.addObjective();
                          }
                        }}
                        size="small"
                        sx={{ mb: 2 }}
                      />

                      {currentSession.objectives.map((objective, index) => (
                        <Box 
                          key={index}
                          sx={{ 
                            display: 'flex', 
                            alignItems: 'center', 
                            justifyContent: 'space-between',
                            p: 2,
                            mb: 1,
                            bgcolor: '#f8fafc',
                            borderRadius: 2,
                            border: '1px solid #e2e8f0'
                          }}
                        >
                          <Typography>{objective}</Typography>
                          <IconButton 
                            size="small" 
                            onClick={() => this.removeObjective(index)}
                            sx={{ color: '#ef4444' }}
                          >
                            <Delete fontSize="small" />
                          </IconButton>
                        </Box>
                      ))}
                    </Box>
                  </Box>
                )}

                {/* Videos Tab */}
                {activeTab === 1 && (
                  <Box>
                    {/* Session Video Header */}
                    <Box sx={{ mb: 4, p: 3, bgcolor: '#f8fafc', borderRadius: 2, border: '1px solid #e2e8f0' }}>
                      <Typography variant="h5" sx={{ fontWeight: 600, color: '#25476a', mb: 1 }}>
                        Session {currentSessionNumber} - Video Content
                      </Typography>
                      <Typography variant="body2" sx={{ color: '#64748b' }}>
                        Configure videos for this specific session
                      </Typography>
                    </Box>

                    {/* Introduction Video */}
                    <Box sx={{ mb: 4 }}>
                      <Box sx={{ display: 'flex', alignItems: 'center', mb: 3 }}>
                        <PlayCircleOutline sx={{ color: '#38aae1', mr: 1 }} />
                        <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                          Introduction Video
                        </Typography>
                      </Box>
                      
                      <Grid container spacing={3}>
                        <Grid item xs={12} md={8}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                            Video Title
                          </Typography>
                          <TextField
                            fullWidth
                            value={currentSession.introVideo.title}
                            onChange={(e) => this.updateCurrentSession({
                              introVideo: { ...currentSession.introVideo, title: e.target.value }
                            })}
                            placeholder="Introduction video title"
                            size="small"
                          />
                        </Grid>
                        <Grid item xs={12} md={4}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                            Duration (minutes)
                          </Typography>
                          <TextField
                            fullWidth
                            type="number"
                            value={currentSession.introVideo.duration}
                            onChange={(e) => this.updateCurrentSession({
                              introVideo: { ...currentSession.introVideo, duration: parseInt(e.target.value) || 0 }
                            })}
                            size="small"
                          />
                        </Grid>
                        <Grid item xs={12}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                            Video URL
                          </Typography>
                          <TextField
                            fullWidth
                            value={currentSession.introVideo.url}
                            onChange={(e) => this.updateCurrentSession({
                              introVideo: { ...currentSession.introVideo, url: e.target.value }
                            })}
                            placeholder="Video file path or URL"
                            size="small"
                          />
                        </Grid>
                      </Grid>
                    </Box>

                    {/* Assignment Video */}
                    <Box sx={{ mb: 4 }}>
                      <Box sx={{ display: 'flex', alignItems: 'center', mb: 3 }}>
                        <Assignment sx={{ color: '#ff9800', mr: 1 }} />
                        <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                          Assignment Video
                        </Typography>
                      </Box>
                      
                      <Grid container spacing={3}>
                        <Grid item xs={12} md={8}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                            Video Title
                          </Typography>
                          <TextField
                            fullWidth
                            value={currentSession.assignmentVideo.title}
                            onChange={(e) => this.updateCurrentSession({
                              assignmentVideo: { ...currentSession.assignmentVideo, title: e.target.value }
                            })}
                            placeholder="Assignment video title"
                            size="small"
                          />
                        </Grid>
                        <Grid item xs={12} md={4}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                            Duration (minutes)
                          </Typography>
                          <TextField
                            fullWidth
                            type="number"
                            value={currentSession.assignmentVideo.duration}
                            onChange={(e) => this.updateCurrentSession({
                              assignmentVideo: { ...currentSession.assignmentVideo, duration: parseInt(e.target.value) || 0 }
                            })}
                            size="small"
                          />
                        </Grid>
                        <Grid item xs={12}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                            Video URL
                          </Typography>
                          <TextField
                            fullWidth
                            value={currentSession.assignmentVideo.url}
                            onChange={(e) => this.updateCurrentSession({
                              assignmentVideo: { ...currentSession.assignmentVideo, url: e.target.value }
                            })}
                            placeholder="Video file path or URL"
                            size="small"
                          />
                        </Grid>
                      </Grid>
                    </Box>

                    {/* Assignment Checklist */}
                    <Box sx={{ mb: 4 }}>
                      <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 3 }}>
                        <Box sx={{ display: 'flex', alignItems: 'center' }}>
                          <CheckCircle sx={{ color: '#4caf50', mr: 1 }} />
                          <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                            Assignment Checklist
                          </Typography>
                        </Box>
                        <Button
                          startIcon={<Add />}
                          variant="outlined"
                          size="small"
                          onClick={this.addChecklistItem}
                          disabled={!this.state.newChecklistItem.trim()}
                          sx={{ 
                            borderColor: '#4caf50', 
                            color: '#4caf50',
                            textTransform: 'none',
                            '&:hover': {
                              borderColor: '#45a049',
                              backgroundColor: 'rgba(76, 175, 80, 0.05)',
                            }
                          }}
                        >
                          Add Item
                        </Button>
                      </Box>

                      <Box sx={{ mb: 2 }}>
                        <TextField
                          fullWidth
                          placeholder="Enter checklist item for the assignment"
                          value={this.state.newChecklistItem}
                          onChange={(e) => this.setState({ newChecklistItem: e.target.value })}
                          onKeyPress={(e) => {
                            if (e.key === 'Enter') {
                              this.addChecklistItem();
                            }
                          }}
                          size="small"
                          sx={{ mb: 2 }}
                        />
                      </Box>

                      {currentSession.assignmentChecklist.length === 0 ? (
                        <Box sx={{ 
                          p: 3, 
                          textAlign: 'center', 
                          bgcolor: '#f8fafc', 
                          borderRadius: 2,
                          border: '1px dashed #e2e8f0' 
                        }}>
                          <Typography variant="body2" sx={{ color: '#64748b' }}>
                            No checklist items added yet. Add items to create an assignment checklist.
                          </Typography>
                        </Box>
                      ) : (
                        <Box sx={{ 
                          border: '1px solid #e2e8f0', 
                          borderRadius: 2,
                          bgcolor: 'white'
                        }}>
                          {currentSession.assignmentChecklist.map((item, index) => (
                            <Box
                              key={index}
                              sx={{
                                display: 'flex',
                                alignItems: 'center',
                                justifyContent: 'space-between',
                                p: 2,
                                borderBottom: index < currentSession.assignmentChecklist.length - 1 ? '1px solid #e2e8f0' : 'none',
                                '&:hover': {
                                  bgcolor: '#f8fafc'
                                }
                              }}
                            >
                              <Box sx={{ display: 'flex', alignItems: 'center', flex: 1 }}>
                                <CheckCircle sx={{ color: '#4caf50', mr: 2, fontSize: 20 }} />
                                <Typography variant="body2" sx={{ color: '#25476a' }}>
                                  {item}
                                </Typography>
                              </Box>
                              <IconButton 
                                size="small" 
                                onClick={() => this.removeChecklistItem(index)}
                                sx={{ color: '#ef4444' }}
                              >
                                <Delete fontSize="small" />
                              </IconButton>
                            </Box>
                          ))}
                        </Box>
                      )}
                    </Box>

                    {/* Activity Videos */}
                    <Box>
                      <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 3 }}>
                        <Box sx={{ display: 'flex', alignItems: 'center' }}>
                          <SportsEsports sx={{ color: '#9c27b0', mr: 1 }} />
                          <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                            Activity Videos
                          </Typography>
                        </Box>
                        <Button
                          startIcon={<Add />}
                          variant="contained"
                          onClick={this.addActivityVideo}
                          sx={{ 
                            bgcolor: '#38aae1',
                            '&:hover': { bgcolor: '#2e8bc4' },
                            textTransform: 'none'
                          }}
                        >
                          Add Activity Video
                        </Button>
                      </Box>
                      
                      {currentSession.activityVideos.map((activity, index) => (
                        <Box key={index} sx={{ mb: 4, p: 3, border: '1px solid #e2e8f0', borderRadius: 2 }}>
                          <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 2 }}>
                            <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                              Activity Video {index + 1}
                            </Typography>
                            <IconButton 
                              size="small" 
                              onClick={() => this.removeActivityVideo(index)}
                              sx={{ color: '#ef4444' }}
                            >
                              <Delete />
                            </IconButton>
                          </Box>
                          
                          <Grid container spacing={3}>
                            <Grid item xs={12} md={8}>
                              <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                                Video Title
                              </Typography>
                              <TextField
                                fullWidth
                                value={activity.title}
                                onChange={(e) => this.updateActivityVideo(index, 'title', e.target.value)}
                                placeholder="Activity video title"
                                size="small"
                              />
                            </Grid>
                            <Grid item xs={12} md={4}>
                              <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                                Duration (minutes)
                              </Typography>
                              <TextField
                                fullWidth
                                type="number"
                                value={activity.duration}
                                onChange={(e) => this.updateActivityVideo(index, 'duration', parseInt(e.target.value) || 0)}
                                size="small"
                              />
                            </Grid>
                            <Grid item xs={12}>
                              <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                                Upload Video
                              </Typography>
                              <ActivityVideoUploader
                                activityIndex={index}
                                currentUrl={activity.url}
                                currentTitle={activity.title}
                                onUpload={(file) => {
                                  const videoUrl = URL.createObjectURL(file);
                                  this.updateActivityVideo(index, 'url', videoUrl);
                                  if (!activity.title) {
                                    this.updateActivityVideo(index, 'title', file.name);
                                  }
                                }}
                              />
                            </Grid>
                            <Grid item xs={12}>
                              <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                                Or Enter Video URL
                              </Typography>
                              <TextField
                                fullWidth
                                value={activity.url}
                                onChange={(e) => this.updateActivityVideo(index, 'url', e.target.value)}
                                placeholder="Video file path or URL"
                                size="small"
                              />
                            </Grid>
                          </Grid>
                        </Box>
                      ))}
                    </Box>
                  </Box>
                )}

                {/* Resources Tab */}
                {activeTab === 2 && (
                  <Box>
                    {/* Session Resources Header */}
                    <Box sx={{ mb: 4, p: 3, bgcolor: '#f8fafc', borderRadius: 2, border: '1px solid #e2e8f0' }}>
                      <Typography variant="h5" sx={{ fontWeight: 600, color: '#25476a', mb: 1 }}>
                        Session {currentSessionNumber} - Learning Resources
                      </Typography>
                      <Typography variant="body2" sx={{ color: '#64748b' }}>
                        Add resources specific to this session
                      </Typography>
                    </Box>

                    <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 3 }}>
                      <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                        Resources ({currentSession.resources.length})
                      </Typography>
                      <Button
                        startIcon={<Add />}
                        variant="outlined"
                        onClick={this.addResource}
                        disabled={!newResource.name.trim()}
                        sx={{ 
                          borderColor: '#38aae1', 
                          color: '#38aae1',
                          textTransform: 'none'
                        }}
                      >
                        Add Resource
                      </Button>
                    </Box>

                    {/* Add New Resource */}
                    <Box sx={{ mb: 4, p: 3, border: '1px solid #e2e8f0', borderRadius: 2, bgcolor: '#f8fafc' }}>
                      <Typography variant="h6" sx={{ mb: 2, fontWeight: 600, color: '#25476a' }}>
                        Add New Resource
                      </Typography>
                      <Grid container spacing={3}>
                        <Grid item xs={12}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                            Resource Input Method
                          </Typography>
                          <FormControl component="fieldset">
                            <RadioGroup
                              row
                              value={this.state.resourceInputMethod}
                              onChange={(e) => {
                                const method = e.target.value;
                                this.setState({ 
                                  resourceInputMethod: method,
                                  newResource: { 
                                    ...newResource, 
                                    url: '',
                                    type: method === 'upload' ? 'pdf' : 'youtube'
                                  }
                                });
                              }}
                            >
                              <FormControlLabel 
                                value="url" 
                                control={<Radio sx={{ color: '#38aae1', '&.Mui-checked': { color: '#38aae1' } }} />} 
                                label="URL" 
                              />
                              <FormControlLabel 
                                value="upload" 
                                control={<Radio sx={{ color: '#38aae1', '&.Mui-checked': { color: '#38aae1' } }} />} 
                                label="Upload File" 
                              />
                            </RadioGroup>
                          </FormControl>
                        </Grid>
                        <Grid item xs={12} md={6}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                            Resource Name
                          </Typography>
                          <TextField
                            fullWidth
                            value={newResource.name}
                            onChange={(e) => this.setState({
                              newResource: { ...newResource, name: e.target.value }
                            })}
                            placeholder="Resource name"
                            size="small"
                          />
                        </Grid>
                        <Grid item xs={12} md={6}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                            Resource Type
                          </Typography>
                          <FormControl fullWidth size="small">
                            <Select
                              value={newResource.type}
                              onChange={(e) => this.setState({
                                newResource: { ...newResource, type: e.target.value }
                              })}
                            >
                              {this.state.resourceInputMethod === 'upload' ? (
                                <>
                                  <MenuItem value="pdf">PDF Document</MenuItem>
                                  <MenuItem value="doc">Word Document</MenuItem>
                                  <MenuItem value="ppt">PowerPoint Presentation</MenuItem>
                                </>
                              ) : (
                                <>
                                  <MenuItem value="youtube">YouTube Video</MenuItem>
                                  <MenuItem value="pdf">PDF Link</MenuItem>
                                  <MenuItem value="doc">Document Link</MenuItem>
                                  <MenuItem value="ppt">Presentation Link</MenuItem>
                                </>
                              )}
                            </Select>
                          </FormControl>
                        </Grid>
                        {this.state.resourceInputMethod === 'url' ? (
                          <Grid item xs={12}>
                            <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                              {newResource.type === 'youtube' ? 'YouTube URL' : 'Resource URL'}
                            </Typography>
                            <TextField
                              fullWidth
                              value={newResource.url}
                              onChange={(e) => this.setState({
                                newResource: { ...newResource, url: e.target.value }
                              })}
                              placeholder={
                                newResource.type === 'youtube' 
                                  ? 'Enter YouTube URL (e.g., https://www.youtube.com/watch?v=...)' 
                                  : 'Enter resource URL (e.g., https://example.com/document.pdf)'
                              }
                              size="small"
                            />
                          </Grid>
                        ) : (
                          <Grid item xs={12}>
                            <Typography variant="body2" sx={{ mb: 1, fontWeight: 500, color: '#25476a' }}>
                              Upload File
                            </Typography>
                            <Box sx={{ 
                              border: '2px dashed #38aae1', 
                              borderRadius: 2, 
                              p: 3, 
                              textAlign: 'center',
                              bgcolor: 'white',
                              cursor: 'pointer',
                              '&:hover': {
                                bgcolor: '#f8fafc',
                                borderColor: '#25476a'
                              }
                            }}>
                              <input
                                accept=".pdf,.doc,.docx,.ppt,.pptx"
                                style={{ display: 'none' }}
                                id="resource-file-upload"
                                type="file"
                                onChange={this.handleResourceFileUpload}
                              />
                              <label htmlFor="resource-file-upload" style={{ cursor: 'pointer', width: '100%' }}>
                                <CloudUpload sx={{ fontSize: 48, color: '#38aae1', mb: 1 }} />
                                <Typography variant="body1" sx={{ color: '#25476a', fontWeight: 600, mb: 0.5 }}>
                                  {newResource.url ? 'File Selected' : 'Click to Upload File'}
                                </Typography>
                                <Typography variant="body2" sx={{ color: '#64748b' }}>
                                  {newResource.url ? newResource.name || 'File uploaded' : 'PDF, DOC, DOCX, PPT, PPTX files supported'}
                                </Typography>
                              </label>
                            </Box>
                          </Grid>
                        )}
                      </Grid>
                    </Box>

                    {/* Existing Resources */}
                    {currentSession.resources.map((resource, index) => (
                      <Box 
                        key={index}
                        sx={{ 
                          display: 'flex', 
                          alignItems: 'center', 
                          justifyContent: 'space-between',
                          p: 2,
                          mb: 1,
                          bgcolor: 'white',
                          borderRadius: 2,
                          border: '1px solid #e2e8f0'
                        }}
                      >
                        <Box>
                          <Typography variant="body1" sx={{ fontWeight: 600, color: '#25476a' }}>
                            {resource.name}
                          </Typography>
                          <Typography variant="caption" sx={{ color: '#64748b', textTransform: 'uppercase' }}>
                            {resource.type}
                          </Typography>
                        </Box>
                        <IconButton 
                          size="small" 
                          onClick={() => this.removeResource(index)}
                          sx={{ color: '#ef4444' }}
                        >
                          <Delete fontSize="small" />
                        </IconButton>
                      </Box>
                    ))}
                  </Box>
                )}
              </CardContent>
            </Card>
          </Grid>
        </Grid>
      </Box>
    );
  }
}