import React, { Component } from 'react';
import {
  Box,
  Typography,
  Button,
  Card,
  CardContent,
  Grid,
  TextField,
  InputAdornment,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Chip,
  Avatar,
  IconButton,
  Menu,
  MenuItem,
  LinearProgress,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Tabs,
  Tab,
  Select,
  FormControl,
  InputLabel,
} from '@mui/material';
import {
  Add,
  Search,
  MoreVert,
  School,
  TrendingUp,
  CheckCircle,
  AccessTime,
  Group,
  Edit,
  Delete,
  Visibility,
  Mail,
  Phone,
  CalendarMonth,
  Person,
} from '@mui/icons-material';

interface Learner {
  id: string;
  name: string;
  email: string;
  phone: string;
  age: number;
  enrollmentDate: string;
  status: 'Active' | 'Inactive' | 'Suspended' | 'Graduated';
  coursesEnrolled: number;
  coursesCompleted: number;
  overallProgress: number;
  currentClass: string;
  mentor: string;
  lastActive: string;
  totalHours: number;
  grade: string;
}

interface LearnersState {
  learners: Learner[];
  anchorEl: HTMLElement | null;
  selectedLearner: Learner | null;
  viewDialogOpen: boolean;
  deleteDialogOpen: boolean;
  searchQuery: string;
  selectedTab: number;
  filterClass: string;
}

export class Learners extends Component<{}, LearnersState> {
  constructor(props: {}) {
    super(props);
    this.state = {
      anchorEl: null,
      selectedLearner: null,
      viewDialogOpen: false,
      deleteDialogOpen: false,
      searchQuery: '',
      selectedTab: 0,
      filterClass: 'all',
      learners: [
        {
          id: '1',
          name: 'Sarah Johnson',
          email: 'sarah.j@email.com',
          phone: '+1 234 567 8901',
          age: 22,
          enrollmentDate: '2024-01-15',
          status: 'Active',
          coursesEnrolled: 3,
          coursesCompleted: 1,
          overallProgress: 78,
          currentClass: 'Web Development Bootcamp',
          mentor: 'Michael Chen',
          lastActive: '2024-04-18',
          totalHours: 156,
          grade: 'A',
        },
        {
          id: '2',
          name: 'David Martinez',
          email: 'david.m@email.com',
          phone: '+1 234 567 8902',
          age: 25,
          enrollmentDate: '2024-02-01',
          status: 'Active',
          coursesEnrolled: 2,
          coursesCompleted: 1,
          overallProgress: 92,
          currentClass: 'Data Science Track',
          mentor: 'Emily Rodriguez',
          lastActive: '2024-04-19',
          totalHours: 184,
          grade: 'A+',
        },
        {
          id: '3',
          name: 'Emma Wilson',
          email: 'emma.w@email.com',
          phone: '+1 234 567 8903',
          age: 20,
          enrollmentDate: '2024-01-20',
          status: 'Graduated',
          coursesEnrolled: 1,
          coursesCompleted: 1,
          overallProgress: 100,
          currentClass: 'UI/UX Design Fundamentals',
          mentor: 'James Wilson',
          lastActive: '2024-04-17',
          totalHours: 120,
          grade: 'A',
        },
        {
          id: '4',
          name: 'Michael Brown',
          email: 'michael.b@email.com',
          phone: '+1 234 567 8904',
          age: 23,
          enrollmentDate: '2024-02-10',
          status: 'Active',
          coursesEnrolled: 2,
          coursesCompleted: 0,
          overallProgress: 42,
          currentClass: 'Mobile App Development',
          mentor: 'Sarah Kim',
          lastActive: '2024-04-16',
          totalHours: 84,
          grade: 'B+',
        },
        {
          id: '5',
          name: 'Lisa Anderson',
          email: 'lisa.a@email.com',
          phone: '+1 234 567 8905',
          age: 28,
          enrollmentDate: '2023-10-01',
          status: 'Graduated',
          coursesEnrolled: 2,
          coursesCompleted: 2,
          overallProgress: 100,
          currentClass: 'Digital Marketing Essentials',
          mentor: 'Alex Johnson',
          lastActive: '2024-01-30',
          totalHours: 240,
          grade: 'A+',
        },
        {
          id: '6',
          name: 'James Taylor',
          email: 'james.t@email.com',
          phone: '+1 234 567 8906',
          age: 19,
          enrollmentDate: '2024-03-01',
          status: 'Active',
          coursesEnrolled: 1,
          coursesCompleted: 0,
          overallProgress: 35,
          currentClass: 'Web Development Bootcamp',
          mentor: 'Michael Chen',
          lastActive: '2024-04-18',
          totalHours: 48,
          grade: 'B',
        },
        {
          id: '7',
          name: 'Sophia Garcia',
          email: 'sophia.g@email.com',
          phone: '+1 234 567 8907',
          age: 24,
          enrollmentDate: '2024-01-25',
          status: 'Inactive',
          coursesEnrolled: 1,
          coursesCompleted: 0,
          overallProgress: 15,
          currentClass: 'Data Science Track',
          mentor: 'Emily Rodriguez',
          lastActive: '2024-03-10',
          totalHours: 24,
          grade: 'C',
        },
        {
          id: '8',
          name: 'Robert Lee',
          email: 'robert.l@email.com',
          phone: '+1 234 567 8908',
          age: 26,
          enrollmentDate: '2024-02-15',
          status: 'Active',
          coursesEnrolled: 2,
          coursesCompleted: 1,
          overallProgress: 68,
          currentClass: 'Mobile App Development',
          mentor: 'Sarah Kim',
          lastActive: '2024-04-17',
          totalHours: 136,
          grade: 'A-',
        },
      ],
    };
  }

  handleMenuClick = (event: React.MouseEvent<HTMLElement>, learner: Learner) => {
    this.setState({ anchorEl: event.currentTarget, selectedLearner: learner });
  };

  handleMenuClose = () => {
    this.setState({ anchorEl: null });
  };

  handleView = () => {
    this.setState({ viewDialogOpen: true, anchorEl: null });
  };

  handleCloseViewDialog = () => {
    this.setState({ viewDialogOpen: false });
  };

  handleDelete = () => {
    this.setState({ deleteDialogOpen: true, anchorEl: null });
  };

  handleConfirmDelete = () => {
    const { selectedLearner, learners } = this.state;
    if (selectedLearner) {
      const updatedLearners = learners.filter(l => l.id !== selectedLearner.id);
      this.setState({ learners: updatedLearners, deleteDialogOpen: false, selectedLearner: null });
    }
  };

  handleCancelDelete = () => {
    this.setState({ deleteDialogOpen: false });
  };

  handleSearchChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    this.setState({ searchQuery: event.target.value });
  };

  handleTabChange = (event: any, newValue: number) => {
    this.setState({ selectedTab: newValue });
  };

  handleFilterChange = (event: any) => {
    this.setState({ filterClass: event.target.value });
  };

  getStatusColor = (status: string) => {
    switch (status) {
      case 'Active':
        return { bg: '#dcfce7', color: '#166534' };
      case 'Inactive':
        return { bg: '#fee2e2', color: '#991b1b' };
      case 'Suspended':
        return { bg: '#fef3c7', color: '#92400e' };
      case 'Graduated':
        return { bg: '#dbeafe', color: '#1e40af' };
      default:
        return { bg: '#f3f4f6', color: '#374151' };
    }
  };

  getGradeColor = (grade: string) => {
    if (grade.startsWith('A')) {
      return { bg: '#dcfce7', color: '#166534' };
    } else if (grade.startsWith('B')) {
      return { bg: '#dbeafe', color: '#1e40af' };
    } else if (grade.startsWith('C')) {
      return { bg: '#fef3c7', color: '#92400e' };
    } else {
      return { bg: '#fee2e2', color: '#991b1b' };
    }
  };

  getFilteredLearners = () => {
    const { learners, searchQuery, selectedTab, filterClass } = this.state;
    let filtered = learners;

    // Filter by tab
    if (selectedTab === 1) {
      filtered = filtered.filter(l => l.status === 'Active');
    } else if (selectedTab === 2) {
      filtered = filtered.filter(l => l.status === 'Graduated');
    } else if (selectedTab === 3) {
      filtered = filtered.filter(l => l.status === 'Inactive');
    }

    // Filter by class
    if (filterClass !== 'all') {
      filtered = filtered.filter(l => l.currentClass === filterClass);
    }

    // Filter by search
    if (searchQuery) {
      filtered = filtered.filter(l =>
        l.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
        l.email.toLowerCase().includes(searchQuery.toLowerCase()) ||
        l.currentClass.toLowerCase().includes(searchQuery.toLowerCase())
      );
    }

    return filtered;
  };

  render() {
    const { anchorEl, selectedLearner, viewDialogOpen, deleteDialogOpen, searchQuery, selectedTab, filterClass, learners } = this.state;
    const filteredLearners = this.getFilteredLearners();

    const totalLearners = learners.length;
    const activeLearners = learners.filter(l => l.status === 'Active').length;
    const graduatedLearners = learners.filter(l => l.status === 'Graduated').length;
    const avgProgress = Math.round(learners.reduce((sum, l) => sum + l.overallProgress, 0) / learners.length);

    const uniqueClasses = Array.from(new Set(learners.map(l => l.currentClass)));

    return (
      <Box sx={{ p: 4, bgcolor: '#f8fafc', minHeight: '100vh' }}>
        {/* Header */}
        <Box sx={{ mb: 4 }}>
          <Typography variant="h4" sx={{ color: '#25476a', fontWeight: 700, mb: 1 }}>
            Learners Management
          </Typography>
          <Typography variant="body1" sx={{ color: '#64748b' }}>
            Manage all learners, track progress, and monitor performance
          </Typography>
        </Box>

        {/* Summary Cards */}
        <Grid container spacing={3} sx={{ mb: 4 }}>
          <Grid item xs={12} sm={6} md={3}>
            <Card
              sx={{
                borderRadius: 3,
                border: '1px solid #e2e8f0',
                boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
                transition: 'all 0.3s',
                '&:hover': {
                  boxShadow: '0 8px 16px rgba(56, 170, 225, 0.15)',
                  transform: 'translateY(-4px)',
                },
              }}
            >
              <CardContent sx={{ p: 3 }}>
                <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                  <Box sx={{ p: 1.5, bgcolor: 'rgba(56, 170, 225, 0.1)', borderRadius: 2 }}>
                    <Group sx={{ color: '#38aae1', fontSize: 28 }} />
                  </Box>
                </Box>
                <Typography variant="body2" sx={{ color: '#64748b', mb: 1, fontWeight: 500 }}>
                  Total Learners
                </Typography>
                <Typography variant="h3" sx={{ color: '#25476a', fontWeight: 700 }}>
                  {totalLearners}
                </Typography>
              </CardContent>
            </Card>
          </Grid>

          <Grid item xs={12} sm={6} md={3}>
            <Card
              sx={{
                borderRadius: 3,
                border: '1px solid #e2e8f0',
                boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
                transition: 'all 0.3s',
                '&:hover': {
                  boxShadow: '0 8px 16px rgba(34, 197, 94, 0.15)',
                  transform: 'translateY(-4px)',
                },
              }}
            >
              <CardContent sx={{ p: 3 }}>
                <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                  <Box sx={{ p: 1.5, bgcolor: 'rgba(34, 197, 94, 0.1)', borderRadius: 2 }}>
                    <CheckCircle sx={{ color: '#22c55e', fontSize: 28 }} />
                  </Box>
                  <Chip
                    label="Active"
                    size="small"
                    sx={{
                      bgcolor: '#dcfce7',
                      color: '#166534',
                      fontWeight: 600,
                      fontSize: '0.75rem',
                    }}
                  />
                </Box>
                <Typography variant="body2" sx={{ color: '#64748b', mb: 1, fontWeight: 500 }}>
                  Active Learners
                </Typography>
                <Typography variant="h3" sx={{ color: '#25476a', fontWeight: 700 }}>
                  {activeLearners}
                </Typography>
              </CardContent>
            </Card>
          </Grid>

          <Grid item xs={12} sm={6} md={3}>
            <Card
              sx={{
                borderRadius: 3,
                border: '1px solid #e2e8f0',
                boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
                transition: 'all 0.3s',
                '&:hover': {
                  boxShadow: '0 8px 16px rgba(139, 92, 246, 0.15)',
                  transform: 'translateY(-4px)',
                },
              }}
            >
              <CardContent sx={{ p: 3 }}>
                <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                  <Box sx={{ p: 1.5, bgcolor: 'rgba(139, 92, 246, 0.1)', borderRadius: 2 }}>
                    <School sx={{ color: '#8b5cf6', fontSize: 28 }} />
                  </Box>
                  <Chip
                    label="Graduated"
                    size="small"
                    sx={{
                      bgcolor: '#dbeafe',
                      color: '#1e40af',
                      fontWeight: 600,
                      fontSize: '0.75rem',
                    }}
                  />
                </Box>
                <Typography variant="body2" sx={{ color: '#64748b', mb: 1, fontWeight: 500 }}>
                  Graduated
                </Typography>
                <Typography variant="h3" sx={{ color: '#25476a', fontWeight: 700 }}>
                  {graduatedLearners}
                </Typography>
              </CardContent>
            </Card>
          </Grid>

          <Grid item xs={12} sm={6} md={3}>
            <Card
              sx={{
                borderRadius: 3,
                border: '1px solid #e2e8f0',
                boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
                transition: 'all 0.3s',
                '&:hover': {
                  boxShadow: '0 8px 16px rgba(254, 177, 57, 0.15)',
                  transform: 'translateY(-4px)',
                },
              }}
            >
              <CardContent sx={{ p: 3 }}>
                <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                  <Box sx={{ p: 1.5, bgcolor: 'rgba(254, 177, 57, 0.1)', borderRadius: 2 }}>
                    <TrendingUp sx={{ color: '#feb139', fontSize: 28 }} />
                  </Box>
                </Box>
                <Typography variant="body2" sx={{ color: '#64748b', mb: 1, fontWeight: 500 }}>
                  Avg Progress
                </Typography>
                <Typography variant="h3" sx={{ color: '#25476a', fontWeight: 700 }}>
                  {avgProgress}%
                </Typography>
              </CardContent>
            </Card>
          </Grid>
        </Grid>

        {/* Main Content Card */}
        <Card
          sx={{
            borderRadius: 3,
            border: '1px solid #e2e8f0',
            boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
          }}
        >
          <CardContent sx={{ p: 3 }}>
            {/* Toolbar */}
            <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3, gap: 2 }}>
              <Box sx={{ display: 'flex', gap: 2, flex: 1 }}>
                <TextField
                  placeholder="Search learners..."
                  value={searchQuery}
                  onChange={this.handleSearchChange}
                  variant="outlined"
                  size="small"
                  sx={{
                    width: 300,
                    '& .MuiOutlinedInput-root': {
                      borderRadius: 2,
                    },
                  }}
                  InputProps={{
                    startAdornment: (
                      <InputAdornment position="start">
                        <Search sx={{ color: '#64748b' }} />
                      </InputAdornment>
                    ),
                  }}
                />
                <FormControl size="small" sx={{ minWidth: 200 }}>
                  <InputLabel>Filter by Class</InputLabel>
                  <Select
                    value={filterClass}
                    onChange={this.handleFilterChange}
                    label="Filter by Class"
                  >
                    <MenuItem value="all">All Classes</MenuItem>
                    {uniqueClasses.map((className) => (
                      <MenuItem key={className} value={className}>
                        {className}
                      </MenuItem>
                    ))}
                  </Select>
                </FormControl>
              </Box>
              <Button
                variant="contained"
                startIcon={<Add />}
                sx={{
                  bgcolor: '#38aae1',
                  color: '#ffffff',
                  textTransform: 'none',
                  borderRadius: 2,
                  px: 3,
                  '&:hover': {
                    bgcolor: '#2c8fc9',
                  },
                }}
              >
                Add New Learner
              </Button>
            </Box>

            {/* Tabs */}
            <Box sx={{ borderBottom: 1, borderColor: '#e2e8f0', mb: 3 }}>
              <Tabs value={selectedTab} onChange={this.handleTabChange}>
                <Tab label="All Learners" />
                <Tab label="Active" />
                <Tab label="Graduated" />
                <Tab label="Inactive" />
              </Tabs>
            </Box>

            {/* Learners Table */}
            <TableContainer>
              <Table>
                <TableHead>
                  <TableRow sx={{ bgcolor: '#f8fafc' }}>
                    <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                      Learner Name
                    </TableCell>
                    <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                      Contact
                    </TableCell>
                    <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                      Current Class
                    </TableCell>
                    <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                      Courses
                    </TableCell>
                    <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                      Progress
                    </TableCell>
                    <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                      Grade
                    </TableCell>
                    <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                      Status
                    </TableCell>
                    <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                      Actions
                    </TableCell>
                  </TableRow>
                </TableHead>
                <TableBody>
                  {filteredLearners.map((learner) => {
                    const statusColor = this.getStatusColor(learner.status);
                    const gradeColor = this.getGradeColor(learner.grade);
                    return (
                      <TableRow key={learner.id} sx={{ '&:hover': { bgcolor: '#f8fafc' } }}>
                        <TableCell sx={{ border: 'none' }}>
                          <Box sx={{ display: 'flex', alignItems: 'center' }}>
                            <Avatar
                              sx={{
                                width: 40,
                                height: 40,
                                bgcolor: '#38aae1',
                                mr: 2,
                                fontSize: '0.875rem',
                                fontWeight: 600,
                              }}
                            >
                              {learner.name.split(' ').map(n => n[0]).join('')}
                            </Avatar>
                            <Box>
                              <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                                {learner.name}
                              </Typography>
                              <Typography variant="caption" sx={{ color: '#94a3b8' }}>
                                Age: {learner.age}
                              </Typography>
                            </Box>
                          </Box>
                        </TableCell>
                        <TableCell sx={{ border: 'none' }}>
                          <Box>
                            <Box sx={{ display: 'flex', alignItems: 'center', mb: 0.5 }}>
                              <Mail sx={{ fontSize: 14, color: '#64748b', mr: 0.5 }} />
                              <Typography variant="caption" sx={{ color: '#64748b' }}>
                                {learner.email}
                              </Typography>
                            </Box>
                            <Box sx={{ display: 'flex', alignItems: 'center' }}>
                              <Phone sx={{ fontSize: 14, color: '#64748b', mr: 0.5 }} />
                              <Typography variant="caption" sx={{ color: '#64748b' }}>
                                {learner.phone}
                              </Typography>
                            </Box>
                          </Box>
                        </TableCell>
                        <TableCell sx={{ border: 'none' }}>
                          <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 500 }}>
                            {learner.currentClass}
                          </Typography>
                          <Typography variant="caption" sx={{ color: '#94a3b8' }}>
                            Mentor: {learner.mentor}
                          </Typography>
                        </TableCell>
                        <TableCell sx={{ border: 'none' }}>
                          <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                            {learner.coursesCompleted}/{learner.coursesEnrolled}
                          </Typography>
                          <Typography variant="caption" sx={{ color: '#94a3b8' }}>
                            completed
                          </Typography>
                        </TableCell>
                        <TableCell sx={{ border: 'none' }}>
                          <Box sx={{ minWidth: 100 }}>
                            <Box sx={{ display: 'flex', justifyContent: 'space-between', mb: 0.5 }}>
                              <Typography variant="caption" sx={{ color: '#64748b' }}>
                                {learner.overallProgress}%
                              </Typography>
                            </Box>
                            <LinearProgress
                              variant="determinate"
                              value={learner.overallProgress}
                              sx={{
                                height: 6,
                                borderRadius: 3,
                                bgcolor: '#e2e8f0',
                                '& .MuiLinearProgress-bar': {
                                  bgcolor: learner.overallProgress === 100 ? '#22c55e' : '#38aae1',
                                  borderRadius: 3,
                                },
                              }}
                            />
                          </Box>
                        </TableCell>
                        <TableCell sx={{ border: 'none' }}>
                          <Chip
                            label={learner.grade}
                            size="small"
                            sx={{
                              bgcolor: gradeColor.bg,
                              color: gradeColor.color,
                              fontWeight: 700,
                              fontSize: '0.75rem',
                              minWidth: 45,
                            }}
                          />
                        </TableCell>
                        <TableCell sx={{ border: 'none' }}>
                          <Chip
                            label={learner.status}
                            size="small"
                            sx={{
                              bgcolor: statusColor.bg,
                              color: statusColor.color,
                              fontWeight: 600,
                              fontSize: '0.75rem',
                            }}
                          />
                        </TableCell>
                        <TableCell sx={{ border: 'none' }}>
                          <IconButton
                            size="small"
                            onClick={(e) => this.handleMenuClick(e, learner)}
                            sx={{ color: '#64748b' }}
                          >
                            <MoreVert />
                          </IconButton>
                        </TableCell>
                      </TableRow>
                    );
                  })}
                </TableBody>
              </Table>
            </TableContainer>
          </CardContent>
        </Card>

        {/* Menu */}
        <Menu
          anchorEl={anchorEl}
          open={Boolean(anchorEl)}
          onClose={this.handleMenuClose}
        >
          <MenuItem onClick={this.handleView}>
            <Visibility sx={{ mr: 1, fontSize: 20 }} />
            View Details
          </MenuItem>
          <MenuItem onClick={this.handleMenuClose}>
            <Edit sx={{ mr: 1, fontSize: 20 }} />
            Edit
          </MenuItem>
          <MenuItem onClick={this.handleDelete} sx={{ color: '#dc2626' }}>
            <Delete sx={{ mr: 1, fontSize: 20 }} />
            Delete
          </MenuItem>
        </Menu>

        {/* View Dialog */}
        <Dialog
          open={viewDialogOpen}
          onClose={this.handleCloseViewDialog}
          maxWidth="md"
          fullWidth
        >
          {selectedLearner && (
            <>
              <DialogTitle sx={{ bgcolor: '#f8fafc', borderBottom: '1px solid #e2e8f0' }}>
                <Box sx={{ display: 'flex', alignItems: 'center' }}>
                  <Avatar
                    sx={{
                      width: 56,
                      height: 56,
                      bgcolor: '#38aae1',
                      mr: 2,
                      fontSize: '1.25rem',
                      fontWeight: 700,
                    }}
                  >
                    {selectedLearner.name.split(' ').map(n => n[0]).join('')}
                  </Avatar>
                  <Box>
                    <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 700 }}>
                      {selectedLearner.name}
                    </Typography>
                    <Typography variant="caption" sx={{ color: '#64748b' }}>
                      {selectedLearner.email}
                    </Typography>
                  </Box>
                </Box>
              </DialogTitle>
              <DialogContent sx={{ p: 4 }}>
                <Grid container spacing={3}>
                  <Grid item xs={12} md={6}>
                    <Typography variant="caption" sx={{ color: '#94a3b8', display: 'block', mb: 0.5 }}>
                      Phone
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {selectedLearner.phone}
                    </Typography>
                  </Grid>
                  <Grid item xs={12} md={6}>
                    <Typography variant="caption" sx={{ color: '#94a3b8', display: 'block', mb: 0.5 }}>
                      Age
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {selectedLearner.age} years
                    </Typography>
                  </Grid>
                  <Grid item xs={12} md={6}>
                    <Typography variant="caption" sx={{ color: '#94a3b8', display: 'block', mb: 0.5 }}>
                      Current Class
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {selectedLearner.currentClass}
                    </Typography>
                  </Grid>
                  <Grid item xs={12} md={6}>
                    <Typography variant="caption" sx={{ color: '#94a3b8', display: 'block', mb: 0.5 }}>
                      Mentor
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {selectedLearner.mentor}
                    </Typography>
                  </Grid>
                  <Grid item xs={12} md={6}>
                    <Typography variant="caption" sx={{ color: '#94a3b8', display: 'block', mb: 0.5 }}>
                      Enrollment Date
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {new Date(selectedLearner.enrollmentDate).toLocaleDateString()}
                    </Typography>
                  </Grid>
                  <Grid item xs={12} md={6}>
                    <Typography variant="caption" sx={{ color: '#94a3b8', display: 'block', mb: 0.5 }}>
                      Last Active
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {new Date(selectedLearner.lastActive).toLocaleDateString()}
                    </Typography>
                  </Grid>
                  <Grid item xs={12} md={6}>
                    <Typography variant="caption" sx={{ color: '#94a3b8', display: 'block', mb: 0.5 }}>
                      Total Learning Hours
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {selectedLearner.totalHours} hours
                    </Typography>
                  </Grid>
                  <Grid item xs={12} md={6}>
                    <Typography variant="caption" sx={{ color: '#94a3b8', display: 'block', mb: 0.5 }}>
                      Current Grade
                    </Typography>
                    <Chip
                      label={selectedLearner.grade}
                      sx={{
                        bgcolor: this.getGradeColor(selectedLearner.grade).bg,
                        color: this.getGradeColor(selectedLearner.grade).color,
                        fontWeight: 700,
                      }}
                    />
                  </Grid>
                  <Grid item xs={12}>
                    <Typography variant="caption" sx={{ color: '#94a3b8', display: 'block', mb: 1 }}>
                      Overall Progress
                    </Typography>
                    <Box sx={{ display: 'flex', alignItems: 'center' }}>
                      <Box sx={{ flex: 1, mr: 2 }}>
                        <LinearProgress
                          variant="determinate"
                          value={selectedLearner.overallProgress}
                          sx={{
                            height: 10,
                            borderRadius: 5,
                            bgcolor: '#e2e8f0',
                            '& .MuiLinearProgress-bar': {
                              bgcolor: selectedLearner.overallProgress === 100 ? '#22c55e' : '#38aae1',
                              borderRadius: 5,
                            },
                          }}
                        />
                      </Box>
                      <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 700 }}>
                        {selectedLearner.overallProgress}%
                      </Typography>
                    </Box>
                  </Grid>
                  <Grid item xs={12}>
                    <Typography variant="caption" sx={{ color: '#94a3b8', display: 'block', mb: 0.5 }}>
                      Course Completion
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {selectedLearner.coursesCompleted} of {selectedLearner.coursesEnrolled} courses completed
                    </Typography>
                  </Grid>
                </Grid>
              </DialogContent>
              <DialogActions sx={{ p: 3, bgcolor: '#f8fafc' }}>
                <Button onClick={this.handleCloseViewDialog} sx={{ color: '#64748b' }}>
                  Close
                </Button>
                <Button
                  variant="contained"
                  startIcon={<Edit />}
                  sx={{
                    bgcolor: '#38aae1',
                    color: '#ffffff',
                    '&:hover': {
                      bgcolor: '#2c8fc9',
                    },
                  }}
                >
                  Edit Learner
                </Button>
              </DialogActions>
            </>
          )}
        </Dialog>

        {/* Delete Confirmation Dialog */}
        <Dialog open={deleteDialogOpen} onClose={this.handleCancelDelete}>
          <DialogTitle>Confirm Delete</DialogTitle>
          <DialogContent>
            <Typography>
              Are you sure you want to delete <strong>{selectedLearner?.name}</strong>?
              This action cannot be undone.
            </Typography>
          </DialogContent>
          <DialogActions>
            <Button onClick={this.handleCancelDelete} sx={{ color: '#64748b' }}>
              Cancel
            </Button>
            <Button onClick={this.handleConfirmDelete} sx={{ color: '#dc2626' }}>
              Delete
            </Button>
          </DialogActions>
        </Dialog>
      </Box>
    );
  }
}
