import React, { Component } from 'react';
import {
  Box,
  Typography,
  Grid,
  Card,
  CardContent,
  Button,
  Avatar,
  IconButton,
  CircularProgress,
} from '@mui/material';
import {
  People,
  School,
  AttachMoney,
  TrendingUp,
  Person,
  EventAvailable,
  AccessTime,
} from '@mui/icons-material';
import FullCalendar from '@fullcalendar/react';
import dayGridPlugin from '@fullcalendar/daygrid';
import timeGridPlugin from '@fullcalendar/timegrid';
import interactionPlugin from '@fullcalendar/interaction';
import listPlugin from '@fullcalendar/list';

interface Lesson {
  id: string;
  title: string;
  start: Date | string;
  end: Date | string;
  instructor?: string;
  students?: number;
  type?: 'class' | 'mentor' | 'break';
  backgroundColor?: string;
  borderColor?: string;
  textColor?: string;
}

interface DashboardContentState {
  todayLessons: Lesson[];
  isLoadingLessons: boolean;
  isLessonsLoaded: boolean;
}

export class DashboardContent extends Component<{}, DashboardContentState> {
  private calendarRef = React.createRef<FullCalendar>();

  constructor(props: {}) {
    super(props);
    this.state = {
      todayLessons: [],
      isLoadingLessons: true,
      isLessonsLoaded: false,
    };
  }

  componentDidMount() {
    this.loadTodayLessons();
  }

  loadTodayLessons = () => {
    this.setState({ isLoadingLessons: true });

    // Simulate API call
    setTimeout(() => {
      // Example today's lessons - uncomment to show data
      // const today = new Date();
      // const todayStr = today.toISOString().split('T')[0];
      // const sampleLessons: Lesson[] = [
      //   {
      //     id: '1',
      //     title: 'Introduction to Algebra',
      //     start: `${todayStr}T10:00:00`,
      //     end: `${todayStr}T11:00:00`,
      //     instructor: 'Dr. Sarah Johnson',
      //     students: 25,
      //     type: 'class',
      //     backgroundColor: '#38aae1',
      //     borderColor: '#2c8fc9',
      //     textColor: '#ffffff',
      //   },
      //   {
      //     id: '2',
      //     title: 'Chemistry Lab Session',
      //     start: `${todayStr}T14:00:00`,
      //     end: `${todayStr}T15:30:00`,
      //     instructor: 'Prof. Michael Chen',
      //     students: 18,
      //     type: 'class',
      //     backgroundColor: '#feb139',
      //     borderColor: '#e69b2a',
      //     textColor: '#ffffff',
      //   },
      // ];

      this.setState({
        todayLessons: [], // Empty to show empty state
        isLoadingLessons: false,
        isLessonsLoaded: true,
      });
    }, 1500);
  };

  handleDateClick = (arg: any) => {
    console.log('Date clicked:', arg.dateStr);
  };

  handleEventClick = (clickInfo: any) => {
    console.log('Lesson clicked:', clickInfo.event.title);
  };

  renderTodayTimetable = () => {
    const { isLoadingLessons, isLessonsLoaded, todayLessons } = this.state;

    if (isLoadingLessons) {
      return (
        <Box
          sx={{
            display: 'flex',
            flexDirection: 'column',
            alignItems: 'center',
            justifyContent: 'center',
            minHeight: 300,
            textAlign: 'center',
            px: 3,
          }}
        >
          <CircularProgress
            size={60}
            thickness={4}
            sx={{
              color: '#38aae1',
              mb: 3,
            }}
          />
          <Box
            sx={{
              display: 'flex',
              alignItems: 'center',
              gap: 1,
              mb: 1,
            }}
          >
            <AccessTime
              sx={{
                fontSize: 24,
                color: '#38aae1',
              }}
            />
            <Typography
              variant="h6"
              sx={{
                color: '#25476a',
                fontWeight: 600,
              }}
            >
              Fetching lessons...
            </Typography>
          </Box>
          <Typography
            variant="body2"
            sx={{
              color: '#64748b',
              maxWidth: 300,
            }}
          >
            Please wait while we load today's schedule
          </Typography>
        </Box>
      );
    }

    if (!isLoadingLessons && isLessonsLoaded && todayLessons.length === 0) {
      return (
        <Box
          sx={{
            display: 'flex',
            flexDirection: 'column',
            alignItems: 'center',
            justifyContent: 'center',
            minHeight: 300,
            textAlign: 'center',
            px: 3,
          }}
        >
          <Box
            sx={{
              width: 80,
              height: 80,
              borderRadius: '50%',
              backgroundColor: '#e3f2fd',
              display: 'flex',
              alignItems: 'center',
              justifyContent: 'center',
              mb: 2,
            }}
          >
            <EventAvailable
              sx={{
                fontSize: 40,
                color: '#38aae1',
              }}
            />
          </Box>

          <Typography
            variant="h6"
            sx={{
              color: '#25476a',
              fontWeight: 600,
              mb: 1,
            }}
          >
            There are no lessons scheduled today
          </Typography>

          <Typography
            variant="body2"
            sx={{
              color: '#64748b',
              maxWidth: 300,
            }}
          >
            Enjoy your day off or add a new lesson to the calendar.
          </Typography>
        </Box>
      );
    }

    return (
      <Box
        sx={{
          '& .fc': {
            fontFamily: '"Inter", "Roboto", "Helvetica", "Arial", sans-serif',
          },
          '& .fc-toolbar-title': {
            color: '#25476a',
            fontWeight: 700,
            fontSize: '1.25rem',
          },
          '& .fc-button': {
            backgroundColor: '#38aae1',
            borderColor: '#38aae1',
            textTransform: 'none',
            fontWeight: 600,
            fontSize: '0.875rem',
            padding: '6px 12px',
            '&:hover': {
              backgroundColor: '#2c8fc9',
              borderColor: '#2c8fc9',
            },
            '&:focus': {
              boxShadow: '0 0 0 0.2rem rgba(56, 170, 225, 0.5)',
            },
            '&.fc-button-active': {
              backgroundColor: '#25476a',
              borderColor: '#25476a',
            },
          },
          '& .fc-col-header-cell': {
            backgroundColor: '#f8fafc',
            borderColor: '#e2e8f0',
            padding: '8px 0',
          },
          '& .fc-col-header-cell-cushion': {
            color: '#64748b',
            fontWeight: 600,
            fontSize: '0.8125rem',
          },
          '& .fc-timegrid-slot': {
            height: '2.5em',
          },
          '& .fc-timegrid-slot-label': {
            color: '#64748b',
            fontSize: '0.8125rem',
          },
          '& .fc-event': {
            borderRadius: '4px',
            padding: '2px 4px',
            fontSize: '0.8125rem',
            fontWeight: 500,
            cursor: 'pointer',
          },
          '& .fc-daygrid-day.fc-day-today': {
            backgroundColor: '#e3f2fd !important',
          },
          '& .fc-timegrid-col.fc-day-today': {
            backgroundColor: '#e3f2fd !important',
          },
        }}
      >
        <FullCalendar
          ref={this.calendarRef}
          plugins={[dayGridPlugin, timeGridPlugin, interactionPlugin, listPlugin]}
          initialView="timeGridDay"
          headerToolbar={{
            left: 'prev,next',
            center: 'title',
            right: 'timeGridDay,listDay',
          }}
          buttonText={{
            day: 'Day',
            list: 'List',
          }}
          events={todayLessons}
          dateClick={this.handleDateClick}
          eventClick={this.handleEventClick}
          editable={false}
          selectable={true}
          selectMirror={true}
          dayMaxEvents={true}
          height="auto"
          contentHeight={350}
          slotMinTime="08:00:00"
          slotMaxTime="18:00:00"
          allDaySlot={false}
        />
      </Box>
    );
  };

  render() {
    return (
      <Box sx={{ p: 4, backgroundColor: '#f8fafc', minHeight: '100vh' }}>
        {/* Header */}
        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 4 }}>
          <Box>
            <Typography
              variant="h4"
              sx={{
                fontWeight: 700,
                color: '#25476a',
                mb: 0.5,
              }}
            >
              Hello <span style={{ color: '#38aae1' }}>Thika West Center</span>
            </Typography>
            <Typography
              variant="body2"
              sx={{
                color: '#64748b',
              }}
            >
              Welcome back
            </Typography>
          </Box>
        </Box>

        {/* Stats Cards */}
        <Grid container spacing={3} sx={{ mb: 4 }}>
          <Grid item xs={12} sm={6} lg={3}>
            <Card
              sx={{
                boxShadow: '0 2px 8px rgba(0,0,0,0.08)',
                borderRadius: 3,
                position: 'relative',
                overflow: 'visible',
              }}
            >
              <CardContent sx={{ p: 3 }}>
                <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
                  <Box>
                    <Typography
                      variant="body2"
                      sx={{
                        color: '#64748b',
                        mb: 1,
                      }}
                    >
                      Learners
                    </Typography>
                    <Typography
                      variant="h4"
                      sx={{
                        fontWeight: 700,
                        color: '#25476a',
                      }}
                    >
                      63
                    </Typography>
                  </Box>
                  <Avatar
                    sx={{
                      bgcolor: '#e3f2fd',
                      width: 56,
                      height: 56,
                    }}
                  >
                    <Person sx={{ color: '#38aae1', fontSize: 28 }} />
                  </Avatar>
                </Box>
              </CardContent>
            </Card>
          </Grid>

          <Grid item xs={12} sm={6} lg={3}>
            <Card
              sx={{
                boxShadow: '0 2px 8px rgba(0,0,0,0.08)',
                borderRadius: 3,
              }}
            >
              <CardContent sx={{ p: 3 }}>
                <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
                  <Box>
                    <Typography
                      variant="body2"
                      sx={{
                        color: '#64748b',
                        mb: 1,
                      }}
                    >
                      Parents
                    </Typography>
                    <Typography
                      variant="h4"
                      sx={{
                        fontWeight: 700,
                        color: '#25476a',
                      }}
                    >
                      0
                    </Typography>
                  </Box>
                  <Avatar
                    sx={{
                      bgcolor: '#fff3e0',
                      width: 56,
                      height: 56,
                    }}
                  >
                    <People sx={{ color: '#feb139', fontSize: 28 }} />
                  </Avatar>
                </Box>
              </CardContent>
            </Card>
          </Grid>

          <Grid item xs={12} sm={6} lg={3}>
            <Card
              sx={{
                boxShadow: '0 2px 8px rgba(0,0,0,0.08)',
                borderRadius: 3,
              }}
            >
              <CardContent sx={{ p: 3 }}>
                <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
                  <Box>
                    <Typography
                      variant="body2"
                      sx={{
                        color: '#64748b',
                        mb: 1,
                      }}
                    >
                      Teachers
                    </Typography>
                    <Typography
                      variant="h4"
                      sx={{
                        fontWeight: 700,
                        color: '#25476a',
                      }}
                    >
                      0
                    </Typography>
                  </Box>
                  <Avatar
                    sx={{
                      bgcolor: '#e8f5e9',
                      width: 56,
                      height: 56,
                    }}
                  >
                    <School sx={{ color: '#4caf50', fontSize: 28 }} />
                  </Avatar>
                </Box>
              </CardContent>
            </Card>
          </Grid>

          <Grid item xs={12} sm={6} lg={3}>
            <Card
              sx={{
                boxShadow: '0 2px 8px rgba(0,0,0,0.08)',
                borderRadius: 3,
              }}
            >
              <CardContent sx={{ p: 3 }}>
                <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
                  <Box>
                    <Typography
                      variant="body2"
                      sx={{
                        color: '#64748b',
                        mb: 1,
                      }}
                    >
                      Revenue
                    </Typography>
                    <Typography
                      variant="h4"
                      sx={{
                        fontWeight: 700,
                        color: '#25476a',
                      }}
                    >
                      $0
                    </Typography>
                  </Box>
                  <Avatar
                    sx={{
                      bgcolor: '#fce4ec',
                      width: 56,
                      height: 56,
                    }}
                  >
                    <AttachMoney sx={{ color: '#e91e63', fontSize: 28 }} />
                  </Avatar>
                </Box>
              </CardContent>
            </Card>
          </Grid>
        </Grid>

        {/* Timetable Section */}
        <Box sx={{ mb: 4 }}>
          <Box
            sx={{
              display: 'flex',
              alignItems: 'center',
              justifyContent: 'space-between',
              mb: 3,
            }}
          >
            <Typography
              variant="h5"
              sx={{
                color: '#25476a',
                fontWeight: 700,
              }}
            >
              Today's Timetable
            </Typography>
            <Button
              variant="outlined"
              onClick={() => {}}
              sx={{
                borderColor: '#38aae1',
                color: '#38aae1',
                textTransform: 'none',
                fontWeight: 600,
                '&:hover': {
                  borderColor: '#2c8fc9',
                  backgroundColor: 'rgba(56, 170, 225, 0.04)',
                },
              }}
            >
              View Full Timetable
            </Button>
          </Box>

          <Card
            sx={{
              boxShadow: '0 4px 20px rgba(37, 71, 106, 0.08)',
              borderRadius: 3,
              border: '1px solid rgba(37, 71, 106, 0.08)',
              overflow: 'hidden',
            }}
          >
            <Box
              sx={{
                background: 'linear-gradient(135deg, #38aae1 0%, #2c8fc9 100%)',
                p: 2.5,
                borderBottom: '1px solid rgba(255,255,255,0.2)',
              }}
            >
              <Typography
                variant="h6"
                sx={{
                  color: '#ffffff',
                  fontWeight: 600,
                  display: 'flex',
                  alignItems: 'center',
                  gap: 1,
                }}
              >
                <EventAvailable sx={{ fontSize: 24 }} />
                Schedule Overview
              </Typography>
              <Typography
                variant="body2"
                sx={{
                  color: 'rgba(255,255,255,0.9)',
                  mt: 0.5,
                }}
              >
                {new Date().toLocaleDateString('en-US', { 
                  weekday: 'long', 
                  year: 'numeric', 
                  month: 'long', 
                  day: 'numeric' 
                })}
              </Typography>
            </Box>
            <CardContent sx={{ p: 3 }}>
              {this.renderTodayTimetable()}
            </CardContent>
          </Card>
        </Box>
      </Box>
    );
  }
}
