import React, { useState } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  TextField,
  Box,
  Typography,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Paper,
  List,
  ListItem,
  ListItemButton,
  ListItemText,
  ListItemAvatar,
  Avatar,
  IconButton,
  Switch,
  FormControlLabel,
  Divider,
  InputAdornment,
  Chip,
} from '@mui/material';
import {
  Close,
  Search,
  Add,
  Delete,
  DragIndicator,
  Star,
  School,
} from '@mui/icons-material';

interface Course {
  id: string;
  name: string;
  code: string;
  sessions: number;
  price: number;
  isDefault?: boolean;
}

interface CreateClassDialogProps {
  open: boolean;
  onClose: () => void;
  onCreateClass: (classData: any) => void;
  editingClass?: any;
}

const availableCourses: Course[] = [
  {
    id: 'DF002',
    name: 'Animations',
    code: 'DF002',
    sessions: 12,
    price: 3250,
  },
  {
    id: 'DF004',
    name: 'Game Design',
    code: 'DF004',
    sessions: 12,
    price: 4550,
  },
  {
    id: 'DF001',
    name: 'Web Development Basics',
    code: 'DF001',
    sessions: 12,
    price: 5850,
  },
  {
    id: 'DF003',
    name: 'Digital Art Fundamentals',
    code: 'DF003',
    sessions: 12,
    price: 2900,
  },
];

export function CreateClassDialog({ open, onClose, onCreateClass, editingClass }: CreateClassDialogProps) {
  const [selectedClass, setSelectedClass] = useState('');
  const [selectedCourses, setSelectedCourses] = useState<Course[]>([]);
  const [searchTerm, setSearchTerm] = useState('');
  const [publishToWebsite, setPublishToWebsite] = useState(false);
  const [passStandard, setPassStandard] = useState('meeting');

  // Initialize form with editing data
  React.useEffect(() => {
    if (editingClass) {
      setSelectedClass(editingClass.name);
      setSelectedCourses(editingClass.courses || []);
      setPublishToWebsite(editingClass.status === 'Published');
      setPassStandard(editingClass.passType === 'Meeting' ? 'meeting' : 'learning');
    } else {
      // Reset form for create mode
      setSelectedClass('');
      setSelectedCourses([]);
      setPublishToWebsite(false);
      setPassStandard('meeting');
    }
  }, [editingClass, open]);

  const handleAddCourse = (course: Course) => {
    if (!selectedCourses.find(c => c.id === course.id)) {
      const newCourse = { 
        ...course, 
        isDefault: selectedCourses.length === 0 // First course is default
      };
      setSelectedCourses([...selectedCourses, newCourse]);
    }
  };

  const handleRemoveCourse = (courseId: string) => {
    setSelectedCourses(selectedCourses.filter(c => c.id !== courseId));
  };

  const handlePriceChange = (courseId: string, price: number) => {
    setSelectedCourses(selectedCourses.map(c => 
      c.id === courseId ? { ...c, price } : c
    ));
  };

  const handleCreateClass = () => {
    const classData = {
      name: selectedClass === 'Crib' ? 'Crib Digital Skills' : selectedClass,
      coursesCount: selectedCourses.length,
      revenue: selectedCourses.reduce((sum, course) => sum + course.price, 0),
      students: 0,
      passType: passStandard === 'meeting' ? 'Meeting' : 'Learning',
      status: publishToWebsite ? 'Published' : 'Draft',
      courses: selectedCourses,
    };
    onCreateClass(classData);
  };

  const filteredCourses = availableCourses.filter(course =>
    course.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    course.code.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const totalValue = selectedCourses.reduce((sum, course) => sum + course.price, 0);

  return (
    <Dialog open={open} onClose={onClose} maxWidth="lg" fullWidth>
      <DialogTitle sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
        {editingClass ? 'Edit Self-Paced Class' : 'Create Self-Paced Class'}
        <IconButton onClick={onClose}>
          <Close />
        </IconButton>
      </DialogTitle>
      
      <DialogContent sx={{ minHeight: 600 }}>
        <Typography variant="body2" color="text.secondary" sx={{ mb: 3 }}>
          {editingClass ? 'Edit your self-paced class configuration.' : 'Create a new self-paced class with individualized course pacing and analytics.'}
        </Typography>

        {/* Select Class */}
        <Box sx={{ mb: 3 }}>
          <Typography variant="h6" sx={{ mb: 2 }}>Select Class</Typography>
          <FormControl fullWidth>
            <Select
              value={selectedClass}
              onChange={(e) => setSelectedClass(e.target.value)}
              displayEmpty
              renderValue={(selected) => selected || 'Select a class'}
            >
              <MenuItem value="Crib">Crib</MenuItem>
              <MenuItem value="Toddler">Toddler</MenuItem>
              <MenuItem value="Preschool">Preschool</MenuItem>
              <MenuItem value="Elementary">Elementary</MenuItem>
            </Select>
          </FormControl>
        </Box>

        {/* Course Selection */}
        <Box sx={{ mb: 3 }}>
          <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 2 }}>
            <Typography variant="h6">Select Courses (in order)</Typography>
            <Typography variant="body2" color="text.secondary">
              {selectedCourses.length} courses selected • Total: ${totalValue}
            </Typography>
          </Box>

          <Box sx={{ display: 'flex', gap: 3 }}>
            {/* Available Courses */}
            <Box sx={{ flex: 1 }}>
              <Typography variant="subtitle1" sx={{ mb: 2 }}>Available Courses</Typography>
              <TextField
                placeholder="Search courses..."
                size="small"
                fullWidth
                sx={{ mb: 2 }}
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                InputProps={{
                  startAdornment: (
                    <InputAdornment position="start">
                      <Search />
                    </InputAdornment>
                  ),
                }}
              />
              <Paper variant="outlined" sx={{ maxHeight: 300, overflow: 'auto' }}>
                <List>
                  {filteredCourses.map((course) => (
                    <ListItem
                      key={course.id}
                      disabled={selectedCourses.some(c => c.id === course.id)}
                    >
                      <ListItemButton onClick={() => handleAddCourse(course)}>
                        <ListItemAvatar>
                          <Avatar sx={{ bgcolor: '#38aae1' }}>
                            <School />
                          </Avatar>
                        </ListItemAvatar>
                        <ListItemText
                          primary={course.name}
                          secondary={`${course.code} • ${course.sessions} Sessions`}
                        />
                        <IconButton size="small">
                          <Add />
                        </IconButton>
                      </ListItemButton>
                    </ListItem>
                  ))}
                </List>
              </Paper>
            </Box>

            {/* Selected Courses */}
            <Box sx={{ flex: 1 }}>
              <Typography variant="subtitle1" sx={{ mb: 2 }}>Selected Courses (in order)</Typography>
              <Paper variant="outlined" sx={{ minHeight: 300, maxHeight: 300, overflow: 'auto' }}>
                <List>
                  {selectedCourses.map((course, index) => (
                    <ListItem key={course.id}>
                      <Box sx={{ display: 'flex', alignItems: 'center', width: '100%' }}>
                        <DragIndicator sx={{ color: 'text.secondary', mr: 1 }} />
                        <Box sx={{ display: 'flex', alignItems: 'center', mr: 2 }}>
                          <Avatar sx={{ bgcolor: '#38aae1', width: 32, height: 32, mr: 1 }}>
                            <School fontSize="small" />
                          </Avatar>
                          {course.isDefault && (
                            <Star sx={{ color: '#feb139', fontSize: 16, mr: 1 }} />
                          )}
                        </Box>
                        <Box sx={{ flex: 1 }}>
                          <Box sx={{ display: 'flex', alignItems: 'center', gap: 1, mb: 1 }}>
                            <Typography variant="body2" sx={{ fontWeight: 500 }}>
                              #{index + 1}
                            </Typography>
                            <Typography variant="body1">{course.code}</Typography>
                            {course.isDefault && (
                              <Chip 
                                label="Default Starting Point" 
                                size="small" 
                                color="warning"
                                sx={{ fontSize: '0.7rem' }}
                              />
                            )}
                          </Box>
                          <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                            <Typography variant="body2">Price:</Typography>
                            <TextField
                              size="small"
                              type="number"
                              value={course.price}
                              onChange={(e) => handlePriceChange(course.id, parseFloat(e.target.value) || 0)}
                              InputProps={{
                                startAdornment: <InputAdornment position="start">KES</InputAdornment>,
                              }}
                              sx={{ width: 100 }}
                            />
                          </Box>
                          {index === 0 && (
                            <Typography variant="caption" color="warning.main" sx={{ fontStyle: 'italic' }}>
                              Tip: The first course will be the default starting point. Set individual prices for each course.
                            </Typography>
                          )}
                        </Box>
                        <IconButton 
                          size="small" 
                          onClick={() => handleRemoveCourse(course.id)}
                          sx={{ color: 'error.main' }}
                        >
                          <Delete />
                        </IconButton>
                      </Box>
                    </ListItem>
                  ))}
                </List>
              </Paper>
            </Box>
          </Box>
        </Box>

        {/* Pass Mark Standards */}
        <Box sx={{ mb: 3 }}>
          <Typography variant="h6" sx={{ mb: 2 }}>Pass Mark Standards</Typography>
          <Box sx={{ display: 'flex', gap: 4 }}>
            <Box sx={{ flex: 1 }}>
              <FormControl component="fieldset">
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                  <input
                    type="radio"
                    value="meeting"
                    checked={passStandard === 'meeting'}
                    onChange={(e) => setPassStandard(e.target.value)}
                    style={{ marginRight: 8 }}
                  />
                  <Typography variant="subtitle1">Meeting Standards</Typography>
                </Box>
                <Typography variant="body2" color="text.secondary">
                  Students achieve 70-84% to meet curriculum requirements
                </Typography>
                <Box sx={{ display: 'flex', alignItems: 'center', mt: 1 }}>
                  <Box sx={{ width: 12, height: 12, bgcolor: '#4caf50', borderRadius: '50%', mr: 1 }} />
                  <Typography variant="body2" color="success.main">70-84% Pass</Typography>
                </Box>
              </FormControl>
            </Box>
            <Box sx={{ flex: 1 }}>
              <FormControl component="fieldset">
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
                  <input
                    type="radio"
                    value="exceeding"
                    checked={passStandard === 'exceeding'}
                    onChange={(e) => setPassStandard(e.target.value)}
                    style={{ marginRight: 8 }}
                  />
                  <Typography variant="subtitle1">Exceeding Standards</Typography>
                </Box>
                <Typography variant="body2" color="text.secondary">
                  Students achieve 85-100% demonstrating mastery
                </Typography>
                <Box sx={{ display: 'flex', alignItems: 'center', mt: 1 }}>
                  <Box sx={{ width: 12, height: 12, bgcolor: '#38aae1', borderRadius: '50%', mr: 1 }} />
                  <Typography variant="body2" color="primary">85-100% Excellence</Typography>
                </Box>
              </FormControl>
            </Box>
          </Box>
        </Box>

        <Divider sx={{ my: 3 }} />

        {/* Publish to Website */}
        <Box>
          <FormControlLabel
            control={
              <Switch
                checked={publishToWebsite}
                onChange={(e) => setPublishToWebsite(e.target.checked)}
                color="primary"
              />
            }
            label={
              <Box>
                <Typography variant="subtitle1">Publish to Website</Typography>
                <Typography variant="body2" color="text.secondary">
                  Make this class available to students for individual course purchases
                </Typography>
              </Box>
            }
          />
        </Box>
      </DialogContent>

      <DialogActions sx={{ p: 3, bgcolor: '#f8f9fa' }}>
        <Box sx={{ display: 'flex', justifyContent: 'space-between', width: '100%', alignItems: 'center' }}>
          <Typography variant="body2" color="text.secondary">
            {selectedCourses.length} courses • Total value: ${totalValue}
          </Typography>
          <Box>
            <Button onClick={onClose} sx={{ mr: 2 }}>
              Cancel
            </Button>
            <Button
              variant="contained"
              onClick={handleCreateClass}
              disabled={!selectedClass || selectedCourses.length === 0}
              sx={{ bgcolor: '#38aae1', '&:hover': { bgcolor: '#2e8bc4' } }}
            >
              Create Class
            </Button>
          </Box>
        </Box>
      </DialogActions>
    </Dialog>
  );
}