import React, { useState } from 'react';
import {
  Box,
  Typography,
  Button,
  Card,
  CardContent,
  Grid,
  Chip,
  Avatar,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  IconButton,
  Menu,
  MenuItem,
} from '@mui/material';
import {
  Add,
  MoreVert,
  Edit,
  Delete,
  Visibility,
  BookmarkBorder,
  People,
  Schedule,
} from '@mui/icons-material';
import { AddCourseDialog } from './AddCourseDialog';

interface Course {
  id: string;
  name: string;
  category: string;
  sessions: number;
  students: number;
  status: 'active' | 'inactive';
  image?: string;
  description: string;
  createdAt: string;
}

export const Courses: React.FC = () => {
  const [openAddCourse, setOpenAddCourse] = useState(false);
  const [anchorEl, setAnchorEl] = useState<null | HTMLElement>(null);
  const [selectedItem, setSelectedItem] = useState<string | null>(null);

  // Mock data for courses
  const [courses, setCourses] = useState<Course[]>([
    {
      id: '1',
      name: 'Introduction to Python Programming',
      category: 'Programming',
      sessions: 12,
      students: 45,
      status: 'active',
      description: 'Learn Python fundamentals including variables, loops, functions, and basic data structures',
      createdAt: '2024-01-15',
    },
    {
      id: '2',
      name: 'Web Development with HTML & CSS',
      category: 'Web Development',
      sessions: 10,
      students: 38,
      status: 'active',
      description: 'Master the basics of web development using HTML5 and CSS3',
      createdAt: '2024-01-12',
    },
    {
      id: '3',
      name: 'JavaScript Fundamentals',
      category: 'Programming',
      sessions: 14,
      students: 32,
      status: 'active',
      description: 'Learn JavaScript programming from basics to advanced concepts',
      createdAt: '2024-01-10',
    },
    {
      id: '4',
      name: 'Data Structures & Algorithms',
      category: 'Computer Science',
      sessions: 16,
      students: 28,
      status: 'active',
      description: 'Understand fundamental data structures and algorithms',
      createdAt: '2024-01-08',
    },
    {
      id: '5',
      name: 'Mobile App Development',
      category: 'Mobile Development',
      sessions: 18,
      students: 24,
      status: 'active',
      description: 'Build mobile applications using modern frameworks',
      createdAt: '2024-01-05',
    },
    {
      id: '6',
      name: 'Database Design Fundamentals',
      category: 'Database',
      sessions: 8,
      students: 22,
      status: 'active',
      description: 'Learn database design principles and SQL basics',
      createdAt: '2024-01-03',
    },
  ]);

  const handleAddCourse = (courseData: any) => {
    const newCourse: Course = {
      id: Date.now().toString(),
      name: courseData.courseName,
      category: courseData.category,
      sessions: parseInt(courseData.numberOfSessions),
      students: 0,
      status: 'active',
      description: courseData.courseDescription,
      createdAt: new Date().toISOString().split('T')[0],
    };
    setCourses(prev => [...prev, newCourse]);
  };

  const handleMenuClick = (event: React.MouseEvent<HTMLElement>, itemId: string) => {
    setAnchorEl(event.currentTarget);
    setSelectedItem(itemId);
  };

  const handleMenuClose = () => {
    setAnchorEl(null);
    setSelectedItem(null);
  };

  const getStatusColor = (status: string) => {
    return status === 'active' ? '#10b981' : '#ef4444';
  };

  const getCategoryColor = (category: string) => {
    const colors: Record<string, { backgroundColor: string; color: string }> = {
      'Programming': { backgroundColor: '#dbeafe', color: '#1d4ed8' },
      'Web Development': { backgroundColor: '#dcfce7', color: '#166534' },
      'Computer Science': { backgroundColor: '#f3e8ff', color: '#7c3aed' },
      'Mobile Development': { backgroundColor: '#fef3c7', color: '#92400e' },
      'Database': { backgroundColor: '#fee2e2', color: '#991b1b' },
    };
    return colors[category] || { backgroundColor: '#e0f2fe', color: '#0284c7' };
  };

  return (
    <Box sx={{ p: 3 }}>
      {/* Header */}
      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 4 }}>
        <Box>
          <Typography variant="h4" sx={{ color: '#25476a', fontWeight: 700, mb: 1 }}>
            Course Management
          </Typography>
          <Typography variant="body1" sx={{ color: '#64748b' }}>
            Manage individual courses that can be assigned to self-paced classes. These courses are the building blocks of your learning programs.
          </Typography>
        </Box>
        <Button
          variant="contained"
          startIcon={<Add />}
          onClick={() => setOpenAddCourse(true)}
          sx={{
            backgroundColor: '#38aae1',
            textTransform: 'none',
            fontWeight: 600,
            '&:hover': {
              backgroundColor: '#2563eb',
            },
          }}
        >
          Add Course
        </Button>
      </Box>

      {/* Statistics Cards */}
      <Grid container spacing={3} sx={{ mb: 4 }}>
        <Grid item xs={12} md={3}>
          <Card sx={{ height: '100%' }}>
            <CardContent>
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                <Avatar sx={{ bgcolor: '#feb139', mr: 2 }}>
                  <BookmarkBorder />
                </Avatar>
                <Box>
                  <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 600 }}>
                    {courses.length}
                  </Typography>
                  <Typography variant="body2" sx={{ color: '#64748b' }}>
                    Total Courses
                  </Typography>
                </Box>
              </Box>
            </CardContent>
          </Card>
        </Grid>
        <Grid item xs={12} md={3}>
          <Card sx={{ height: '100%' }}>
            <CardContent>
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                <Avatar sx={{ bgcolor: '#38aae1', mr: 2 }}>
                  <People />
                </Avatar>
                <Box>
                  <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 600 }}>
                    {courses.reduce((sum, course) => sum + course.students, 0)}
                  </Typography>
                  <Typography variant="body2" sx={{ color: '#64748b' }}>
                    Total Students
                  </Typography>
                </Box>
              </Box>
            </CardContent>
          </Card>
        </Grid>
        <Grid item xs={12} md={3}>
          <Card sx={{ height: '100%' }}>
            <CardContent>
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                <Avatar sx={{ bgcolor: '#25476a', mr: 2 }}>
                  <Schedule />
                </Avatar>
                <Box>
                  <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 600 }}>
                    {courses.reduce((sum, course) => sum + course.sessions, 0)}
                  </Typography>
                  <Typography variant="body2" sx={{ color: '#64748b' }}>
                    Total Sessions
                  </Typography>
                </Box>
              </Box>
            </CardContent>
          </Card>
        </Grid>
        <Grid item xs={12} md={3}>
          <Card sx={{ height: '100%' }}>
            <CardContent>
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                <Avatar sx={{ bgcolor: '#10b981', mr: 2 }}>
                  <BookmarkBorder />
                </Avatar>
                <Box>
                  <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 600 }}>
                    {courses.filter(course => course.status === 'active').length}
                  </Typography>
                  <Typography variant="body2" sx={{ color: '#64748b' }}>
                    Active Courses
                  </Typography>
                </Box>
              </Box>
            </CardContent>
          </Card>
        </Grid>
      </Grid>

      {/* Courses Table */}
      <Card>
        <Box sx={{ p: 3, pb: 0 }}>
          <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 600, mb: 2 }}>
            All Courses
          </Typography>
        </Box>
        <TableContainer>
          <Table>
            <TableHead>
              <TableRow sx={{ backgroundColor: '#f8fafc' }}>
                <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Course Name</TableCell>
                <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Category</TableCell>
                <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Sessions</TableCell>
                <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Students</TableCell>
                <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Status</TableCell>
                <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Actions</TableCell>
              </TableRow>
            </TableHead>
            <TableBody>
              {courses.map((course) => (
                <TableRow key={course.id} hover>
                  <TableCell>
                    <Box>
                      <Typography variant="body1" sx={{ fontWeight: 600, color: '#25476a' }}>
                        {course.name}
                      </Typography>
                      <Typography variant="caption" sx={{ color: '#64748b' }}>
                        {course.description}
                      </Typography>
                    </Box>
                  </TableCell>
                  <TableCell>
                    <Chip
                      label={course.category}
                      size="small"
                      sx={{
                        ...getCategoryColor(course.category),
                        fontWeight: 500,
                      }}
                    />
                  </TableCell>
                  <TableCell>{course.sessions}</TableCell>
                  <TableCell>{course.students}</TableCell>
                  <TableCell>
                    <Chip
                      label={course.status}
                      size="small"
                      sx={{
                        backgroundColor: course.status === 'active' ? '#dcfce7' : '#fee2e2',
                        color: getStatusColor(course.status),
                        fontWeight: 500,
                      }}
                    />
                  </TableCell>
                  <TableCell>
                    <IconButton
                      onClick={(e) => handleMenuClick(e, course.id)}
                      size="small"
                    >
                      <MoreVert />
                    </IconButton>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </TableContainer>
      </Card>

      {/* Context Menu */}
      <Menu
        anchorEl={anchorEl}
        open={Boolean(anchorEl)}
        onClose={handleMenuClose}
      >
        <MenuItem onClick={handleMenuClose}>
          <Visibility sx={{ mr: 1, fontSize: 20 }} />
          View Details
        </MenuItem>
        <MenuItem onClick={handleMenuClose}>
          <Edit sx={{ mr: 1, fontSize: 20 }} />
          Edit
        </MenuItem>
        <MenuItem onClick={handleMenuClose} sx={{ color: '#ef4444' }}>
          <Delete sx={{ mr: 1, fontSize: 20 }} />
          Delete
        </MenuItem>
      </Menu>

      {/* Add Course Dialog */}
      <AddCourseDialog
        open={openAddCourse}
        onClose={() => setOpenAddCourse(false)}
        onSubmit={handleAddCourse}
      />
    </Box>
  );
};