import React, { Component } from 'react';
import {
  Box,
  Typography,
  Button,
  Card,
  CardContent,
  Grid,
  Chip,
  Avatar,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  IconButton,
  Menu,
  MenuItem,
  TextField,
  InputAdornment,
  Tabs,
  Tab,
  Paper,
  LinearProgress,
  List,
  ListItem,
  ListItemText,
  ListItemIcon,
  Accordion,
  AccordionSummary,
  AccordionDetails,
} from '@mui/material';
import {
  ArrowBack,
  School,
  Schedule,
  People,
  TrendingUp,
  MoreVert,
  Edit,
  Delete,
  Visibility,
  Search,
  Star,
  PlayCircleOutline,
  Assignment,
  SportsEsports,
  PictureAsPdf,
  Description,
  Slideshow,
  YouTube,
  ExpandMore,
  CheckCircle,
  Add,
} from '@mui/icons-material';
import { AllSessionsEditor } from './AllSessionsEditor.jsx';
import { SingleSessionEditor } from './SingleSessionEditor.jsx';
import { VidstackPlayer } from './VidstackPlayer.jsx';

export class CourseView extends Component {
  constructor(props) {
    super(props);
    this.state = {
      selectedTab: 0,
      searchTerm: '',
      anchorEl: null,
      selectedItem: null,
      editingLesson: null,
      isAllSessionsEditorOpen: false,
      isSingleSessionEditorOpen: false,
      editingSessionNumber: 1,
      lessonPlans: [
        {
          id: '1',
          title: 'Introduction to Web Development',
          duration: 45,
          description: 'Learn the basics of web development including HTML, CSS, and JavaScript fundamentals.',
          objectives: [
            'Understand HTML structure and semantics',
            'Apply CSS styling and layout techniques',
            'Implement basic JavaScript functionality'
          ],
          introVideo: { title: 'Welcome to Web Development', duration: 10, url: 'https://www.youtube.com/watch?v=7mUJH2zcsWs' },
          assignmentVideo: { title: 'Build Your First Webpage', duration: 15, url: 'https://www.youtube.com/watch?v=7mUJH2zcsWs' },
          activityVideos: [
            { title: 'HTML Basics Exercise', duration: 20, url: 'https://www.youtube.com/watch?v=7mUJH2zcsWs' },
            { title: 'CSS Styling Practice', duration: 25, url: 'https://www.youtube.com/watch?v=7mUJH2zcsWs' }
          ],
          resources: [
            { name: 'HTML Reference Guide', type: 'pdf', url: 'https://example.com/html-guide.pdf' },
            { name: 'CSS Cheat Sheet', type: 'pdf', url: 'https://example.com/css-cheat-sheet.pdf' }
          ]
        },
        {
          id: '2',
          title: 'Advanced JavaScript Concepts',
          duration: 60,
          description: 'Dive deep into JavaScript concepts including closures, promises, and async/await.',
          objectives: [
            'Master JavaScript closures and scope',
            'Understand asynchronous programming',
            'Implement error handling strategies'
          ],
          introVideo: { title: 'Advanced JS Overview', duration: 12, url: 'https://www.youtube.com/watch?v=7mUJH2zcsWs' },
          assignmentVideo: { title: 'Async Programming Challenge', duration: 20, url: 'https://www.youtube.com/watch?v=7mUJH2zcsWs' },
          activityVideos: [
            { title: 'Closure Examples', duration: 15, url: 'https://www.youtube.com/watch?v=7mUJH2zcsWs' },
            { title: 'Promise Chain Practice', duration: 18, url: 'https://www.youtube.com/watch?v=7mUJH2zcsWs' },
            { title: 'Async/Await Workshop', duration: 22, url: 'https://www.youtube.com/watch?v=7mUJH2zcsWs' }
          ],
          resources: [
            { name: 'JavaScript Advanced Guide', type: 'pdf', url: 'https://example.com/js-advanced.pdf' },
            { name: 'Async Programming Tutorial', type: 'youtube', url: 'https://youtube.com/watch?v=example' }
          ]
        },
        {
          id: '3',
          title: 'React Components and Props',
          duration: 50,
          description: 'Master React component architecture and prop management.',
          objectives: [
            'Create functional and class components',
            'Pass and validate props effectively',
            'Implement component composition patterns'
          ],
          introVideo: { title: 'React Components Overview', duration: 8, url: 'https://www.youtube.com/watch?v=7mUJH2zcsWs' },
          assignmentVideo: { title: 'Component Building Challenge', duration: 18, url: 'https://www.youtube.com/watch?v=7mUJH2zcsWs' },
          activityVideos: [
            { title: 'Functional Components', duration: 16, url: 'https://www.youtube.com/watch?v=7mUJH2zcsWs' },
            { title: 'Props and State', duration: 20, url: 'https://www.youtube.com/watch?v=7mUJH2zcsWs' }
          ],
          resources: [
            { name: 'React Documentation', type: 'doc', url: 'https://example.com/react-docs.docx' },
            { name: 'Component Patterns', type: 'ppt', url: 'https://example.com/patterns.pptx' }
          ]
        }
      ]
    };
  }

  handleTabChange = (event, newValue) => {
    this.setState({ selectedTab: newValue });
  };

  handleMenuClick = (event, itemId) => {
    this.setState({
      anchorEl: event.currentTarget,
      selectedItem: itemId
    });
  };

  handleMenuClose = () => {
    this.setState({
      anchorEl: null,
      selectedItem: null
    });
  };

  handleEditAllSessions = (lesson) => {
    this.setState({
      editingLesson: lesson,
      isAllSessionsEditorOpen: true
    });
  };

  handleEditSingleSession = (lesson, sessionNumber = 1) => {
    this.setState({
      editingLesson: lesson,
      editingSessionNumber: sessionNumber,
      isSingleSessionEditorOpen: true
    });
  };

  handleSaveLesson = (updatedLesson) => {
    const existingIndex = this.state.lessonPlans.findIndex(lesson => lesson.id === updatedLesson.id);
    
    if (existingIndex >= 0) {
      // Update existing lesson
      this.setState({
        lessonPlans: this.state.lessonPlans.map(lesson =>
          lesson.id === updatedLesson.id ? updatedLesson : lesson
        ),
        isAllSessionsEditorOpen: false,
        isSingleSessionEditorOpen: false,
        editingLesson: null
      });
    } else {
      // Add new lesson
      this.setState({
        lessonPlans: [...this.state.lessonPlans, updatedLesson],
        isAllSessionsEditorOpen: false,
        isSingleSessionEditorOpen: false,
        editingLesson: null
      });
    }
  };

  handleCloseAllSessionsEditor = () => {
    this.setState({
      isAllSessionsEditorOpen: false,
      editingLesson: null
    });
  };

  handleCloseSingleSessionEditor = () => {
    this.setState({
      isSingleSessionEditorOpen: false,
      editingLesson: null
    });
  };

  handleAddNewLesson = () => {
    const newLesson = {
      id: Date.now().toString(),
      title: 'New Lesson Plan',
      duration: 60,
      description: 'Description for the new lesson plan',
      objectives: ['New objective 1', 'New objective 2'],
      introVideo: { title: 'Introduction Video', duration: 10, url: '' },
      assignmentVideo: { title: 'Assignment Video', duration: 15, url: '' },
      activityVideos: [
        { title: 'Activity Video 1', duration: 20, url: '' }
      ],
      resources: []
    };
    
    // Add the new lesson to the list first
    this.setState({
      lessonPlans: [...this.state.lessonPlans, newLesson],
      editingLesson: newLesson,
      isAllSessionsEditorOpen: true
    });
  };

  getResourceIcon = (type) => {
    switch (type) {
      case 'pdf': return <PictureAsPdf sx={{ color: '#d32f2f' }} />;
      case 'doc': return <Description sx={{ color: '#1976d2' }} />;
      case 'ppt': return <Slideshow sx={{ color: '#ff9800' }} />;
      case 'youtube': return <YouTube sx={{ color: '#f44336' }} />;
      default: return <Description sx={{ color: '#64748b' }} />;
    }
  };

  getLevelColor = (level) => {
    switch (level) {
      case 'Beginner':
        return { backgroundColor: '#dcfce7', color: '#166534' };
      case 'Intermediate':
        return { backgroundColor: '#fef3c7', color: '#92400e' };
      case 'Advanced':
        return { backgroundColor: '#fee2e2', color: '#991b1b' };
      default:
        return { backgroundColor: '#e0f2fe', color: '#0284c7' };
    }
  };

  render() {
    const { course, onBack } = this.props;
    const { 
      selectedTab, 
      searchTerm, 
      anchorEl, 
      editingLesson, 
      isAllSessionsEditorOpen, 
      isSingleSessionEditorOpen,
      editingSessionNumber,
      lessonPlans 
    } = this.state;

    const filteredLessonPlans = lessonPlans.filter(lesson =>
      lesson.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
      lesson.description.toLowerCase().includes(searchTerm.toLowerCase())
    );

    return (
      <Box sx={{ p: 3 }}>
        {/* Header */}
        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 4 }}>
          <Box>
            <Button
              startIcon={<ArrowBack />}
              onClick={onBack}
              sx={{ 
                mb: 2,
                color: '#64748b',
                textTransform: 'none',
                '&:hover': { bgcolor: 'rgba(100, 116, 139, 0.1)' }
              }}
            >
              Back to Class
            </Button>
            <Typography variant="h4" sx={{ color: '#25476a', fontWeight: 700, mb: 1 }}>
              {course.title}
            </Typography>
            <Typography variant="body1" sx={{ color: '#64748b' }}>
              Manage lesson plans, track student progress, and monitor course analytics
            </Typography>
          </Box>
          <Button
            variant="contained"
            startIcon={<Add />}
            onClick={this.handleAddNewLesson}
            sx={{
              backgroundColor: '#38aae1',
              textTransform: 'none',
              fontWeight: 600,
              '&:hover': {
                backgroundColor: '#2563eb',
              },
            }}
          >
            Add Lesson Plan
          </Button>
        </Box>

        {/* Statistics Cards */}
        <Grid container spacing={3} sx={{ mb: 4 }}>
          <Grid item xs={12} md={3}>
            <Card sx={{ height: '100%' }}>
              <CardContent>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                  <Avatar sx={{ bgcolor: '#25476a', mr: 2 }}>
                    <People />
                  </Avatar>
                  <Box>
                    <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {course.students.toLocaleString()}
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#64748b' }}>
                      Enrolled Students
                    </Typography>
                  </Box>
                </Box>
              </CardContent>
            </Card>
          </Grid>
          <Grid item xs={12} md={3}>
            <Card sx={{ height: '100%' }}>
              <CardContent>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                  <Avatar sx={{ bgcolor: '#38aae1', mr: 2 }}>
                    <CheckCircle />
                  </Avatar>
                  <Box>
                    <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {course.completionRate}%
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#64748b' }}>
                      Completion Rate
                    </Typography>
                  </Box>
                </Box>
              </CardContent>
            </Card>
          </Grid>
          <Grid item xs={12} md={3}>
            <Card sx={{ height: '100%' }}>
              <CardContent>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                  <Avatar sx={{ bgcolor: '#feb139', mr: 2 }}>
                    <Star />
                  </Avatar>
                  <Box>
                    <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {course.rating}/5
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#64748b' }}>
                      Average Rating
                    </Typography>
                  </Box>
                </Box>
              </CardContent>
            </Card>
          </Grid>
          <Grid item xs={12} md={3}>
            <Card sx={{ height: '100%' }}>
              <CardContent>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                  <Avatar sx={{ bgcolor: '#10b981', mr: 2 }}>
                    <School />
                  </Avatar>
                  <Box>
                    <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {lessonPlans.length}
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#64748b' }}>
                      Lesson Plans
                    </Typography>
                  </Box>
                </Box>
              </CardContent>
            </Card>
          </Grid>
        </Grid>

        {/* Tabs */}
        <Paper sx={{ borderRadius: 3, boxShadow: '0 4px 20px rgba(0,0,0,0.08)', overflow: 'hidden' }}>
          <Tabs 
            value={selectedTab} 
            onChange={this.handleTabChange}
            sx={{ 
              borderBottom: '1px solid #e2e8f0',
              '& .MuiTab-root': {
                textTransform: 'none',
                fontWeight: 600,
                fontSize: '1rem',
                color: '#64748b',
                '&.Mui-selected': {
                  color: '#38aae1',
                }
              }
            }}
          >
            <Tab label="Lesson Plans" />
            <Tab label="Students" />
            <Tab label="Analytics" />
            <Tab label="Settings" />
          </Tabs>

          {/* Tab Content */}
          <Box sx={{ p: selectedTab === 0 ? 2 : 4 }}>
            {/* Lesson Plans Tab */}
            {selectedTab === 0 && (
              <Box>
                <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 4 }}>
                  <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                    Lesson Plans ({filteredLessonPlans.length})
                  </Typography>
                  <Box sx={{ display: 'flex', gap: 2, alignItems: 'center' }}>
                    <Button
                      variant="outlined"
                      startIcon={<Edit />}
                      onClick={() => filteredLessonPlans.length > 0 && this.handleEditAllSessions(filteredLessonPlans[0])}
                      disabled={filteredLessonPlans.length === 0}
                      sx={{
                        borderColor: '#38aae1',
                        color: '#38aae1',
                        textTransform: 'none',
                        fontWeight: 600,
                        '&:hover': {
                          backgroundColor: 'rgba(56, 170, 225, 0.1)',
                          borderColor: '#38aae1',
                        },
                      }}
                    >
                      Edit All Sessions
                    </Button>
                    <TextField
                      placeholder="Search lesson plans..."
                      value={searchTerm}
                      onChange={(e) => this.setState({ searchTerm: e.target.value })}
                      sx={{ width: 300 }}
                      size="small"
                      InputProps={{
                        startAdornment: (
                          <InputAdornment position="start">
                            <Search sx={{ color: '#64748b' }} />
                          </InputAdornment>
                        ),
                      }}
                    />
                  </Box>
                </Box>

                {/* Lesson Plans Table */}
                <Card>
                  <TableContainer>
                    <Table>
                      <TableHead>
                        <TableRow sx={{ backgroundColor: '#f8fafc' }}>
                          <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Lesson Plan</TableCell>
                          <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Duration</TableCell>
                          <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Videos</TableCell>
                          <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Resources</TableCell>
                          <TableCell sx={{ fontWeight: 600, color: '#25476a' }}>Actions</TableCell>
                        </TableRow>
                      </TableHead>
                      <TableBody>
                        {filteredLessonPlans.map((lesson, index) => (
                          <TableRow key={lesson.id} hover>
                            <TableCell>
                              <Box sx={{ display: 'flex', alignItems: 'center' }}>
                                <Box sx={{ 
                                  width: 32, 
                                  height: 32, 
                                  borderRadius: '50%', 
                                  bgcolor: '#38aae1', 
                                  display: 'flex', 
                                  alignItems: 'center', 
                                  justifyContent: 'center',
                                  mr: 2
                                }}>
                                  <Typography variant="body2" sx={{ color: 'white', fontWeight: 600 }}>
                                    {index + 1}
                                  </Typography>
                                </Box>
                                <Box>
                                  <Typography variant="body1" sx={{ fontWeight: 600, color: '#25476a' }}>
                                    {lesson.title}
                                  </Typography>
                                  <Typography variant="caption" sx={{ color: '#64748b' }}>
                                    {lesson.description.length > 60 
                                      ? `${lesson.description.substring(0, 60)}...` 
                                      : lesson.description}
                                  </Typography>
                                </Box>
                              </Box>
                            </TableCell>
                            <TableCell>
                              <Chip
                                icon={<Schedule />}
                                label={`${lesson.duration} min`}
                                size="small"
                                sx={{ bgcolor: 'rgba(56, 170, 225, 0.1)', color: '#38aae1' }}
                              />
                            </TableCell>
                            <TableCell>
                              <Typography variant="body2" sx={{ color: '#25476a' }}>
                                {1 + 1 + lesson.activityVideos.length} videos
                              </Typography>
                            </TableCell>
                            <TableCell>
                              <Typography variant="body2" sx={{ color: '#25476a' }}>
                                {lesson.resources.length} resources
                              </Typography>
                            </TableCell>
                            <TableCell>
                              <Box sx={{ display: 'flex', gap: 1 }}>
                                <IconButton
                                  onClick={() => this.handleEditSingleSession(lesson, index + 1)}
                                  size="small"
                                  sx={{ color: '#38aae1' }}
                                  title="Edit this session"
                                >
                                  <Edit />
                                </IconButton>
                                <IconButton
                                  onClick={(e) => this.handleMenuClick(e, lesson.id)}
                                  size="small"
                                >
                                  <MoreVert />
                                </IconButton>
                              </Box>
                            </TableCell>
                          </TableRow>
                        ))}
                      </TableBody>
                    </Table>
                  </TableContainer>
                </Card>

                {/* Lesson Plan Details */}
                <Box sx={{ mt: 4 }}>
                  <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a', mb: 3 }}>
                    Lesson Plan Details
                  </Typography>
                  {filteredLessonPlans.map((lesson, index) => (
                    <Accordion key={lesson.id} sx={{ mb: 2 }}>
                      <AccordionSummary expandIcon={<ExpandMore />}>
                        <Box sx={{ display: 'flex', alignItems: 'center', width: '100%' }}>
                          <Box sx={{ 
                            width: 24, 
                            height: 24, 
                            borderRadius: '50%', 
                            bgcolor: '#38aae1', 
                            display: 'flex', 
                            alignItems: 'center', 
                            justifyContent: 'center',
                            mr: 2
                          }}>
                            <Typography variant="caption" sx={{ color: 'white', fontWeight: 600 }}>
                              {index + 1}
                            </Typography>
                          </Box>
                          <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                            {lesson.title}
                          </Typography>
                        </Box>
                      </AccordionSummary>
                      <AccordionDetails>
                        <Grid container spacing={3}>
                          {/* Learning Objectives */}
                          <Grid item xs={12}>
                            <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a', mb: 2 }}>
                              Learning Objectives
                            </Typography>
                            <List dense>
                              {lesson.objectives.map((objective, idx) => (
                                <ListItem key={idx}>
                                  <ListItemIcon>
                                    <CheckCircle sx={{ color: '#4caf50', fontSize: 20 }} />
                                  </ListItemIcon>
                                  <ListItemText 
                                    primary={objective}
                                    sx={{ '& .MuiListItemText-primary': { fontSize: '0.9rem' } }}
                                  />
                                </ListItem>
                              ))}
                            </List>
                          </Grid>

                          {/* Videos */}
                          <Grid item xs={12}>
                            <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a', mb: 2 }}>
                              Videos
                            </Typography>
                            <Grid container spacing={3}>
                              {/* Intro Video */}
                              <Grid item xs={12} md={6} lg={4}>
                                <Box sx={{ border: '1px solid #e2e8f0', borderRadius: 2, p: 2 }}>
                                  <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                                    <PlayCircleOutline sx={{ color: '#38aae1', mr: 1 }} />
                                    <Typography variant="body2" sx={{ fontWeight: 600, color: '#25476a' }}>
                                      {lesson.introVideo.title}
                                    </Typography>
                                  </Box>
                                  <Typography variant="caption" sx={{ color: '#64748b', display: 'block', mb: 2 }}>
                                    Duration: {lesson.introVideo.duration} minutes
                                  </Typography>
                                  {lesson.introVideo.url && (
                                    <VidstackPlayer
                                      src={lesson.introVideo.url}
                                      title={lesson.introVideo.title}
                                      aspectRatio="16/9"
                                    />
                                  )}
                                  {!lesson.introVideo.url && (
                                    <Box sx={{ 
                                      width: '100%', 
                                      aspectRatio: '16/9', 
                                      bgcolor: '#f8fafc', 
                                      display: 'flex', 
                                      alignItems: 'center', 
                                      justifyContent: 'center',
                                      borderRadius: 1,
                                      border: '2px dashed #cbd5e1'
                                    }}>
                                      <Typography variant="caption" sx={{ color: '#94a3b8' }}>
                                        No video uploaded
                                      </Typography>
                                    </Box>
                                  )}
                                </Box>
                              </Grid>

                              {/* Assignment Video */}
                              <Grid item xs={12} md={6} lg={4}>
                                <Box sx={{ border: '1px solid #e2e8f0', borderRadius: 2, p: 2 }}>
                                  <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                                    <Assignment sx={{ color: '#ff9800', mr: 1 }} />
                                    <Typography variant="body2" sx={{ fontWeight: 600, color: '#25476a' }}>
                                      {lesson.assignmentVideo.title}
                                    </Typography>
                                  </Box>
                                  <Typography variant="caption" sx={{ color: '#64748b', display: 'block', mb: 2 }}>
                                    Duration: {lesson.assignmentVideo.duration} minutes
                                  </Typography>
                                  {lesson.assignmentVideo.url && (
                                    <VidstackPlayer
                                      src={lesson.assignmentVideo.url}
                                      title={lesson.assignmentVideo.title}
                                      aspectRatio="16/9"
                                    />
                                  )}
                                  {!lesson.assignmentVideo.url && (
                                    <Box sx={{ 
                                      width: '100%', 
                                      aspectRatio: '16/9', 
                                      bgcolor: '#f8fafc', 
                                      display: 'flex', 
                                      alignItems: 'center', 
                                      justifyContent: 'center',
                                      borderRadius: 1,
                                      border: '2px dashed #cbd5e1'
                                    }}>
                                      <Typography variant="caption" sx={{ color: '#94a3b8' }}>
                                        No video uploaded
                                      </Typography>
                                    </Box>
                                  )}
                                </Box>
                              </Grid>

                              {/* Activity Videos */}
                              {lesson.activityVideos.map((video, idx) => (
                                <Grid item xs={12} md={6} lg={4} key={idx}>
                                  <Box sx={{ border: '1px solid #e2e8f0', borderRadius: 2, p: 2 }}>
                                    <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                                      <SportsEsports sx={{ color: '#9c27b0', mr: 1 }} />
                                      <Typography variant="body2" sx={{ fontWeight: 600, color: '#25476a' }}>
                                        {video.title}
                                      </Typography>
                                    </Box>
                                    <Typography variant="caption" sx={{ color: '#64748b', display: 'block', mb: 2 }}>
                                      Duration: {video.duration} minutes
                                    </Typography>
                                    {video.url && (
                                      <VidstackPlayer
                                        src={video.url}
                                        title={video.title}
                                        aspectRatio="16/9"
                                      />
                                    )}
                                    {!video.url && (
                                      <Box sx={{ 
                                        width: '100%', 
                                        aspectRatio: '16/9', 
                                        bgcolor: '#f8fafc', 
                                        display: 'flex', 
                                        alignItems: 'center', 
                                        justifyContent: 'center',
                                        borderRadius: 1,
                                        border: '2px dashed #cbd5e1'
                                      }}>
                                        <Typography variant="caption" sx={{ color: '#94a3b8' }}>
                                          No video uploaded
                                        </Typography>
                                      </Box>
                                    )}
                                  </Box>
                                </Grid>
                              ))}
                            </Grid>
                          </Grid>

                          {/* Resources */}
                          <Grid item xs={12}>
                            <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a', mb: 2 }}>
                              Resources
                            </Typography>
                            <Grid container spacing={2}>
                              {lesson.resources.map((resource, idx) => (
                                <Grid item xs={12} sm={6} md={4} key={idx}>
                                  <Card sx={{ p: 2, border: '1px solid #e2e8f0' }}>
                                    <Box sx={{ display: 'flex', alignItems: 'center' }}>
                                      {this.getResourceIcon(resource.type)}
                                      <Box sx={{ ml: 2 }}>
                                        <Typography variant="body2" sx={{ fontWeight: 600, color: '#25476a' }}>
                                          {resource.name}
                                        </Typography>
                                        <Typography variant="caption" sx={{ color: '#64748b', textTransform: 'uppercase' }}>
                                          {resource.type}
                                        </Typography>
                                      </Box>
                                    </Box>
                                  </Card>
                                </Grid>
                              ))}
                            </Grid>
                          </Grid>
                        </Grid>
                      </AccordionDetails>
                    </Accordion>
                  ))}
                </Box>
              </Box>
            )}

            {/* Students Tab */}
            {selectedTab === 1 && (
              <Box>
                <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a', mb: 4 }}>
                  Course Students
                </Typography>
                <Typography variant="body1" sx={{ color: '#64748b' }}>
                  Student progress tracking and management functionality will be implemented here.
                </Typography>
              </Box>
            )}

            {/* Analytics Tab */}
            {selectedTab === 2 && (
              <Box>
                <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a', mb: 4 }}>
                  Course Analytics
                </Typography>
                <Typography variant="body1" sx={{ color: '#64748b' }}>
                  Detailed course analytics and performance metrics will be implemented here.
                </Typography>
              </Box>
            )}

            {/* Settings Tab */}
            {selectedTab === 3 && (
              <Box>
                <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a', mb: 4 }}>
                  Course Settings
                </Typography>
                <Typography variant="body1" sx={{ color: '#64748b' }}>
                  Course configuration and settings will be implemented here.
                </Typography>
              </Box>
            )}
          </Box>
        </Paper>

        {/* Context Menu */}
        <Menu
          anchorEl={anchorEl}
          open={Boolean(anchorEl)}
          onClose={this.handleMenuClose}
        >
          <MenuItem onClick={() => {
            const selectedLesson = lessonPlans.find(l => l.id === this.state.selectedItem);
            if (selectedLesson) {
              const lessonIndex = lessonPlans.findIndex(l => l.id === this.state.selectedItem);
              this.handleEditSingleSession(selectedLesson, lessonIndex + 1);
            }
            this.handleMenuClose();
          }}>
            <Edit sx={{ mr: 1, fontSize: 20 }} />
            Edit Session
          </MenuItem>
          <MenuItem onClick={() => {
            const selectedLesson = lessonPlans.find(l => l.id === this.state.selectedItem);
            if (selectedLesson) {
              this.handleEditAllSessions(selectedLesson);
            }
            this.handleMenuClose();
          }}>
            <Edit sx={{ mr: 1, fontSize: 20 }} />
            Edit All Sessions
          </MenuItem>
          <MenuItem onClick={this.handleMenuClose}>
            <Visibility sx={{ mr: 1, fontSize: 20 }} />
            Preview Lesson
          </MenuItem>
          <MenuItem onClick={this.handleMenuClose} sx={{ color: '#ef4444' }}>
            <Delete sx={{ mr: 1, fontSize: 20 }} />
            Delete Lesson
          </MenuItem>
        </Menu>

        {/* All Sessions Editor Modal */}
        {editingLesson && (
          <AllSessionsEditor
            open={isAllSessionsEditorOpen}
            lesson={editingLesson}
            onSave={this.handleSaveLesson}
            onClose={this.handleCloseAllSessionsEditor}
          />
        )}

        {/* Single Session Editor Modal */}
        {editingLesson && (
          <SingleSessionEditor
            open={isSingleSessionEditorOpen}
            lesson={editingLesson}
            sessionNumber={editingSessionNumber}
            onSave={this.handleSaveLesson}
            onClose={this.handleCloseSingleSessionEditor}
          />
        )}
      </Box>
    );
  }
}