import React, { useState } from 'react';
import {
  Box,
  Typography,
  Grid,
  Card,
  CardContent,
  Paper,
  Select,
  MenuItem,
  FormControl,
  InputLabel,
  Chip,
  LinearProgress,
} from '@mui/material';
import {
  PieChart,
  Pie,
  Cell,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  LineChart,
  Line,
  Area,
  AreaChart,
} from 'recharts@2.15.2';
import {
  TrendingUp,
  TrendingDown,
  AccessTime,
  School,
  Group,
  CheckCircle,
} from '@mui/icons-material';

const genderData = [
  { name: 'Female', value: 58, color: '#38aae1' },
  { name: 'Male', value: 42, color: '#feb139' },
];

const completionData = [
  { month: 'Jan', completed: 85, dropped: 15 },
  { month: 'Feb', completed: 78, dropped: 22 },
  { month: 'Mar', completed: 92, dropped: 8 },
  { month: 'Apr', completed: 88, dropped: 12 },
  { month: 'May', completed: 94, dropped: 6 },
  { month: 'Jun', completed: 89, dropped: 11 },
];

const timeToTaskData = [
  { course: 'Mathematics', avgTime: 45 },
  { course: 'Science', avgTime: 38 },
  { course: 'English', avgTime: 52 },
  { course: 'History', avgTime: 41 },
  { course: 'Geography', avgTime: 35 },
];

const mentorHoursData = [
  { week: 'Week 1', hours: 124 },
  { week: 'Week 2', hours: 142 },
  { week: 'Week 3', hours: 118 },
  { week: 'Week 4', hours: 156 },
  { week: 'Week 5', hours: 139 },
  { week: 'Week 6', hours: 147 },
];

export function CourseAnalytics() {
  const [selectedCourse, setSelectedCourse] = useState('all');
  const [selectedTimeframe, setSelectedTimeframe] = useState('month');

  const analyticsCards = [
    {
      title: 'Total Enrollments',
      value: '1,247',
      change: '+12.5%',
      trend: 'up',
      icon: <Group />,
      color: '#38aae1',
    },
    {
      title: 'Completion Rate',
      value: '89.2%',
      change: '+3.2%',
      trend: 'up',
      icon: <CheckCircle />,
      color: '#4caf50',
    },
    {
      title: 'Drop-off Rate',
      value: '10.8%',
      change: '-2.1%',
      trend: 'down',
      icon: <TrendingDown />,
      color: '#f44336',
    },
    {
      title: 'Avg. Time to Complete',
      value: '42 min',
      change: '-5.3%',
      trend: 'down',
      icon: <AccessTime />,
      color: '#feb139',
    },
    {
      title: 'Pass Rate',
      value: '94.7%',
      change: '+1.8%',
      trend: 'up',
      icon: <School />,
      color: '#9c27b0',
    },
    {
      title: 'Mentor Hours Booked',
      value: '1,156 hrs',
      change: '+8.4%',
      trend: 'up',
      icon: <TrendingUp />,
      color: '#ff9800',
    },
  ];

  return (
    <Box sx={{ p: 3 }}>
      {/* Header */}
      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
        <Box>
          <Typography variant="h4" sx={{ fontWeight: 600, mb: 1 }}>
            Course Analytics
          </Typography>
          <Typography variant="body2" color="text.secondary">
            Comprehensive analytics for self-paced courses and student performance.
          </Typography>
        </Box>
        <Box sx={{ display: 'flex', gap: 2 }}>
          <FormControl size="small" sx={{ minWidth: 150 }}>
            <InputLabel>Course</InputLabel>
            <Select
              value={selectedCourse}
              onChange={(e) => setSelectedCourse(e.target.value)}
              label="Course"
            >
              <MenuItem value="all">All Courses</MenuItem>
              <MenuItem value="math">Mathematics</MenuItem>
              <MenuItem value="science">Science</MenuItem>
              <MenuItem value="english">English</MenuItem>
            </Select>
          </FormControl>
          <FormControl size="small" sx={{ minWidth: 120 }}>
            <InputLabel>Timeframe</InputLabel>
            <Select
              value={selectedTimeframe}
              onChange={(e) => setSelectedTimeframe(e.target.value)}
              label="Timeframe"
            >
              <MenuItem value="week">This Week</MenuItem>
              <MenuItem value="month">This Month</MenuItem>
              <MenuItem value="quarter">This Quarter</MenuItem>
              <MenuItem value="year">This Year</MenuItem>
            </Select>
          </FormControl>
        </Box>
      </Box>

      {/* Analytics Cards */}
      <Grid container spacing={3} sx={{ mb: 4 }}>
        {analyticsCards.map((card, index) => (
          <Grid item xs={12} sm={6} md={4} lg={2} key={index}>
            <Card sx={{ height: '100%' }}>
              <CardContent>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                  <Box
                    sx={{
                      display: 'flex',
                      alignItems: 'center',
                      justifyContent: 'center',
                      width: 40,
                      height: 40,
                      borderRadius: 1,
                      bgcolor: `${card.color}20`,
                      color: card.color,
                      mr: 2,
                    }}
                  >
                    {card.icon}
                  </Box>
                  <Typography variant="body2" color="text.secondary" sx={{ flexGrow: 1 }}>
                    {card.title}
                  </Typography>
                </Box>
                <Typography variant="h5" sx={{ fontWeight: 'bold', mb: 1 }}>
                  {card.value}
                </Typography>
                <Box sx={{ display: 'flex', alignItems: 'center' }}>
                  <Chip
                    size="small"
                    label={card.change}
                    color={card.trend === 'up' ? 'success' : 'error'}
                    variant="outlined"
                  />
                </Box>
              </CardContent>
            </Card>
          </Grid>
        ))}
      </Grid>

      {/* Charts Grid */}
      <Grid container spacing={3}>
        {/* Gender Distribution */}
        <Grid item xs={12} md={6}>
          <Paper sx={{ p: 3, height: 400 }}>
            <Typography variant="h6" sx={{ mb: 2, fontWeight: 600 }}>
              Gender Distribution
            </Typography>
            <ResponsiveContainer width="100%" height="100%">
              <PieChart>
                <Pie
                  data={genderData}
                  cx="50%"
                  cy="50%"
                  outerRadius={120}
                  dataKey="value"
                  label={({ name, value }) => `${name}: ${value}%`}
                >
                  {genderData.map((entry, index) => (
                    <Cell key={`cell-${index}`} fill={entry.color} />
                  ))}
                </Pie>
                <Tooltip />
              </PieChart>
            </ResponsiveContainer>
          </Paper>
        </Grid>

        {/* Completion vs Drop-off Rate */}
        <Grid item xs={12} md={6}>
          <Paper sx={{ p: 3, height: 400 }}>
            <Typography variant="h6" sx={{ mb: 2, fontWeight: 600 }}>
              Completion vs Drop-off Rate
            </Typography>
            <ResponsiveContainer width="100%" height="100%">
              <AreaChart data={completionData}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="month" />
                <YAxis />
                <Tooltip />
                <Area
                  type="monotone"
                  dataKey="completed"
                  stackId="1"
                  stroke="#4caf50"
                  fill="#4caf50"
                  fillOpacity={0.6}
                  name="Completed (%)"
                />
                <Area
                  type="monotone"
                  dataKey="dropped"
                  stackId="1"
                  stroke="#f44336"
                  fill="#f44336"
                  fillOpacity={0.6}
                  name="Dropped (%)"
                />
              </AreaChart>
            </ResponsiveContainer>
          </Paper>
        </Grid>

        {/* Average Time to Task */}
        <Grid item xs={12} md={6}>
          <Paper sx={{ p: 3, height: 400 }}>
            <Typography variant="h6" sx={{ mb: 2, fontWeight: 600 }}>
              Average Time to Task (minutes)
            </Typography>
            <ResponsiveContainer width="100%" height="100%">
              <BarChart data={timeToTaskData} layout="horizontal">
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis type="number" />
                <YAxis dataKey="course" type="category" width={80} />
                <Tooltip />
                <Bar dataKey="avgTime" fill="#38aae1" radius={[0, 4, 4, 0]} />
              </BarChart>
            </ResponsiveContainer>
          </Paper>
        </Grid>

        {/* Mentor Hours Booked */}
        <Grid item xs={12} md={6}>
          <Paper sx={{ p: 3, height: 400 }}>
            <Typography variant="h6" sx={{ mb: 2, fontWeight: 600 }}>
              Mentor Hours Booked (Weekly)
            </Typography>
            <ResponsiveContainer width="100%" height="100%">
              <LineChart data={mentorHoursData}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="week" />
                <YAxis />
                <Tooltip />
                <Line
                  type="monotone"
                  dataKey="hours"
                  stroke="#feb139"
                  strokeWidth={3}
                  dot={{ fill: '#feb139', strokeWidth: 2, r: 6 }}
                />
              </LineChart>
            </ResponsiveContainer>
          </Paper>
        </Grid>

        {/* Course Performance Overview */}
        <Grid item xs={12}>
          <Paper sx={{ p: 3 }}>
            <Typography variant="h6" sx={{ mb: 3, fontWeight: 600 }}>
              Course Performance Overview
            </Typography>
            <Grid container spacing={3}>
              {[
                { name: 'Grade 3 Mathematics', students: 145, completion: 94, rating: 4.8 },
                { name: 'Grade 4 Science', students: 132, completion: 87, rating: 4.6 },
                { name: 'Grade 5 English', students: 98, completion: 91, rating: 4.7 },
                { name: 'Grade 6 History', students: 76, completion: 89, rating: 4.5 },
              ].map((course, index) => (
                <Grid item xs={12} sm={6} md={3} key={index}>
                  <Card variant="outlined">
                    <CardContent>
                      <Typography variant="h6" sx={{ mb: 2, fontSize: '1rem' }}>
                        {course.name}
                      </Typography>
                      <Box sx={{ mb: 2 }}>
                        <Typography variant="body2" color="text.secondary" gutterBottom>
                          Students: {course.students}
                        </Typography>
                        <Typography variant="body2" color="text.secondary" gutterBottom>
                          Completion Rate
                        </Typography>
                        <LinearProgress
                          variant="determinate"
                          value={course.completion}
                          sx={{ height: 8, borderRadius: 4, mb: 1 }}
                        />
                        <Typography variant="body2" color="text.secondary">
                          {course.completion}%
                        </Typography>
                      </Box>
                      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                        <Typography variant="body2" color="text.secondary">
                          Rating: ⭐ {course.rating}
                        </Typography>
                      </Box>
                    </CardContent>
                  </Card>
                </Grid>
              ))}
            </Grid>
          </Paper>
        </Grid>
      </Grid>
    </Box>
  );
}