import React, { Component } from 'react';
import {
  Box,
  Typography,
  Button,
  Card,
  CardContent,
  Grid,
  Chip,
  Avatar,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  IconButton,
  Menu,
  MenuItem,
  TextField,
  InputAdornment,
  LinearProgress,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Tabs,
  Tab,
} from '@mui/material';
import {
  Add,
  School,
  Schedule,
  People,
  MoreVert,
  Edit,
  Delete,
  Visibility,
  Search,
  CalendarMonth,
  TrendingUp,
  Class as ClassIcon,
} from '@mui/icons-material';

export class Classes extends Component {
  constructor(props) {
    super(props);
    this.state = {
      anchorEl: null,
      selectedClass: null,
      viewDialogOpen: false,
      deleteDialogOpen: false,
      searchQuery: '',
      selectedTab: 0,
      classes: [
        {
          id: '1',
          name: 'Web Development Bootcamp',
          description: 'Comprehensive full-stack web development program',
          type: 'Live',
          ageRange: '16-25',
          sessions: 48,
          status: 'active',
          students: 24,
          courses: 3,
          mentor: 'Michael Chen',
          schedule: 'Mon, Wed, Fri - 2:00 PM',
          startDate: '2024-01-15',
          endDate: '2024-06-30',
          progress: 65,
        },
        {
          id: '2',
          name: 'Data Science Track',
          description: 'Learn Python, machine learning, and data analysis',
          type: 'Hybrid',
          ageRange: '18-30',
          sessions: 36,
          status: 'active',
          students: 18,
          courses: 2,
          mentor: 'Emily Rodriguez',
          schedule: 'Tue, Thu - 3:00 PM',
          startDate: '2024-02-01',
          endDate: '2024-07-15',
          progress: 52,
        },
        {
          id: '3',
          name: 'UI/UX Design Fundamentals',
          description: 'Master design principles and tools',
          type: 'Self-Paced',
          ageRange: '16-28',
          sessions: 24,
          status: 'active',
          students: 32,
          courses: 1,
          mentor: 'James Wilson',
          schedule: 'Self-Paced',
          startDate: '2024-01-20',
          endDate: '2024-05-20',
          progress: 78,
        },
        {
          id: '4',
          name: 'Mobile App Development',
          description: 'Build iOS and Android applications',
          type: 'Live',
          ageRange: '18-30',
          sessions: 40,
          status: 'active',
          students: 20,
          courses: 2,
          mentor: 'Sarah Kim',
          schedule: 'Mon, Wed - 4:00 PM',
          startDate: '2024-02-10',
          endDate: '2024-07-30',
          progress: 42,
        },
        {
          id: '5',
          name: 'Digital Marketing Essentials',
          description: 'Social media, SEO, and content marketing',
          type: 'Hybrid',
          ageRange: '20-35',
          sessions: 30,
          status: 'completed',
          students: 28,
          courses: 2,
          mentor: 'Alex Johnson',
          schedule: 'Fri - 6:00 PM',
          startDate: '2023-10-01',
          endDate: '2024-01-31',
          progress: 100,
        },
        {
          id: '6',
          name: 'Graphic Design Mastery',
          description: 'Adobe Suite and visual design principles',
          type: 'Self-Paced',
          ageRange: '16-26',
          sessions: 20,
          status: 'inactive',
          students: 0,
          courses: 1,
          mentor: 'Lisa Park',
          schedule: 'Self-Paced',
          startDate: '2024-05-01',
          endDate: '2024-08-31',
          progress: 0,
        },
      ],
    };
  }

  handleMenuClick = (event, classData) => {
    this.setState({ anchorEl: event.currentTarget, selectedClass: classData });
  };

  handleMenuClose = () => {
    this.setState({ anchorEl: null });
  };

  handleView = () => {
    this.setState({ viewDialogOpen: true, anchorEl: null });
  };

  handleCloseViewDialog = () => {
    this.setState({ viewDialogOpen: false });
  };

  handleDelete = () => {
    this.setState({ deleteDialogOpen: true, anchorEl: null });
  };

  handleConfirmDelete = () => {
    const { selectedClass, classes } = this.state;
    if (selectedClass) {
      const updatedClasses = classes.filter(c => c.id !== selectedClass.id);
      this.setState({ classes: updatedClasses, deleteDialogOpen: false, selectedClass: null });
    }
  };

  handleCancelDelete = () => {
    this.setState({ deleteDialogOpen: false });
  };

  handleSearchChange = (event) => {
    this.setState({ searchQuery: event.target.value });
  };

  handleTabChange = (event, newValue) => {
    this.setState({ selectedTab: newValue });
  };

  getStatusColor = (status) => {
    switch (status) {
      case 'active':
        return { bg: '#dcfce7', color: '#166534' };
      case 'inactive':
        return { bg: '#fee2e2', color: '#991b1b' };
      case 'completed':
        return { bg: '#dbeafe', color: '#1e40af' };
      default:
        return { bg: '#f3f4f6', color: '#374151' };
    }
  };

  getTypeColor = (type) => {
    switch (type) {
      case 'Live':
        return { bg: '#fef3c7', color: '#92400e' };
      case 'Self-Paced':
        return { bg: '#dbeafe', color: '#1e40af' };
      case 'Hybrid':
        return { bg: '#e9d5ff', color: '#6b21a8' };
      default:
        return { bg: '#f3f4f6', color: '#374151' };
    }
  };

  getFilteredClasses = () => {
    const { classes, searchQuery, selectedTab } = this.state;
    let filtered = classes;

    if (selectedTab === 1) {
      filtered = filtered.filter(c => c.status === 'active');
    } else if (selectedTab === 2) {
      filtered = filtered.filter(c => c.status === 'completed');
    } else if (selectedTab === 3) {
      filtered = filtered.filter(c => c.status === 'inactive');
    }

    if (searchQuery) {
      filtered = filtered.filter(c =>
        c.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
        c.description.toLowerCase().includes(searchQuery.toLowerCase()) ||
        c.mentor.toLowerCase().includes(searchQuery.toLowerCase())
      );
    }

    return filtered;
  };

  render() {
    const { anchorEl, selectedClass, viewDialogOpen, deleteDialogOpen, searchQuery, selectedTab, classes } = this.state;
    const filteredClasses = this.getFilteredClasses();

    const totalClasses = classes.length;
    const activeClasses = classes.filter(c => c.status === 'active').length;
    const totalStudents = classes.reduce((sum, c) => sum + c.students, 0);
    const completedClasses = classes.filter(c => c.status === 'completed').length;

    return (
      <Box sx={{ p: 4, bgcolor: '#f8fafc', minHeight: '100vh' }}>
        <Box sx={{ mb: 4 }}>
          <Typography variant="h4" sx={{ color: '#25476a', fontWeight: 700, mb: 1 }}>
            Classes Management
          </Typography>
          <Typography variant="body1" sx={{ color: '#64748b' }}>
            Manage all classes, schedules, and enrollments
          </Typography>
        </Box>

        <Grid container spacing={3} sx={{ mb: 4 }}>
          <Grid item xs={12} sm={6} md={3}>
            <Card
              sx={{
                borderRadius: 3,
                border: '1px solid #e2e8f0',
                boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
                transition: 'all 0.3s',
                '&:hover': {
                  boxShadow: '0 8px 16px rgba(56, 170, 225, 0.15)',
                  transform: 'translateY(-4px)',
                },
              }}
            >
              <CardContent sx={{ p: 3 }}>
                <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                  <Box sx={{ p: 1.5, bgcolor: 'rgba(56, 170, 225, 0.1)', borderRadius: 2 }}>
                    <ClassIcon sx={{ color: '#38aae1', fontSize: 28 }} />
                  </Box>
                </Box>
                <Typography variant="body2" sx={{ color: '#64748b', mb: 1, fontWeight: 500 }}>
                  Total Classes
                </Typography>
                <Typography variant="h3" sx={{ color: '#25476a', fontWeight: 700 }}>
                  {totalClasses}
                </Typography>
              </CardContent>
            </Card>
          </Grid>

          <Grid item xs={12} sm={6} md={3}>
            <Card
              sx={{
                borderRadius: 3,
                border: '1px solid #e2e8f0',
                boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
                transition: 'all 0.3s',
                '&:hover': {
                  boxShadow: '0 8px 16px rgba(34, 197, 94, 0.15)',
                  transform: 'translateY(-4px)',
                },
              }}
            >
              <CardContent sx={{ p: 3 }}>
                <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                  <Box sx={{ p: 1.5, bgcolor: 'rgba(34, 197, 94, 0.1)', borderRadius: 2 }}>
                    <TrendingUp sx={{ color: '#22c55e', fontSize: 28 }} />
                  </Box>
                  <Chip
                    label="Active"
                    size="small"
                    sx={{
                      bgcolor: '#dcfce7',
                      color: '#166534',
                      fontWeight: 600,
                      fontSize: '0.75rem',
                    }}
                  />
                </Box>
                <Typography variant="body2" sx={{ color: '#64748b', mb: 1, fontWeight: 500 }}>
                  Active Classes
                </Typography>
                <Typography variant="h3" sx={{ color: '#25476a', fontWeight: 700 }}>
                  {activeClasses}
                </Typography>
              </CardContent>
            </Card>
          </Grid>

          <Grid item xs={12} sm={6} md={3}>
            <Card
              sx={{
                borderRadius: 3,
                border: '1px solid #e2e8f0',
                boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
                transition: 'all 0.3s',
                '&:hover': {
                  boxShadow: '0 8px 16px rgba(254, 177, 57, 0.15)',
                  transform: 'translateY(-4px)',
                },
              }}
            >
              <CardContent sx={{ p: 3 }}>
                <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                  <Box sx={{ p: 1.5, bgcolor: 'rgba(254, 177, 57, 0.1)', borderRadius: 2 }}>
                    <People sx={{ color: '#feb139', fontSize: 28 }} />
                  </Box>
                </Box>
                <Typography variant="body2" sx={{ color: '#64748b', mb: 1, fontWeight: 500 }}>
                  Total Students
                </Typography>
                <Typography variant="h3" sx={{ color: '#25476a', fontWeight: 700 }}>
                  {totalStudents}
                </Typography>
              </CardContent>
            </Card>
          </Grid>

          <Grid item xs={12} sm={6} md={3}>
            <Card
              sx={{
                borderRadius: 3,
                border: '1px solid #e2e8f0',
                boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
                transition: 'all 0.3s',
                '&:hover': {
                  boxShadow: '0 8px 16px rgba(139, 92, 246, 0.15)',
                  transform: 'translateY(-4px)',
                },
              }}
            >
              <CardContent sx={{ p: 3 }}>
                <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', mb: 2 }}>
                  <Box sx={{ p: 1.5, bgcolor: 'rgba(139, 92, 246, 0.1)', borderRadius: 2 }}>
                    <School sx={{ color: '#8b5cf6', fontSize: 28 }} />
                  </Box>
                  <Chip
                    label="Completed"
                    size="small"
                    sx={{
                      bgcolor: '#dbeafe',
                      color: '#1e40af',
                      fontWeight: 600,
                      fontSize: '0.75rem',
                    }}
                  />
                </Box>
                <Typography variant="body2" sx={{ color: '#64748b', mb: 1, fontWeight: 500 }}>
                  Completed Classes
                </Typography>
                <Typography variant="h3" sx={{ color: '#25476a', fontWeight: 700 }}>
                  {completedClasses}
                </Typography>
              </CardContent>
            </Card>
          </Grid>
        </Grid>

        <Card
          sx={{
            borderRadius: 3,
            border: '1px solid #e2e8f0',
            boxShadow: '0 1px 3px rgba(0, 0, 0, 0.05)',
          }}
        >
          <CardContent sx={{ p: 3 }}>
            <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
              <TextField
                placeholder="Search classes..."
                value={searchQuery}
                onChange={this.handleSearchChange}
                variant="outlined"
                size="small"
                sx={{
                  width: 300,
                  '& .MuiOutlinedInput-root': {
                    borderRadius: 2,
                  },
                }}
                InputProps={{
                  startAdornment: (
                    <InputAdornment position="start">
                      <Search sx={{ color: '#64748b' }} />
                    </InputAdornment>
                  ),
                }}
              />
              <Button
                variant="contained"
                startIcon={<Add />}
                sx={{
                  bgcolor: '#38aae1',
                  color: '#ffffff',
                  textTransform: 'none',
                  borderRadius: 2,
                  px: 3,
                  '&:hover': {
                    bgcolor: '#2c8fc9',
                  },
                }}
              >
                Add New Class
              </Button>
            </Box>

            <Box sx={{ borderBottom: 1, borderColor: '#e2e8f0', mb: 3 }}>
              <Tabs value={selectedTab} onChange={this.handleTabChange}>
                <Tab label="All Classes" />
                <Tab label="Active" />
                <Tab label="Completed" />
                <Tab label="Inactive" />
              </Tabs>
            </Box>

            <TableContainer>
              <Table>
                <TableHead>
                  <TableRow sx={{ bgcolor: '#f8fafc' }}>
                    <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                      Class Name
                    </TableCell>
                    <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                      Type
                    </TableCell>
                    <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                      Mentor
                    </TableCell>
                    <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                      Students
                    </TableCell>
                    <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                      Schedule
                    </TableCell>
                    <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                      Progress
                    </TableCell>
                    <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                      Status
                    </TableCell>
                    <TableCell sx={{ color: '#64748b', fontWeight: 600, border: 'none' }}>
                      Actions
                    </TableCell>
                  </TableRow>
                </TableHead>
                <TableBody>
                  {filteredClasses.map((classData) => {
                    const statusColor = this.getStatusColor(classData.status);
                    const typeColor = this.getTypeColor(classData.type);
                    return (
                      <TableRow key={classData.id} sx={{ '&:hover': { bgcolor: '#f8fafc' } }}>
                        <TableCell sx={{ border: 'none' }}>
                          <Box sx={{ display: 'flex', alignItems: 'center' }}>
                            <Avatar
                              sx={{
                                width: 40,
                                height: 40,
                                bgcolor: '#38aae1',
                                mr: 2,
                                fontSize: '0.875rem',
                                fontWeight: 600,
                              }}
                            >
                              {classData.name.substring(0, 2).toUpperCase()}
                            </Avatar>
                            <Box>
                              <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                                {classData.name}
                              </Typography>
                              <Typography variant="caption" sx={{ color: '#94a3b8' }}>
                                {classData.sessions} sessions • {classData.courses} courses
                              </Typography>
                            </Box>
                          </Box>
                        </TableCell>
                        <TableCell sx={{ border: 'none' }}>
                          <Chip
                            label={classData.type}
                            size="small"
                            sx={{
                              bgcolor: typeColor.bg,
                              color: typeColor.color,
                              fontWeight: 600,
                              fontSize: '0.75rem',
                            }}
                          />
                        </TableCell>
                        <TableCell sx={{ border: 'none', color: '#64748b' }}>
                          {classData.mentor}
                        </TableCell>
                        <TableCell sx={{ border: 'none' }}>
                          <Box sx={{ display: 'flex', alignItems: 'center' }}>
                            <People sx={{ fontSize: 18, color: '#38aae1', mr: 0.5 }} />
                            <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                              {classData.students}
                            </Typography>
                          </Box>
                        </TableCell>
                        <TableCell sx={{ border: 'none', color: '#64748b', fontSize: '0.875rem' }}>
                          {classData.schedule}
                        </TableCell>
                        <TableCell sx={{ border: 'none' }}>
                          <Box sx={{ minWidth: 100 }}>
                            <Box sx={{ display: 'flex', justifyContent: 'space-between', mb: 0.5 }}>
                              <Typography variant="caption" sx={{ color: '#64748b' }}>
                                {classData.progress}%
                              </Typography>
                            </Box>
                            <LinearProgress
                              variant="determinate"
                              value={classData.progress}
                              sx={{
                                height: 6,
                                borderRadius: 3,
                                bgcolor: '#e2e8f0',
                                '& .MuiLinearProgress-bar': {
                                  bgcolor: classData.progress === 100 ? '#22c55e' : '#38aae1',
                                  borderRadius: 3,
                                },
                              }}
                            />
                          </Box>
                        </TableCell>
                        <TableCell sx={{ border: 'none' }}>
                          <Chip
                            label={classData.status.charAt(0).toUpperCase() + classData.status.slice(1)}
                            size="small"
                            sx={{
                              bgcolor: statusColor.bg,
                              color: statusColor.color,
                              fontWeight: 600,
                              fontSize: '0.75rem',
                            }}
                          />
                        </TableCell>
                        <TableCell sx={{ border: 'none' }}>
                          <IconButton
                            size="small"
                            onClick={(e) => this.handleMenuClick(e, classData)}
                            sx={{ color: '#64748b' }}
                          >
                            <MoreVert />
                          </IconButton>
                        </TableCell>
                      </TableRow>
                    );
                  })}
                </TableBody>
              </Table>
            </TableContainer>
          </CardContent>
        </Card>

        <Menu
          anchorEl={anchorEl}
          open={Boolean(anchorEl)}
          onClose={this.handleMenuClose}
        >
          <MenuItem onClick={this.handleView}>
            <Visibility sx={{ mr: 1, fontSize: 20 }} />
            View Details
          </MenuItem>
          <MenuItem onClick={this.handleMenuClose}>
            <Edit sx={{ mr: 1, fontSize: 20 }} />
            Edit
          </MenuItem>
          <MenuItem onClick={this.handleDelete} sx={{ color: '#dc2626' }}>
            <Delete sx={{ mr: 1, fontSize: 20 }} />
            Delete
          </MenuItem>
        </Menu>

        <Dialog
          open={viewDialogOpen}
          onClose={this.handleCloseViewDialog}
          maxWidth="md"
          fullWidth
        >
          {selectedClass && (
            <>
              <DialogTitle sx={{ bgcolor: '#f8fafc', borderBottom: '1px solid #e2e8f0' }}>
                <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 700 }}>
                  {selectedClass.name}
                </Typography>
              </DialogTitle>
              <DialogContent sx={{ p: 4 }}>
                <Grid container spacing={3}>
                  <Grid item xs={12}>
                    <Typography variant="body2" sx={{ color: '#64748b', mb: 2 }}>
                      {selectedClass.description}
                    </Typography>
                  </Grid>
                  <Grid item xs={6}>
                    <Typography variant="caption" sx={{ color: '#94a3b8' }}>Type</Typography>
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {selectedClass.type}
                    </Typography>
                  </Grid>
                  <Grid item xs={6}>
                    <Typography variant="caption" sx={{ color: '#94a3b8' }}>Age Range</Typography>
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {selectedClass.ageRange}
                    </Typography>
                  </Grid>
                  <Grid item xs={6}>
                    <Typography variant="caption" sx={{ color: '#94a3b8' }}>Mentor</Typography>
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {selectedClass.mentor}
                    </Typography>
                  </Grid>
                  <Grid item xs={6}>
                    <Typography variant="caption" sx={{ color: '#94a3b8' }}>Students Enrolled</Typography>
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {selectedClass.students}
                    </Typography>
                  </Grid>
                  <Grid item xs={6}>
                    <Typography variant="caption" sx={{ color: '#94a3b8' }}>Start Date</Typography>
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {new Date(selectedClass.startDate).toLocaleDateString()}
                    </Typography>
                  </Grid>
                  <Grid item xs={6}>
                    <Typography variant="caption" sx={{ color: '#94a3b8' }}>End Date</Typography>
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {new Date(selectedClass.endDate).toLocaleDateString()}
                    </Typography>
                  </Grid>
                  <Grid item xs={12}>
                    <Typography variant="caption" sx={{ color: '#94a3b8' }}>Schedule</Typography>
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {selectedClass.schedule}
                    </Typography>
                  </Grid>
                </Grid>
              </DialogContent>
              <DialogActions sx={{ p: 3, bgcolor: '#f8fafc' }}>
                <Button onClick={this.handleCloseViewDialog} sx={{ color: '#64748b' }}>
                  Close
                </Button>
              </DialogActions>
            </>
          )}
        </Dialog>

        <Dialog open={deleteDialogOpen} onClose={this.handleCancelDelete}>
          <DialogTitle>Confirm Delete</DialogTitle>
          <DialogContent>
            <Typography>
              Are you sure you want to delete <strong>{selectedClass?.name}</strong>?
              This action cannot be undone.
            </Typography>
          </DialogContent>
          <DialogActions>
            <Button onClick={this.handleCancelDelete} sx={{ color: '#64748b' }}>
              Cancel
            </Button>
            <Button onClick={this.handleConfirmDelete} sx={{ color: '#dc2626' }}>
              Delete
            </Button>
          </DialogActions>
        </Dialog>
      </Box>
    );
  }
}
