import React, { Component } from 'react';
import {
  Box,
  Typography,
  Button,
  Card,
  CardContent,
  Grid,
  Chip,
  Avatar,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  IconButton,
  Menu,
  MenuItem,
  TextField,
  InputAdornment,
  Collapse,
  LinearProgress,
  Divider,
  Tabs,
  Tab,
  Paper,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
} from '@mui/material';
import {
  ArrowBack,
  School,
  Schedule,
  People,
  TrendingUp,
  MoreVert,
  Edit,
  Delete,
  Visibility,
  Search,
  KeyboardArrowDown,
  KeyboardArrowUp,
  Male,
  Female,
  Star,
  Person,
  Group,
  CheckCircle,
  VideoLibrary,
  Close,
  CloudUpload,
  DeleteOutline,
  PlayCircleOutline,
} from '@mui/icons-material';
import { CourseView } from './CourseView';
import { VidstackPlayer } from './VidstackPlayer';
import { VideoUploadDragger } from './VideoUploadDragger';

interface ClassData {
  id: number;
  name: string;
  coursesCount: number;
  revenue: number;
  students: number;
  passType: string;
  status: 'Published' | 'Draft';
}

interface Course {
  id: string;
  title: string;
  description: string;
  instructor: string;
  duration: string;
  lessons: number;
  students: number;
  rating: number;
  price: number;
  level: string;
  category: string;
  thumbnail: string;
  enrollments: number;
  completionRate: number;
  lastUpdated: string;
  sequence: number;
  revenue: number;
  activeStudents: number;
  introductionVideo?: string;
  introductionVideoFile?: File;
  introductionVideoName?: string;
  analytics: {
    genderDistribution: { male: number; female: number };
    completionRate: number;
    dropOffRate: number;
    timeToTask: number;
    passRate: number;
    totalEnrolled: number;
    mentorHours: number;
  };
}

interface ClassViewProps {
  classData: ClassData;
  onBack: () => void;
}

interface ClassViewState {
  selectedTab: number;
  searchTerm: string;
  viewingCourse: Course | null;
  expandedCourse: string | null;
  anchorEl: HTMLElement | null;
  selectedItem: string | null;
  courses: Course[];
  editingVideoId: string | null;
  videoUrls: { [key: string]: string };
  videoDialogOpen: boolean;
  selectedCourseForVideo: Course | null;
  uploadedVideoFiles: { [key: string]: File | null };
  videoPreviewUrls: { [key: string]: string };
  previewDialogOpen: boolean;
  previewCourse: Course | null;
}

export class ClassView extends Component<ClassViewProps, ClassViewState> {
  constructor(props: ClassViewProps) {
    super(props);
    this.state = {
      selectedTab: 0,
      searchTerm: '',
      viewingCourse: null,
      expandedCourse: null,
      anchorEl: null,
      selectedItem: null,
      editingVideoId: null,
      videoUrls: {},
      videoDialogOpen: false,
      selectedCourseForVideo: null,
      uploadedVideoFiles: {},
      videoPreviewUrls: {},
      previewDialogOpen: false,
      previewCourse: null,
      courses: [
        {
          id: '1',
          title: 'Introduction to Web Development',
          description: 'Learn the basics of web development including HTML, CSS, and JavaScript fundamentals.',
          instructor: 'Sarah Johnson',
          duration: '8 weeks',
          lessons: 24,
          students: 156,
          rating: 4.8,
          price: 2500,
          level: 'Beginner',
          category: 'Programming',
          thumbnail: 'https://images.unsplash.com/photo-1593720213681-e9a8778330a7?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHx3ZWIlMjBkZXZlbG9wbWVudCUyMHByb2dyYW1taW5nfGVufDF8fHx8MTc1OTMxMjEzNXww&ixlib=rb-4.1.0&q=80&w=1080',
          enrollments: 1247,
          completionRate: 78,
          lastUpdated: '2024-01-15',
          sequence: 1,
          revenue: 390000,
          activeStudents: 142,
          introductionVideo: 'https://commondatastorage.googleapis.com/gtv-videos-bucket/sample/BigBuckBunny.mp4',
          analytics: {
            genderDistribution: { male: 45, female: 55 },
            completionRate: 78,
            dropOffRate: 15,
            timeToTask: 42,
            passRate: 89,
            totalEnrolled: 156,
            mentorHours: 234,
          },
        },
        {
          id: '2',
          title: 'Advanced React Development',
          description: 'Master React concepts including hooks, context, and performance optimization.',
          instructor: 'Michael Chen',
          duration: '10 weeks',
          lessons: 32,
          students: 89,
          rating: 4.9,
          price: 3500,
          level: 'Advanced',
          category: 'Programming',
          thumbnail: 'https://images.unsplash.com/photo-1670057046254-3b5095eb4b66?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHxyZWFjdCUyMGphdmFzY3JpcHQlMjBjb2Rpbmd8ZW58MXx8fHwxNzU5NDEzNTQ5fDA&ixlib=rb-4.1.0&q=80&w=1080',
          enrollments: 892,
          completionRate: 85,
          lastUpdated: '2024-01-20',
          sequence: 2,
          revenue: 311500,
          activeStudents: 76,
          introductionVideo: 'https://commondatastorage.googleapis.com/gtv-videos-bucket/sample/ElephantsDream.mp4',
          analytics: {
            genderDistribution: { male: 52, female: 48 },
            completionRate: 85,
            dropOffRate: 12,
            timeToTask: 38,
            passRate: 91,
            totalEnrolled: 89,
            mentorHours: 187,
          },
        },
        {
          id: '3',
          title: 'Data Science Fundamentals',
          description: 'Explore data analysis, visualization, and machine learning basics.',
          instructor: 'Dr. Emily Rodriguez',
          duration: '12 weeks',
          lessons: 36,
          students: 67,
          rating: 4.7,
          price: 4200,
          level: 'Intermediate',
          category: 'Data Science',
          thumbnail: 'https://images.unsplash.com/photo-1666875753105-c63a6f3bdc86?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxkYXRhJTIwc2NpZW5jZSUyMGFuYWx5dGljc3xlbnwxfHx8fDE3NTkzMzQ3NzZ8MA&ixlib=rb-4.1.0&q=80&w=1080',
          enrollments: 543,
          completionRate: 72,
          lastUpdated: '2024-01-18',
          sequence: 3,
          revenue: 281400,
          activeStudents: 48,
          introductionVideo: 'https://commondatastorage.googleapis.com/gtv-videos-bucket/sample/ForBiggerBlazes.mp4',
          analytics: {
            genderDistribution: { male: 38, female: 62 },
            completionRate: 72,
            dropOffRate: 18,
            timeToTask: 45,
            passRate: 87,
            totalEnrolled: 67,
            mentorHours: 156,
          },
        },
        {
          id: '4',
          title: 'Mobile App Development',
          description: 'Build cross-platform mobile applications using React Native.',
          instructor: 'James Wilson',
          duration: '9 weeks',
          lessons: 28,
          students: 34,
          rating: 4.6,
          price: 3200,
          level: 'Intermediate',
          category: 'Mobile Development',
          thumbnail: 'https://images.unsplash.com/photo-1512941937669-90a1b58e7e9c?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHxtb2JpbGUlMjBhcHAlMjBkZXZlbG9wbWVudHxlbnwxfHx8fDE3NTkzOTQ1MDF8MA&ixlib=rb-4.1.0&q=80&w=1080',
          enrollments: 298,
          completionRate: 68,
          lastUpdated: '2024-01-22',
          sequence: 4,
          revenue: 108800,
          activeStudents: 23,
          analytics: {
            genderDistribution: { male: 58, female: 42 },
            completionRate: 68,
            dropOffRate: 22,
            timeToTask: 38,
            passRate: 85,
            totalEnrolled: 34,
            mentorHours: 98,
          },
        },
      ]
    };
  }

  handleTabChange = (event: React.SyntheticEvent, newValue: number) => {
    this.setState({ selectedTab: newValue });
  };

  handleViewCourse = (course: Course) => {
    this.setState({ viewingCourse: course });
  };

  handleBackToCourses = () => {
    this.setState({ viewingCourse: null });
  };

  toggleCourseExpansion = (courseId: string) => {
    this.setState({
      expandedCourse: this.state.expandedCourse === courseId ? null : courseId
    });
  };

  handleMenuClick = (event: React.MouseEvent<HTMLElement>, itemId: string) => {
    this.setState({
      anchorEl: event.currentTarget,
      selectedItem: itemId
    });
  };

  handleMenuClose = () => {
    this.setState({
      anchorEl: null,
      selectedItem: null
    });
  };

  getGenderColor = (percentage: number, gender: 'male' | 'female') => {
    return gender === 'male' ? '#38aae1' : '#ff9800';
  };

  getLevelColor = (level: string) => {
    switch (level) {
      case 'Beginner':
        return { backgroundColor: '#dcfce7', color: '#166534' };
      case 'Intermediate':
        return { backgroundColor: '#fef3c7', color: '#92400e' };
      case 'Advanced':
        return { backgroundColor: '#fee2e2', color: '#991b1b' };
      default:
        return { backgroundColor: '#e0f2fe', color: '#0284c7' };
    }
  };

  handleEditVideo = (courseId: string) => {
    const course = this.state.courses.find(c => c.id === courseId);
    this.setState({
      editingVideoId: courseId,
      videoUrls: {
        ...this.state.videoUrls,
        [courseId]: course?.introductionVideo || ''
      }
    });
  };

  handleSaveVideo = (courseId: string) => {
    const videoUrl = this.state.videoUrls[courseId] || '';
    this.setState({
      courses: this.state.courses.map(course =>
        course.id === courseId
          ? { ...course, introductionVideo: videoUrl }
          : course
      ),
      editingVideoId: null
    });
  };

  handleCancelVideoEdit = () => {
    this.setState({ editingVideoId: null });
  };

  handleVideoUrlChange = (courseId: string, value: string) => {
    this.setState({
      videoUrls: {
        ...this.state.videoUrls,
        [courseId]: value
      }
    });
  };

  getEmbedUrl = (url: string): string => {
    // YouTube
    if (url.includes('youtube.com') || url.includes('youtu.be')) {
      const videoId = url.includes('youtu.be') 
        ? url.split('youtu.be/')[1]?.split('?')[0]
        : url.split('v=')[1]?.split('&')[0];
      return `https://www.youtube.com/embed/${videoId}`;
    }
    // Vimeo
    if (url.includes('vimeo.com')) {
      const videoId = url.split('vimeo.com/')[1]?.split('?')[0];
      return `https://player.vimeo.com/video/${videoId}`;
    }
    // Return as-is for direct video URLs
    return url;
  };

  handleOpenVideoDialog = (course: Course) => {
    this.setState({
      videoDialogOpen: true,
      selectedCourseForVideo: course,
      uploadedVideoFiles: {
        ...this.state.uploadedVideoFiles,
        [course.id]: null
      }
    });
  };

  handleCloseVideoDialog = () => {
    const { selectedCourseForVideo, videoPreviewUrls } = this.state;
    // Clean up preview URL if exists
    if (selectedCourseForVideo && videoPreviewUrls[selectedCourseForVideo.id]) {
      URL.revokeObjectURL(videoPreviewUrls[selectedCourseForVideo.id]);
    }
    this.setState({
      videoDialogOpen: false,
      selectedCourseForVideo: null
    });
  };

  handleVideoFileChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const { selectedCourseForVideo, videoPreviewUrls } = this.state;
    if (!selectedCourseForVideo) return;

    const file = event.target.files?.[0];
    if (file) {
      // Validate file type
      if (!file.type.startsWith('video/')) {
        alert('Please select a valid video file');
        return;
      }

      // Clean up old preview URL if exists
      if (videoPreviewUrls[selectedCourseForVideo.id]) {
        URL.revokeObjectURL(videoPreviewUrls[selectedCourseForVideo.id]);
      }

      // Create new preview URL
      const previewUrl = URL.createObjectURL(file);

      this.setState({
        uploadedVideoFiles: {
          ...this.state.uploadedVideoFiles,
          [selectedCourseForVideo.id]: file
        },
        videoPreviewUrls: {
          ...this.state.videoPreviewUrls,
          [selectedCourseForVideo.id]: previewUrl
        }
      });
    }
  };

  handleRemoveUploadedVideo = () => {
    const { selectedCourseForVideo, videoPreviewUrls } = this.state;
    if (!selectedCourseForVideo) return;

    // Clean up preview URL
    if (videoPreviewUrls[selectedCourseForVideo.id]) {
      URL.revokeObjectURL(videoPreviewUrls[selectedCourseForVideo.id]);
    }

    this.setState({
      uploadedVideoFiles: {
        ...this.state.uploadedVideoFiles,
        [selectedCourseForVideo.id]: null
      },
      videoPreviewUrls: {
        ...this.state.videoPreviewUrls,
        [selectedCourseForVideo.id]: ''
      }
    });
  };

  handleSaveVideoFromDialog = () => {
    const { selectedCourseForVideo, uploadedVideoFiles, videoPreviewUrls } = this.state;
    if (!selectedCourseForVideo) return;

    const videoFile = uploadedVideoFiles[selectedCourseForVideo.id];
    const videoUrl = videoPreviewUrls[selectedCourseForVideo.id];

    this.setState({
      courses: this.state.courses.map(course =>
        course.id === selectedCourseForVideo.id
          ? { 
              ...course, 
              introductionVideo: videoUrl,
              introductionVideoFile: videoFile || undefined,
              introductionVideoName: videoFile?.name || undefined
            }
          : course
      ),
      videoDialogOpen: false,
      selectedCourseForVideo: null
    });
  };

  handleOpenPreviewDialog = (course: Course) => {
    this.setState({
      previewDialogOpen: true,
      previewCourse: course
    });
  };

  handleClosePreviewDialog = () => {
    this.setState({
      previewDialogOpen: false,
      previewCourse: null
    });
  };

  render() {
    const { classData, onBack } = this.props;
    const { selectedTab, searchTerm, viewingCourse, expandedCourse, anchorEl, courses, editingVideoId, videoUrls, videoPreviewUrls, uploadedVideoFiles } = this.state;

    const filteredCourses = courses
      .filter(course =>
        course.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
        course.instructor.toLowerCase().includes(searchTerm.toLowerCase())
      )
      .sort((a, b) => a.sequence - b.sequence);

    // Calculate totals for the class summary
    const totalStudents = courses.reduce((sum, course) => sum + course.students, 0);
    const averageCompletionRate = Math.round(
      courses.reduce((sum, course) => sum + course.completionRate, 0) / courses.length
    );
    const totalRevenue = courses.reduce((sum, course) => sum + course.revenue, 0);

    if (viewingCourse) {
      return <CourseView course={viewingCourse} onBack={this.handleBackToCourses} />;
    }

    return (
      <Box sx={{ p: 3 }}>
        {/* Header */}
        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 4 }}>
          <Box>
            <Button
              startIcon={<ArrowBack />}
              onClick={onBack}
              sx={{ 
                mb: 2,
                color: '#64748b',
                textTransform: 'none',
                '&:hover': { bgcolor: 'rgba(100, 116, 139, 0.1)' }
              }}
            >
              Back to Classes
            </Button>
            <Typography variant="h4" sx={{ color: '#25476a', fontWeight: 700, mb: 1 }}>
              {classData.name}
            </Typography>
            <Typography variant="body1" sx={{ color: '#64748b' }}>
              Manage courses, monitor student progress, and track analytics for this class
            </Typography>
          </Box>
        </Box>

        {/* Statistics Cards */}
        <Grid container spacing={3} sx={{ mb: 4 }}>
          <Grid item xs={12} md={3}>
            <Card sx={{ height: '100%' }}>
              <CardContent>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                  <Avatar sx={{ bgcolor: '#25476a', mr: 2 }}>
                    <People />
                  </Avatar>
                  <Box>
                    <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {totalStudents.toLocaleString()}
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#64748b' }}>
                      Total Students
                    </Typography>
                  </Box>
                </Box>
              </CardContent>
            </Card>
          </Grid>
          <Grid item xs={12} md={3}>
            <Card sx={{ height: '100%' }}>
              <CardContent>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                  <Avatar sx={{ bgcolor: '#38aae1', mr: 2 }}>
                    <CheckCircle />
                  </Avatar>
                  <Box>
                    <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {averageCompletionRate}%
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#64748b' }}>
                      Completion Rate
                    </Typography>
                  </Box>
                </Box>
              </CardContent>
            </Card>
          </Grid>
          <Grid item xs={12} md={3}>
            <Card sx={{ height: '100%' }}>
              <CardContent>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                  <Avatar sx={{ bgcolor: '#feb139', mr: 2 }}>
                    <TrendingUp />
                  </Avatar>
                  <Box>
                    <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 600 }}>
                      KES {totalRevenue.toLocaleString()}
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#64748b' }}>
                      Total Revenue
                    </Typography>
                  </Box>
                </Box>
              </CardContent>
            </Card>
          </Grid>
          <Grid item xs={12} md={3}>
            <Card sx={{ height: '100%' }}>
              <CardContent>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                  <Avatar sx={{ bgcolor: '#10b981', mr: 2 }}>
                    <School />
                  </Avatar>
                  <Box>
                    <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {courses.length}
                    </Typography>
                    <Typography variant="body2" sx={{ color: '#64748b' }}>
                      Total Courses
                    </Typography>
                  </Box>
                </Box>
              </CardContent>
            </Card>
          </Grid>
        </Grid>

        {/* Tabs */}
        <Paper sx={{ borderRadius: 3, boxShadow: '0 4px 20px rgba(0,0,0,0.08)', overflow: 'hidden' }}>
          <Tabs 
            value={selectedTab} 
            onChange={this.handleTabChange}
            sx={{ 
              borderBottom: '1px solid #e2e8f0',
              '& .MuiTab-root': {
                textTransform: 'none',
                fontWeight: 600,
                fontSize: '1rem',
                color: '#64748b',
                '&.Mui-selected': {
                  color: '#38aae1',
                }
              }
            }}
          >
            <Tab label="Courses" />
            <Tab label="Students" />
            <Tab label="Analytics" />
            <Tab label="Settings" />
          </Tabs>

          {/* Tab Content */}
          <Box sx={{ p: 4 }}>
            {/* Courses Tab */}
            {selectedTab === 0 && (
              <Box>
                <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 4 }}>
                  <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                    Courses ({filteredCourses.length})
                  </Typography>
                  <TextField
                    placeholder="Search courses..."
                    value={searchTerm}
                    onChange={(e) => this.setState({ searchTerm: e.target.value })}
                    sx={{ width: 300 }}
                    size="small"
                    InputProps={{
                      startAdornment: (
                        <InputAdornment position="start">
                          <Search sx={{ color: '#64748b' }} />
                        </InputAdornment>
                      ),
                    }}
                  />
                </Box>

                {/* Courses Table */}
                <Card sx={{ width: '100%', overflow: 'hidden' }}>
                  <TableContainer sx={{ 
                    maxHeight: 600, 
                    overflowY: 'scroll',
                    overflowX: 'auto',
                    '&::-webkit-scrollbar': {
                      width: '8px',
                      height: '8px',
                    },
                    '&::-webkit-scrollbar-track': {
                      backgroundColor: '#f1f5f9',
                    },
                    '&::-webkit-scrollbar-thumb': {
                      backgroundColor: '#cbd5e1',
                      borderRadius: '4px',
                      '&:hover': {
                        backgroundColor: '#94a3b8',
                      },
                    },
                  }}>
                    <Table stickyHeader sx={{ width: '100%', minWidth: 1200, tableLayout: 'fixed' }}>
                      <TableHead>
                        <TableRow sx={{ backgroundColor: '#f8fafc' }}>
                          <TableCell sx={{ fontWeight: 600, color: '#25476a', bgcolor: '#f8fafc', width: '35%' }}>Course</TableCell>
                          <TableCell sx={{ fontWeight: 600, color: '#25476a', bgcolor: '#f8fafc', width: '10%' }}>Level</TableCell>
                          <TableCell sx={{ fontWeight: 600, color: '#25476a', bgcolor: '#f8fafc', width: '10%' }}>Students</TableCell>
                          <TableCell sx={{ fontWeight: 600, color: '#25476a', bgcolor: '#f8fafc', width: '15%' }}>Completion</TableCell>
                          <TableCell sx={{ fontWeight: 600, color: '#25476a', bgcolor: '#f8fafc', width: '12%' }}>Revenue</TableCell>
                          <TableCell sx={{ fontWeight: 600, color: '#25476a', bgcolor: '#f8fafc', width: '18%' }}>Actions</TableCell>
                        </TableRow>
                      </TableHead>
                      <TableBody>
                        {filteredCourses.map((course) => (
                          <React.Fragment key={course.id}>
                            <TableRow hover>
                              <TableCell>
                                <Box sx={{ display: 'flex', alignItems: 'center' }}>
                                  <Box 
                                    component="img"
                                    src={course.thumbnail}
                                    alt={course.title}
                                    sx={{ 
                                      width: 56, 
                                      height: 56, 
                                      borderRadius: 2, 
                                      objectFit: 'cover',
                                      mr: 2,
                                      border: '2px solid #e2e8f0'
                                    }}
                                  />
                                  <Box>
                                    <Typography variant="body1" sx={{ fontWeight: 600, color: '#25476a' }}>
                                      {course.title}
                                    </Typography>
                                    <Typography variant="caption" sx={{ color: '#64748b' }}>
                                      {course.instructor} • {course.duration}
                                    </Typography>
                                  </Box>
                                </Box>
                              </TableCell>
                              <TableCell>
                                <Chip
                                  label={course.level}
                                  size="small"
                                  sx={{
                                    ...this.getLevelColor(course.level),
                                    fontWeight: 500,
                                  }}
                                />
                              </TableCell>
                              <TableCell>{course.students}</TableCell>
                              <TableCell>
                                <Box sx={{ display: 'flex', alignItems: 'center' }}>
                                  <Typography variant="body2" sx={{ mr: 1 }}>
                                    {course.completionRate}%
                                  </Typography>
                                  <LinearProgress
                                    variant="determinate"
                                    value={course.completionRate}
                                    sx={{
                                      width: 60,
                                      height: 4,
                                      borderRadius: 2,
                                      bgcolor: 'rgba(56, 170, 225, 0.2)',
                                      '& .MuiLinearProgress-bar': { bgcolor: '#38aae1' }
                                    }}
                                  />
                                </Box>
                              </TableCell>
                              <TableCell>KES {course.revenue.toLocaleString()}</TableCell>
                              <TableCell>
                                <Box sx={{ display: 'flex', gap: 1 }}>
                                  <IconButton
                                    onClick={() => this.handleViewCourse(course)}
                                    size="small"
                                    sx={{ color: '#38aae1' }}
                                    title="View Course"
                                  >
                                    <Visibility />
                                  </IconButton>
                                  <IconButton
                                    onClick={() => this.handleOpenVideoDialog(course)}
                                    size="small"
                                    sx={{ color: '#feb139' }}
                                    title="Add Introduction Video"
                                  >
                                    <VideoLibrary />
                                  </IconButton>
                                  {course.introductionVideo && (
                                    <IconButton
                                      onClick={() => this.handleOpenPreviewDialog(course)}
                                      size="small"
                                      sx={{ color: '#4caf50' }}
                                      title="Preview Video"
                                    >
                                      <PlayCircleOutline />
                                    </IconButton>
                                  )}
                                  <IconButton
                                    onClick={() => this.toggleCourseExpansion(course.id)}
                                    size="small"
                                    sx={{ color: '#64748b' }}
                                    title="View Analytics"
                                  >
                                    {expandedCourse === course.id ? <KeyboardArrowUp /> : <KeyboardArrowDown />}
                                  </IconButton>
                                  <IconButton
                                    onClick={(e) => this.handleMenuClick(e, course.id)}
                                    size="small"
                                    title="More Actions"
                                  >
                                    <MoreVert />
                                  </IconButton>
                                </Box>
                              </TableCell>
                            </TableRow>

                            {/* Course Analytics - Collapsible */}
                            <TableRow>
                              <TableCell colSpan={6} sx={{ p: 0, border: 'none' }}>
                                <Collapse in={expandedCourse === course.id}>
                                  <Box sx={{ p: 3, bgcolor: '#f8fafc' }}>
                                    {/* Introduction Video Section */}
                                    <Card sx={{ mb: 3, p: 3, bgcolor: 'white', border: '2px solid #38aae1' }}>
                                      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 2 }}>
                                        <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a', display: 'flex', alignItems: 'center' }}>
                                          <VideoLibrary sx={{ mr: 1, color: '#38aae1' }} />
                                          Course Introduction Video
                                        </Typography>
                                        <Button
                                          variant="outlined"
                                          size="small"
                                          startIcon={course.introductionVideo ? <Edit /> : <VideoLibrary />}
                                          onClick={() => this.handleOpenVideoDialog(course)}
                                          sx={{
                                            borderColor: '#38aae1',
                                            color: '#38aae1',
                                            textTransform: 'none',
                                            fontWeight: 600,
                                            '&:hover': {
                                              borderColor: '#2c8fc9',
                                              backgroundColor: 'rgba(56, 170, 225, 0.05)',
                                            }
                                          }}
                                        >
                                          {course.introductionVideo ? 'Change Video' : 'Add Video'}
                                        </Button>
                                      </Box>

                                      {course.introductionVideo ? (
                                        <Box>
                                          <Box sx={{ mb: 2, maxWidth: '100%', width: '100%' }}>
                                            <VidstackPlayer
                                              src={course.introductionVideo}
                                              title="Course Introduction"
                                              aspectRatio="16/9"
                                              controls={true}
                                            />
                                          </Box>
                                          {course.introductionVideoName && (
                                            <Box sx={{ display: 'flex', alignItems: 'center', p: 2, bgcolor: '#f8fafc', borderRadius: 2 }}>
                                              <PlayCircleOutline sx={{ color: '#38aae1', mr: 1 }} />
                                              <Typography variant="body2" sx={{ color: '#64748b' }}>
                                                <strong>File:</strong> {course.introductionVideoName}
                                              </Typography>
                                            </Box>
                                          )}
                                        </Box>
                                      ) : (
                                        <Box sx={{ 
                                          p: 4, 
                                          textAlign: 'center', 
                                          bgcolor: '#f8fafc', 
                                          borderRadius: 2,
                                          border: '2px dashed #e2e8f0'
                                        }}>
                                          <School sx={{ fontSize: 48, color: '#cbd5e1', mb: 1 }} />
                                          <Typography variant="body1" sx={{ color: '#64748b', mb: 1 }}>
                                            No introduction video added yet
                                          </Typography>
                                          <Typography variant="body2" sx={{ color: '#94a3b8' }}>
                                            Click the video icon to upload an introduction video for this course
                                          </Typography>
                                        </Box>
                                      )}
                                    </Card>

                                    <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a', mb: 3 }}>
                                      Course Analytics
                                    </Typography>
                                    
                                    <Grid container spacing={3}>
                                      {/* Analytics Cards */}
                                      <Grid item xs={12} sm={6} md={4}>
                                        <Card sx={{ p: 2, bgcolor: 'rgba(56, 170, 225, 0.05)', border: '1px solid rgba(56, 170, 225, 0.2)' }}>
                                          <Typography variant="body2" sx={{ color: '#64748b', mb: 1 }}>
                                            Completion Rate
                                          </Typography>
                                          <Typography variant="h5" sx={{ fontWeight: 700, color: '#38aae1', mb: 1 }}>
                                            {course.analytics.completionRate}%
                                          </Typography>
                                          <LinearProgress
                                            variant="determinate"
                                            value={course.analytics.completionRate}
                                            sx={{
                                              height: 6,
                                              borderRadius: 3,
                                              bgcolor: 'rgba(56, 170, 225, 0.2)',
                                              '& .MuiLinearProgress-bar': { bgcolor: '#38aae1' }
                                            }}
                                          />
                                        </Card>
                                      </Grid>
                                      
                                      <Grid item xs={12} sm={6} md={4}>
                                        <Card sx={{ p: 2, bgcolor: 'rgba(76, 175, 80, 0.05)', border: '1px solid rgba(76, 175, 80, 0.2)' }}>
                                          <Typography variant="body2" sx={{ color: '#64748b', mb: 1 }}>
                                            Pass Rate
                                          </Typography>
                                          <Typography variant="h5" sx={{ fontWeight: 700, color: '#4caf50', mb: 1 }}>
                                            {course.analytics.passRate}%
                                          </Typography>
                                          <LinearProgress
                                            variant="determinate"
                                            value={course.analytics.passRate}
                                            sx={{
                                              height: 6,
                                              borderRadius: 3,
                                              bgcolor: 'rgba(76, 175, 80, 0.2)',
                                              '& .MuiLinearProgress-bar': { bgcolor: '#4caf50' }
                                            }}
                                          />
                                        </Card>
                                      </Grid>
                                      
                                      <Grid item xs={12} sm={6} md={4}>
                                        <Card sx={{ p: 2, bgcolor: 'rgba(255, 152, 0, 0.05)', border: '1px solid rgba(255, 152, 0, 0.2)' }}>
                                          <Typography variant="body2" sx={{ color: '#64748b', mb: 1 }}>
                                            Drop-off Rate
                                          </Typography>
                                          <Typography variant="h5" sx={{ fontWeight: 700, color: '#ff9800', mb: 1 }}>
                                            {course.analytics.dropOffRate}%
                                          </Typography>
                                          <LinearProgress
                                            variant="determinate"
                                            value={course.analytics.dropOffRate}
                                            sx={{
                                              height: 6,
                                              borderRadius: 3,
                                              bgcolor: 'rgba(255, 152, 0, 0.2)',
                                              '& .MuiLinearProgress-bar': { bgcolor: '#ff9800' }
                                            }}
                                          />
                                        </Card>
                                      </Grid>

                                      {/* Gender Distribution */}
                                      <Grid item xs={12} sm={6}>
                                        <Card sx={{ p: 3 }}>
                                          <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a', mb: 2 }}>
                                            Gender Distribution
                                          </Typography>
                                          <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                                            <Male sx={{ color: '#38aae1', mr: 1 }} />
                                            <Typography variant="body2" sx={{ color: '#64748b', mr: 2 }}>
                                              Male: {course.analytics.genderDistribution.male}%
                                            </Typography>
                                            <Box sx={{ flex: 1, mx: 2 }}>
                                              <LinearProgress
                                                variant="determinate"
                                                value={course.analytics.genderDistribution.male}
                                                sx={{
                                                  height: 8,
                                                  borderRadius: 4,
                                                  bgcolor: 'rgba(56, 170, 225, 0.2)',
                                                  '& .MuiLinearProgress-bar': { bgcolor: '#38aae1' }
                                                }}
                                              />
                                            </Box>
                                          </Box>
                                          <Box sx={{ display: 'flex', alignItems: 'center' }}>
                                            <Female sx={{ color: '#ff9800', mr: 1 }} />
                                            <Typography variant="body2" sx={{ color: '#64748b', mr: 2 }}>
                                              Female: {course.analytics.genderDistribution.female}%
                                            </Typography>
                                            <Box sx={{ flex: 1, mx: 2 }}>
                                              <LinearProgress
                                                variant="determinate"
                                                value={course.analytics.genderDistribution.female}
                                                sx={{
                                                  height: 8,
                                                  borderRadius: 4,
                                                  bgcolor: 'rgba(255, 152, 0, 0.2)',
                                                  '& .MuiLinearProgress-bar': { bgcolor: '#ff9800' }
                                                }}
                                              />
                                            </Box>
                                          </Box>
                                        </Card>
                                      </Grid>

                                      {/* Additional Metrics */}
                                      <Grid item xs={12} sm={6}>
                                        <Card sx={{ p: 3 }}>
                                          <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a', mb: 2 }}>
                                            Performance Metrics
                                          </Typography>
                                          <Box sx={{ display: 'flex', justifyContent: 'space-between', mb: 1 }}>
                                            <Typography variant="body2" sx={{ color: '#64748b' }}>Time to Task</Typography>
                                            <Typography variant="body2" sx={{ fontWeight: 600, color: '#25476a' }}>
                                              {course.analytics.timeToTask} min
                                            </Typography>
                                          </Box>
                                          <Box sx={{ display: 'flex', justifyContent: 'space-between', mb: 1 }}>
                                            <Typography variant="body2" sx={{ color: '#64748b' }}>Total Enrolled</Typography>
                                            <Typography variant="body2" sx={{ fontWeight: 600, color: '#25476a' }}>
                                              {course.analytics.totalEnrolled}
                                            </Typography>
                                          </Box>
                                          <Box sx={{ display: 'flex', justifyContent: 'space-between' }}>
                                            <Typography variant="body2" sx={{ color: '#64748b' }}>Mentor Hours</Typography>
                                            <Typography variant="body2" sx={{ fontWeight: 600, color: '#25476a' }}>
                                              {course.analytics.mentorHours}h
                                            </Typography>
                                          </Box>
                                        </Card>
                                      </Grid>
                                    </Grid>
                                  </Box>
                                </Collapse>
                              </TableCell>
                            </TableRow>
                          </React.Fragment>
                        ))}
                      </TableBody>
                    </Table>
                  </TableContainer>
                </Card>
              </Box>
            )}

            {/* Students Tab */}
            {selectedTab === 1 && (
              <Box>
                <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a', mb: 4 }}>
                  Class Students
                </Typography>
                <Typography variant="body1" sx={{ color: '#64748b' }}>
                  Student management functionality will be implemented here.
                </Typography>
              </Box>
            )}

            {/* Analytics Tab */}
            {selectedTab === 2 && (
              <Box>
                <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a', mb: 4 }}>
                  Class Analytics
                </Typography>
                <Typography variant="body1" sx={{ color: '#64748b' }}>
                  Advanced analytics and reporting functionality will be implemented here.
                </Typography>
              </Box>
            )}

            {/* Settings Tab */}
            {selectedTab === 3 && (
              <Box>
                <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a', mb: 4 }}>
                  Class Settings
                </Typography>
                <Typography variant="body1" sx={{ color: '#64748b' }}>
                  Class configuration and settings will be implemented here.
                </Typography>
              </Box>
            )}
          </Box>
        </Paper>

        {/* Context Menu */}
        <Menu
          anchorEl={anchorEl}
          open={Boolean(anchorEl)}
          onClose={this.handleMenuClose}
        >
          <MenuItem onClick={this.handleMenuClose}>
            <Visibility sx={{ mr: 1, fontSize: 20 }} />
            View Course
          </MenuItem>
          <MenuItem onClick={this.handleMenuClose}>
            <Edit sx={{ mr: 1, fontSize: 20 }} />
            Edit Course
          </MenuItem>
          <MenuItem onClick={this.handleMenuClose} sx={{ color: '#ef4444' }}>
            <Delete sx={{ mr: 1, fontSize: 20 }} />
            Remove Course
          </MenuItem>
        </Menu>

        {/* Introduction Video Dialog */}
        <Dialog
          open={this.state.videoDialogOpen}
          onClose={this.handleCloseVideoDialog}
          maxWidth="md"
          fullWidth
          PaperProps={{
            sx: {
              borderRadius: 3,
              boxShadow: '0 8px 32px rgba(0, 0, 0, 0.12)',
            }
          }}
        >
          <DialogTitle
            sx={{
              backgroundColor: '#25476a',
              color: '#ffffff',
              display: 'flex',
              justifyContent: 'space-between',
              alignItems: 'center',
              py: 2.5,
              px: 3,
            }}
          >
            <Box sx={{ display: 'flex', alignItems: 'center' }}>
              <VideoLibrary sx={{ mr: 1.5, fontSize: 28 }} />
              <Typography variant="h6" sx={{ fontWeight: 600 }}>
                {this.state.selectedCourseForVideo?.introductionVideo ? 'Edit' : 'Add'} Introduction Video
              </Typography>
            </Box>
            <IconButton
              onClick={this.handleCloseVideoDialog}
              size="small"
              sx={{
                color: '#ffffff',
                '&:hover': {
                  backgroundColor: 'rgba(255, 255, 255, 0.1)',
                },
              }}
            >
              <Close />
            </IconButton>
          </DialogTitle>

          <DialogContent sx={{ pt: 3, px: 3, pb: 2 }}>
            {this.state.selectedCourseForVideo && (
              <Box>
                <Box sx={{ mb: 3, p: 2, bgcolor: '#f8fafc', borderRadius: 2 }}>
                  <Typography variant="body2" sx={{ color: '#64748b', mb: 0.5 }}>
                    Course:
                  </Typography>
                  <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 600 }}>
                    {this.state.selectedCourseForVideo.title}
                  </Typography>
                </Box>

                {/* File Upload Area */}
                {!videoPreviewUrls[this.state.selectedCourseForVideo.id] && !this.state.selectedCourseForVideo.introductionVideo ? (
                  <VideoUploadDragger
                    accept="video/*"
                    maxSize={500}
                    onUpload={(file) => {
                      const previewUrl = URL.createObjectURL(file);
                      this.setState({
                        uploadedVideoFiles: {
                          ...this.state.uploadedVideoFiles,
                          [this.state.selectedCourseForVideo!.id]: file
                        },
                        videoPreviewUrls: {
                          ...this.state.videoPreviewUrls,
                          [this.state.selectedCourseForVideo!.id]: previewUrl
                        }
                      });
                    }}
                    icon={<CloudUpload sx={{ fontSize: 64, color: '#38aae1', mb: 2 }} />}
                    title="Upload Introduction Video"
                    description="Click to browse or drag and drop your video file"
                    helpText="Supported formats: MP4, MOV, AVI, WebM (Max 500MB)"
                    style={{
                      border: '2px dashed #38aae1',
                      borderRadius: '12px',
                      padding: '32px',
                    }}
                  />
                ) : (
                  <Box>
                    {/* Video Preview */}
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600, mb: 2 }}>
                      Video Preview:
                    </Typography>
                    <Box sx={{ mb: 2 }}>
                      <VidstackPlayer
                        src={videoPreviewUrls[this.state.selectedCourseForVideo.id] || this.state.selectedCourseForVideo.introductionVideo}
                        title="Course Introduction Video"
                        aspectRatio="16/9"
                        controls={true}
                      />
                    </Box>

                    {/* Video Info */}
                    {uploadedVideoFiles[this.state.selectedCourseForVideo.id] && (
                      <Box sx={{ mb: 2, p: 2, bgcolor: '#f8fafc', borderRadius: 2 }}>
                        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                          <Box sx={{ display: 'flex', alignItems: 'center', flex: 1 }}>
                            <PlayCircleOutline sx={{ color: '#38aae1', mr: 1 }} />
                            <Box>
                              <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                                {uploadedVideoFiles[this.state.selectedCourseForVideo.id].name}
                              </Typography>
                              <Typography variant="caption" sx={{ color: '#64748b' }}>
                                {(uploadedVideoFiles[this.state.selectedCourseForVideo.id].size / (1024 * 1024)).toFixed(2)} MB
                              </Typography>
                            </Box>
                          </Box>
                          <IconButton
                            onClick={this.handleRemoveUploadedVideo}
                            size="small"
                            sx={{ color: '#ef4444' }}
                          >
                            <DeleteOutline />
                          </IconButton>
                        </Box>
                      </Box>
                    )}

                    {/* Change Video Button */}
                    <Box sx={{ position: 'relative' }}>
                      <input
                        type="file"
                        accept="video/*"
                        style={{ display: 'none' }}
                        id={`change-video-${this.state.selectedCourseForVideo.id}`}
                        onChange={(e) => {
                          const file = e.target.files?.[0];
                          if (file) {
                            const isVideo = file.type.startsWith('video/');
                            if (!isVideo) {
                              alert('You can only upload video files!');
                              return;
                            }
                            const isLt500M = file.size / 1024 / 1024 < 500;
                            if (!isLt500M) {
                              alert('Video must be smaller than 500MB!');
                              return;
                            }
                            
                            const previewUrl = URL.createObjectURL(file);
                            this.setState({
                              uploadedVideoFiles: {
                                ...this.state.uploadedVideoFiles,
                                [this.state.selectedCourseForVideo!.id]: file
                              },
                              videoPreviewUrls: {
                                ...this.state.videoPreviewUrls,
                                [this.state.selectedCourseForVideo!.id]: previewUrl
                              }
                            });
                          }
                        }}
                      />
                      <Button
                        variant="outlined"
                        startIcon={<CloudUpload />}
                        fullWidth
                        onClick={() => document.getElementById(`change-video-${this.state.selectedCourseForVideo!.id}`)?.click()}
                        sx={{
                          borderColor: '#38aae1',
                          color: '#38aae1',
                          textTransform: 'none',
                          fontWeight: 600,
                          mb: 2,
                          '&:hover': {
                            borderColor: '#2c8fc9',
                            backgroundColor: 'rgba(56, 170, 225, 0.05)',
                          }
                        }}
                      >
                        Change Video
                      </Button>
                    </Box>
                  </Box>
                )}

                <Box
                  sx={{
                    backgroundColor: '#e3f2fd',
                    borderLeft: '4px solid #38aae1',
                    p: 2,
                    borderRadius: 1,
                    mt: 3,
                  }}
                >
                  <Typography 
                    variant="body2" 
                    sx={{ 
                      color: '#25476a',
                      fontSize: '0.8125rem',
                      lineHeight: 1.6,
                    }}
                  >
                    <strong>Tip:</strong> The introduction video should give students an overview of what they'll learn in this course. 
                    Keep it concise (2-5 minutes) and engaging. Ensure good audio and video quality.
                  </Typography>
                </Box>
              </Box>
            )}
          </DialogContent>

          <DialogActions sx={{ px: 3, pb: 3, pt: 2 }}>
            <Button
              onClick={this.handleCloseVideoDialog}
              variant="outlined"
              sx={{
                borderColor: '#e2e8f0',
                color: '#64748b',
                borderRadius: 2,
                px: 3,
                textTransform: 'none',
                fontWeight: 600,
                borderWidth: '2px',
                '&:hover': {
                  borderColor: '#cbd5e1',
                  backgroundColor: '#f8fafc',
                  borderWidth: '2px',
                },
              }}
            >
              Cancel
            </Button>
            <Button
              onClick={this.handleSaveVideoFromDialog}
              variant="contained"
              startIcon={<VideoLibrary />}
              sx={{
                backgroundColor: '#38aae1',
                color: '#ffffff',
                borderRadius: 2,
                px: 3,
                textTransform: 'none',
                fontWeight: 600,
                boxShadow: '0 2px 8px rgba(56, 170, 225, 0.3)',
                '&:hover': {
                  backgroundColor: '#2c8fc9',
                  boxShadow: '0 4px 12px rgba(56, 170, 225, 0.4)',
                },
              }}
            >
              Save Video
            </Button>
          </DialogActions>
        </Dialog>

        {/* Video Preview Dialog */}
        <Dialog
          open={this.state.previewDialogOpen}
          onClose={this.handleClosePreviewDialog}
          maxWidth="lg"
          fullWidth
          PaperProps={{
            sx: {
              borderRadius: 3,
              boxShadow: '0 8px 32px rgba(0, 0, 0, 0.12)',
            }
          }}
        >
          <DialogTitle
            sx={{
              backgroundColor: '#25476a',
              color: '#ffffff',
              display: 'flex',
              justifyContent: 'space-between',
              alignItems: 'center',
              py: 2.5,
              px: 3,
            }}
          >
            <Box sx={{ display: 'flex', alignItems: 'center' }}>
              <PlayCircleOutline sx={{ mr: 1.5, fontSize: 28 }} />
              <Box>
                <Typography variant="h6" sx={{ fontWeight: 600 }}>
                  Preview Introduction Video
                </Typography>
                {this.state.previewCourse && (
                  <Typography variant="caption" sx={{ color: 'rgba(255, 255, 255, 0.8)' }}>
                    {this.state.previewCourse.title}
                  </Typography>
                )}
              </Box>
            </Box>
            <IconButton
              onClick={this.handleClosePreviewDialog}
              size="small"
              sx={{
                color: '#ffffff',
                '&:hover': {
                  backgroundColor: 'rgba(255, 255, 255, 0.1)',
                },
              }}
            >
              <Close />
            </IconButton>
          </DialogTitle>

          <DialogContent sx={{ p: 0, bgcolor: '#000' }}>
            {this.state.previewCourse?.introductionVideo ? (
              <Box sx={{ width: '100%' }}>
                <VidstackPlayer
                  src={this.state.previewCourse.introductionVideo}
                  title="Course Introduction Video"
                  aspectRatio="16/9"
                  controls={true}
                  autoplay={true}
                />
              </Box>
            ) : (
              <Box sx={{ 
                p: 8, 
                textAlign: 'center', 
                bgcolor: '#f8fafc',
              }}>
                <VideoLibrary sx={{ fontSize: 64, color: '#cbd5e1', mb: 2 }} />
                <Typography variant="h6" sx={{ color: '#64748b', mb: 1 }}>
                  No Video Available
                </Typography>
                <Typography variant="body2" sx={{ color: '#94a3b8' }}>
                  This course doesn't have an introduction video yet.
                </Typography>
              </Box>
            )}
          </DialogContent>

          {this.state.previewCourse?.introductionVideoName && (
            <Box sx={{ px: 3, py: 2, bgcolor: '#f8fafc', borderTop: '1px solid #e2e8f0' }}>
              <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
                <Box sx={{ display: 'flex', alignItems: 'center' }}>
                  <PlayCircleOutline sx={{ color: '#38aae1', mr: 1.5, fontSize: 20 }} />
                  <Box>
                    <Typography variant="body2" sx={{ color: '#25476a', fontWeight: 600 }}>
                      {this.state.previewCourse.introductionVideoName}
                    </Typography>
                    <Typography variant="caption" sx={{ color: '#64748b' }}>
                      Introduction Video
                    </Typography>
                  </Box>
                </Box>
                <Button
                  onClick={() => {
                    this.handleClosePreviewDialog();
                    this.handleOpenVideoDialog(this.state.previewCourse!);
                  }}
                  variant="outlined"
                  size="small"
                  startIcon={<Edit />}
                  sx={{
                    borderColor: '#38aae1',
                    color: '#38aae1',
                    textTransform: 'none',
                    fontWeight: 600,
                    '&:hover': {
                      borderColor: '#2c8fc9',
                      backgroundColor: 'rgba(56, 170, 225, 0.05)',
                    }
                  }}
                >
                  Change Video
                </Button>
              </Box>
            </Box>
          )}
        </Dialog>
      </Box>
    );
  }
}