import React, { useState } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  Box,
  Typography,
  IconButton,
  Chip,
  Grid,
  Paper,
  FormControlLabel,
  Checkbox,
} from '@mui/material';
import {
  Close,
  AccessTime,
  CalendarToday,
} from '@mui/icons-material';

const DAYS_OF_WEEK = [
  { short: 'Mon', full: 'Monday' },
  { short: 'Tue', full: 'Tuesday' },
  { short: 'Wed', full: 'Wednesday' },
  { short: 'Thu', full: 'Thursday' },
  { short: 'Fri', full: 'Friday' },
  { short: 'Sat', full: 'Saturday' },
  { short: 'Sun', full: 'Sunday' },
];

const SESSION_DURATIONS = [
  { label: '30 minutes', value: 30, icon: '⏰' },
  { label: '1 hour', value: 60, icon: '⏱️' },
  { label: '1.5 hours', value: 90, icon: '⏲️' },
  { label: '2 hours', value: 120, icon: '⏰' },
];

const TIME_SLOTS = [
  '08:00', '08:30', '09:00', '09:30', '10:00', '10:30',
  '11:00', '11:30', '12:00', '12:30', '13:00', '13:30',
  '14:00', '14:30', '15:00', '15:30', '16:00', '16:30',
  '17:00', '17:30', '18:00', '18:30', '19:00', '19:30',
  '20:00', '20:30', '21:00', '21:30',
];

export function AvailabilityScheduleDialog({ open, onClose, mentor }) {
  const [selectedDuration, setSelectedDuration] = useState(60);
  const [selectedDays, setSelectedDays] = useState(['Tue']);
  const [timeSlots, setTimeSlots] = useState({
    Monday: [],
    Tuesday: ['09:00', '10:00', '14:00', '15:00'],
    Wednesday: [],
    Thursday: [],
    Friday: [],
    Saturday: [],
    Sunday: [],
  });

  const handleDayToggle = (dayFull) => {
    const dayShort = DAYS_OF_WEEK.find(d => d.full === dayFull)?.short;
    setSelectedDays(prev => 
      prev.includes(dayShort) 
        ? prev.filter(d => d !== dayShort)
        : [...prev, dayShort]
    );
  };

  const handleTimeSlotToggle = (day, time) => {
    setTimeSlots(prev => ({
      ...prev,
      [day]: prev[day].includes(time)
        ? prev[day].filter(t => t !== time)
        : [...prev[day], time].sort()
    }));
  };

  const handleSave = () => {
    const scheduleData = {
      mentorId: mentor?.id,
      duration: selectedDuration,
      availability: timeSlots,
      activeDays: selectedDays,
    };
    console.log('Saving availability schedule:', scheduleData);
    onClose();
  };

  const getTotalSlotsConfigured = () => {
    return Object.values(timeSlots).reduce((total, slots) => total + slots.length, 0);
  };

  return (
    <Dialog 
      open={open} 
      onClose={onClose} 
      maxWidth="lg" 
      fullWidth
      PaperProps={{
        sx: { 
          borderRadius: 3,
          minHeight: '80vh',
          maxHeight: '90vh',
        }
      }}
    >
      <DialogTitle sx={{ p: 3, pb: 2 }}>
        <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
          <Box sx={{ display: 'flex', alignItems: 'center' }}>
            <CalendarToday sx={{ color: '#38aae1', mr: 2, fontSize: 28 }} />
            <Box>
              <Typography variant="h5" sx={{ fontWeight: 600, color: '#25476a' }}>
                Set Availability Schedule
              </Typography>
              <Typography variant="body2" sx={{ color: '#64748b', mt: 0.5 }}>
                Configure when {mentor?.name || 'Alex Johnson'} is available for student consultations
              </Typography>
            </Box>
          </Box>
          <IconButton onClick={onClose} sx={{ color: '#64748b' }}>
            <Close />
          </IconButton>
        </Box>
      </DialogTitle>

      <DialogContent sx={{ p: 3, pt: 1 }}>
        {/* Session Duration */}
        <Box sx={{ mb: 4 }}>
          <Typography variant="h6" sx={{ mb: 2, color: '#25476a', fontWeight: 600 }}>
            Session Duration
          </Typography>
          <Box sx={{ display: 'flex', gap: 1, flexWrap: 'wrap' }}>
            {SESSION_DURATIONS.map((duration) => (
              <Chip
                key={duration.value}
                icon={<AccessTime />}
                label={duration.label}
                onClick={() => setSelectedDuration(duration.value)}
                variant={selectedDuration === duration.value ? 'filled' : 'outlined'}
                color={selectedDuration === duration.value ? 'primary' : 'default'}
                sx={{
                  py: 2,
                  px: 1,
                  fontSize: '0.9rem',
                  fontWeight: 500,
                  backgroundColor: selectedDuration === duration.value ? '#38aae1' : 'transparent',
                  color: selectedDuration === duration.value ? 'white' : '#64748b',
                  borderColor: selectedDuration === duration.value ? '#38aae1' : '#e2e8f0',
                  '&:hover': {
                    backgroundColor: selectedDuration === duration.value ? '#2e8bc4' : '#f1f5f9',
                  },
                }}
              />
            ))}
          </Box>
        </Box>

        {/* Available Days */}
        <Box sx={{ mb: 4 }}>
          <Typography variant="h6" sx={{ mb: 2, color: '#25476a', fontWeight: 600 }}>
            Available Days
          </Typography>
          <Box sx={{ display: 'flex', gap: 1, flexWrap: 'wrap' }}>
            {DAYS_OF_WEEK.map((day) => (
              <Button
                key={day.short}
                variant={selectedDays.includes(day.short) ? 'contained' : 'outlined'}
                onClick={() => handleDayToggle(day.full)}
                sx={{
                  minWidth: 80,
                  py: 1.5,
                  backgroundColor: selectedDays.includes(day.short) ? '#38aae1' : 'transparent',
                  borderColor: selectedDays.includes(day.short) ? '#38aae1' : '#e2e8f0',
                  color: selectedDays.includes(day.short) ? 'white' : '#64748b',
                  '&:hover': {
                    backgroundColor: selectedDays.includes(day.short) ? '#2e8bc4' : '#f1f5f9',
                  },
                }}
              >
                <Box sx={{ textAlign: 'center' }}>
                  <Typography variant="body2" sx={{ fontWeight: 600 }}>
                    {day.short}
                  </Typography>
                  <Typography variant="caption">
                    {day.short}
                  </Typography>
                </Box>
              </Button>
            ))}
          </Box>
        </Box>

        {/* Configure Time Slots */}
        <Box>
          <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
            <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 600 }}>
              Configure Time Slots
            </Typography>
            <Chip
              label={`${getTotalSlotsConfigured()} slots configured`}
              size="small"
              sx={{
                backgroundColor: '#fff7ed',
                color: '#ea580c',
                fontWeight: 500,
              }}
            />
          </Box>

          {/* Time Slots for Selected Days */}
          {selectedDays.map(dayShort => {
            const dayFull = DAYS_OF_WEEK.find(d => d.short === dayShort)?.full;
            const daySlots = timeSlots[dayFull] || [];
            
            return (
              <Paper 
                key={dayFull}
                sx={{ 
                  p: 3, 
                  mb: 3, 
                  borderRadius: 2,
                  border: '1px solid #e2e8f0',
                  backgroundColor: '#fafbfc',
                }}
              >
                <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
                  <Typography variant="h6" sx={{ color: '#25476a', fontWeight: 600 }}>
                    {dayFull}
                  </Typography>
                  <Chip
                    label={`${daySlots.length} slots`}
                    size="small"
                    variant="outlined"
                    sx={{
                      backgroundColor: '#38aae1',
                      color: 'white',
                      borderColor: '#38aae1',
                      fontWeight: 500,
                    }}
                  />
                </Box>
                
                <Typography variant="body2" sx={{ mb: 2, color: '#64748b' }}>
                  Add time slots ({selectedDuration} minutes each):
                </Typography>
                
                <Grid container spacing={1}>
                  {TIME_SLOTS.map((time) => (
                    <Grid item xs={6} sm={4} md={3} lg={2} key={time}>
                      <Button
                        variant={daySlots.includes(time) ? 'contained' : 'outlined'}
                        onClick={() => handleTimeSlotToggle(dayFull, time)}
                        fullWidth
                        size="small"
                        sx={{
                          py: 1,
                          fontSize: '0.8rem',
                          backgroundColor: daySlots.includes(time) ? '#38aae1' : 'white',
                          borderColor: daySlots.includes(time) ? '#38aae1' : '#e2e8f0',
                          color: daySlots.includes(time) ? 'white' : '#64748b',
                          '&:hover': {
                            backgroundColor: daySlots.includes(time) ? '#2e8bc4' : '#f8fafc',
                          },
                        }}
                      >
                        {time}
                      </Button>
                    </Grid>
                  ))}
                </Grid>
              </Paper>
            );
          })}

          {selectedDays.length === 0 && (
            <Paper sx={{ p: 4, textAlign: 'center', backgroundColor: '#f8fafc' }}>
              <Typography variant="body1" sx={{ color: '#64748b' }}>
                Please select at least one day to configure time slots
              </Typography>
            </Paper>
          )}
        </Box>
      </DialogContent>

      <DialogActions sx={{ p: 3, pt: 2, gap: 2 }}>
        <Button 
          onClick={onClose}
          variant="outlined"
          sx={{
            borderColor: '#e2e8f0',
            color: '#64748b',
            '&:hover': {
              borderColor: '#cbd5e1',
              backgroundColor: '#f8fafc',
            },
          }}
        >
          Cancel
        </Button>
        <Button 
          onClick={handleSave}
          variant="contained"
          disabled={getTotalSlotsConfigured() === 0}
          sx={{
            backgroundColor: '#38aae1',
            '&:hover': {
              backgroundColor: '#2e8bc4',
            },
            '&:disabled': {
              backgroundColor: '#e2e8f0',
              color: '#94a3b8',
            },
          }}
        >
          Save Availability Schedule
        </Button>
      </DialogActions>
    </Dialog>
  );
}