import React, { Component } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  Box,
  Typography,
  Checkbox,
  FormGroup,
  Avatar,
  Chip,
  Divider,
  Alert,
  TextField,
  InputAdornment,
  Stepper,
  Step,
  StepLabel,
} from '@mui/material';
import {
  Close,
  Search,
  PersonAdd,
  CheckCircle,
  ArrowForward,
  ArrowBack,
  Class as ClassIcon,
  MenuBook,
} from '@mui/icons-material';

interface Mentor {
  id: string;
  name: string;
  specialization: string;
  status: 'Active' | 'Inactive';
  rating: number;
  subjects: string[];
  avatar?: string;
}

interface Course {
  id: string;
  title: string;
  description: string;
  category: string;
  supervisors?: string[];
}

interface Class {
  id: number;
  name: string;
  coursesCount: number;
  students: number;
  status: 'Published' | 'Draft';
  courses: Course[];
}

interface AssignMentorsToCourseDialogProps {
  open: boolean;
  onClose: () => void;
  onAssign: (classId: number, courseIds: string[], mentorIds: string[]) => void;
  mentors: Mentor[];
  classes: Class[];
}

interface AssignMentorsToCourseDialogState {
  activeStep: number;
  selectedClassId: number | null;
  selectedCourseIds: string[];
  selectedMentorIds: string[];
  searchTerm: string;
}

export class AssignMentorsToCourseDialog extends Component<
  AssignMentorsToCourseDialogProps,
  AssignMentorsToCourseDialogState
> {
  constructor(props: AssignMentorsToCourseDialogProps) {
    super(props);
    this.state = {
      activeStep: 0,
      selectedClassId: null,
      selectedCourseIds: [],
      selectedMentorIds: [],
      searchTerm: '',
    };
  }

  componentDidUpdate(prevProps: AssignMentorsToCourseDialogProps) {
    if (this.props.open && !prevProps.open) {
      // Reset state when dialog opens
      this.setState({
        activeStep: 0,
        selectedClassId: null,
        selectedCourseIds: [],
        selectedMentorIds: [],
        searchTerm: '',
      });
    }
  }

  handleMentorToggle = (mentorId: string) => {
    this.setState(prevState => {
      const isSelected = prevState.selectedMentorIds.includes(mentorId);
      return {
        selectedMentorIds: isSelected
          ? prevState.selectedMentorIds.filter(id => id !== mentorId)
          : [...prevState.selectedMentorIds, mentorId]
      };
    });
  };

  handleSelectAll = () => {
    const filteredMentors = this.getFilteredMentors();
    
    if (this.state.selectedMentorIds.length === filteredMentors.length) {
      this.setState({ selectedMentorIds: [] });
    } else {
      this.setState({ 
        selectedMentorIds: filteredMentors.map(m => m.id) 
      });
    }
  };

  handleNext = () => {
    this.setState(prevState => ({
      activeStep: prevState.activeStep + 1
    }));
  };

  handleBack = () => {
    this.setState(prevState => ({
      activeStep: prevState.activeStep - 1
    }));
  };

  handleCourseToggle = (courseId: string) => {
    this.setState(prevState => {
      const isSelected = prevState.selectedCourseIds.includes(courseId);
      return {
        selectedCourseIds: isSelected
          ? prevState.selectedCourseIds.filter(id => id !== courseId)
          : [...prevState.selectedCourseIds, courseId]
      };
    });
  };

  handleSelectAllCourses = () => {
    const { classes } = this.props;
    const { selectedClassId } = this.state;
    const selectedClass = classes.find(c => c.id === selectedClassId);
    const courses = selectedClass?.courses || [];
    
    if (this.state.selectedCourseIds.length === courses.length) {
      this.setState({ selectedCourseIds: [] });
    } else {
      this.setState({ 
        selectedCourseIds: courses.map(c => c.id) 
      });
    }
  };

  handleAssign = () => {
    const { selectedClassId, selectedCourseIds, selectedMentorIds } = this.state;
    if (selectedClassId !== null && selectedCourseIds.length > 0 && selectedMentorIds.length > 0) {
      this.props.onAssign(selectedClassId, selectedCourseIds, selectedMentorIds);
      this.handleClose();
    }
  };

  handleClose = () => {
    this.setState({
      activeStep: 0,
      selectedClassId: null,
      selectedCourseIds: [],
      selectedMentorIds: [],
      searchTerm: '',
    });
    this.props.onClose();
  };

  getFilteredMentors = () => {
    const { mentors } = this.props;
    const { searchTerm } = this.state;
    
    return mentors.filter(mentor => 
      mentor.status === 'Active' && (
        mentor.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        mentor.specialization.toLowerCase().includes(searchTerm.toLowerCase())
      )
    );
  };

  renderClassSelection = () => {
    const { classes } = this.props;
    const { selectedClassId } = this.state;

    return (
      <Box>
        <Typography variant="subtitle2" sx={{ mb: 2, fontWeight: 600, color: '#25476a' }}>
          Step 1: Select a Class
        </Typography>
        <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2 }}>
          {classes.map(classItem => (
            <Box
              key={classItem.id}
              onClick={() => this.setState({ selectedClassId: classItem.id, selectedCourseIds: [] })}
              sx={{
                p: 3,
                border: '2px solid',
                borderColor: selectedClassId === classItem.id ? '#38aae1' : '#e2e8f0',
                borderRadius: 3,
                bgcolor: selectedClassId === classItem.id ? '#f0f9ff' : 'white',
                cursor: 'pointer',
                transition: 'all 0.2s',
                '&:hover': {
                  borderColor: '#38aae1',
                  transform: 'translateY(-2px)',
                  boxShadow: '0 4px 12px rgba(56, 170, 225, 0.2)',
                }
              }}
            >
              <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
                <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                  <Avatar sx={{ bgcolor: '#38aae1', width: 48, height: 48 }}>
                    <ClassIcon />
                  </Avatar>
                  <Box>
                    <Typography variant="h6" sx={{ fontWeight: 700, color: '#25476a' }}>
                      {classItem.name}
                    </Typography>
                    <Box sx={{ display: 'flex', gap: 1, mt: 0.5 }}>
                      <Chip 
                        label={`${classItem.coursesCount} Courses`}
                        size="small"
                        sx={{ fontSize: '0.75rem', height: 24 }}
                      />
                      <Chip 
                        label={`${classItem.students} Students`}
                        size="small"
                        color="primary"
                        variant="outlined"
                        sx={{ fontSize: '0.75rem', height: 24 }}
                      />
                      <Chip 
                        label={classItem.status}
                        size="small"
                        color={classItem.status === 'Published' ? 'success' : 'warning'}
                        sx={{ fontSize: '0.75rem', height: 24 }}
                      />
                    </Box>
                  </Box>
                </Box>
                {selectedClassId === classItem.id && (
                  <CheckCircle sx={{ color: '#38aae1', fontSize: 32 }} />
                )}
              </Box>
            </Box>
          ))}
        </Box>
      </Box>
    );
  };

  renderCourseSelection = () => {
    const { classes } = this.props;
    const { selectedClassId, selectedCourseIds } = this.state;

    const selectedClass = classes.find(c => c.id === selectedClassId);
    const courses = selectedClass?.courses || [];

    return (
      <Box>
        <Box sx={{ mb: 3, display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
          <Box>
            <Typography variant="subtitle2" sx={{ fontWeight: 600, color: '#25476a', mb: 0.5 }}>
              Step 2: Select Courses from {selectedClass?.name}
            </Typography>
            <Typography variant="body2" color="text.secondary">
              {selectedCourseIds.length} of {courses.length} courses selected
            </Typography>
          </Box>
          {courses.length > 0 && (
            <Button
              size="small"
              onClick={this.handleSelectAllCourses}
              sx={{ textTransform: 'none' }}
            >
              {selectedCourseIds.length === courses.length ? 'Deselect All' : 'Select All'}
            </Button>
          )}
        </Box>
        <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2, maxHeight: 400, overflowY: 'auto' }}>
          {courses.length === 0 ? (
            <Box sx={{ textAlign: 'center', py: 4 }}>
              <Typography color="text.secondary">
                No courses available in this class
              </Typography>
            </Box>
          ) : (
            courses.map(course => (
              <Box
                key={course.id}
                onClick={() => this.handleCourseToggle(course.id)}
                sx={{
                  p: 2.5,
                  border: '2px solid',
                  borderColor: selectedCourseIds.includes(course.id) ? '#38aae1' : '#e2e8f0',
                  borderRadius: 2,
                  bgcolor: selectedCourseIds.includes(course.id) ? '#f0f9ff' : 'white',
                  cursor: 'pointer',
                  transition: 'all 0.2s',
                  '&:hover': {
                    borderColor: '#38aae1',
                    bgcolor: '#f8fafc',
                  }
                }}
              >
                <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                  <Checkbox
                    checked={selectedCourseIds.includes(course.id)}
                    onChange={() => this.handleCourseToggle(course.id)}
                    sx={{ p: 0 }}
                  />
                  <Avatar sx={{ bgcolor: '#feb139', width: 40, height: 40 }}>
                    <MenuBook />
                  </Avatar>
                  <Box sx={{ flex: 1 }}>
                    <Typography variant="body1" sx={{ fontWeight: 600, color: '#25476a' }}>
                      {course.title}
                    </Typography>
                    <Typography variant="body2" color="text.secondary" sx={{ mt: 0.5, fontSize: '0.875rem' }}>
                      {course.description}
                    </Typography>
                    {course.supervisors && course.supervisors.length > 0 && (
                      <Chip 
                        label={`${course.supervisors.length} Supervisor(s)`}
                        size="small"
                        sx={{ 
                          mt: 1,
                          fontSize: '0.7rem',
                          height: 22,
                          bgcolor: '#dcfce7',
                          color: '#166534',
                        }}
                      />
                    )}
                  </Box>
                  {selectedCourseIds.includes(course.id) && (
                    <CheckCircle sx={{ color: '#38aae1', fontSize: 28 }} />
                  )}
                </Box>
              </Box>
            ))
          )}
        </Box>
      </Box>
    );
  };

  renderMentorSelection = () => {
    const { selectedMentorIds, searchTerm, selectedClassId, selectedCourseIds } = this.state;
    const { classes } = this.props;
    
    const filteredMentors = this.getFilteredMentors();
    const selectedClass = classes.find(c => c.id === selectedClassId);
    const selectedCourses = selectedClass?.courses.filter(c => selectedCourseIds.includes(c.id)) || [];

    return (
      <Box>
        <Box sx={{ mb: 3 }}>
          <Typography variant="subtitle2" sx={{ fontWeight: 600, color: '#25476a', mb: 1 }}>
            Step 3: Select Technical Mentors
          </Typography>
          <Box sx={{ p: 2, bgcolor: '#f1f5f9', borderRadius: 2 }}>
            <Typography variant="body2" color="text.secondary" sx={{ mb: 1 }}>
              <strong>Class:</strong> {selectedClass?.name}
            </Typography>
            <Typography variant="body2" color="text.secondary">
              <strong>Courses ({selectedCourses.length}):</strong>
            </Typography>
            <Box sx={{ mt: 1, display: 'flex', flexWrap: 'wrap', gap: 1 }}>
              {selectedCourses.map(course => (
                <Chip
                  key={course.id}
                  label={course.title}
                  size="small"
                  sx={{
                    bgcolor: '#e0f2fe',
                    color: '#0369a1',
                    fontSize: '0.75rem',
                  }}
                />
              ))}
            </Box>
          </Box>
        </Box>

        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 2 }}>
          <Typography variant="body2" sx={{ fontWeight: 600, color: '#25476a' }}>
            {selectedMentorIds.length} mentor(s) selected
          </Typography>
          <Button
            size="small"
            onClick={this.handleSelectAll}
            sx={{ textTransform: 'none' }}
          >
            {selectedMentorIds.length === filteredMentors.length ? 'Deselect All' : 'Select All'}
          </Button>
        </Box>

        {/* Search */}
        <TextField
          fullWidth
          placeholder="Search mentors..."
          value={searchTerm}
          onChange={(e) => this.setState({ searchTerm: e.target.value })}
          size="small"
          sx={{ mb: 2 }}
          InputProps={{
            startAdornment: (
              <InputAdornment position="start">
                <Search />
              </InputAdornment>
            ),
          }}
        />

        {/* Mentor List */}
        <Box sx={{ 
          maxHeight: 350, 
          overflowY: 'auto',
          border: '1px solid #e2e8f0',
          borderRadius: 2,
          p: 2,
        }}>
          {filteredMentors.length === 0 ? (
            <Box sx={{ textAlign: 'center', py: 4 }}>
              <Typography color="text.secondary">
                No active mentors found
              </Typography>
            </Box>
          ) : (
            <FormGroup>
              {filteredMentors.map(mentor => (
                <Box
                  key={mentor.id}
                  sx={{
                    display: 'flex',
                    alignItems: 'center',
                    p: 2,
                    mb: 1,
                    bgcolor: selectedMentorIds.includes(mentor.id) ? '#f0f9ff' : 'white',
                    border: '1px solid',
                    borderColor: selectedMentorIds.includes(mentor.id) ? '#38aae1' : '#e2e8f0',
                    borderRadius: 2,
                    transition: 'all 0.2s',
                    cursor: 'pointer',
                    '&:hover': {
                      bgcolor: '#f8fafc',
                      borderColor: '#38aae1',
                    }
                  }}
                  onClick={() => this.handleMentorToggle(mentor.id)}
                >
                  <Checkbox
                    checked={selectedMentorIds.includes(mentor.id)}
                    onChange={() => this.handleMentorToggle(mentor.id)}
                    sx={{ mr: 2 }}
                  />
                  <Avatar 
                    sx={{ 
                      mr: 2, 
                      bgcolor: '#38aae1',
                      width: 40,
                      height: 40,
                    }}
                  >
                    {mentor.name.charAt(0)}
                  </Avatar>
                  <Box sx={{ flexGrow: 1 }}>
                    <Typography variant="body1" sx={{ fontWeight: 600, color: '#25476a' }}>
                      {mentor.name}
                    </Typography>
                    <Box sx={{ display: 'flex', gap: 1, alignItems: 'center', mt: 0.5 }}>
                      <Chip 
                        label={mentor.specialization}
                        size="small"
                        sx={{ 
                          fontSize: '0.75rem',
                          height: 24,
                          bgcolor: '#e0f2fe',
                          color: '#0369a1',
                        }}
                      />
                      <Typography variant="caption" color="text.secondary">
                        ⭐ {mentor.rating.toFixed(1)}
                      </Typography>
                    </Box>
                  </Box>
                  {selectedMentorIds.includes(mentor.id) && (
                    <CheckCircle sx={{ color: '#38aae1', ml: 2 }} />
                  )}
                </Box>
              ))}
            </FormGroup>
          )}
        </Box>
      </Box>
    );
  };

  render() {
    const { open } = this.props;
    const { activeStep, selectedClassId, selectedCourseIds, selectedMentorIds } = this.state;

    const steps = ['Select Class', 'Select Courses', 'Assign Mentors'];

    return (
      <Dialog 
        open={open} 
        onClose={this.handleClose}
        maxWidth="md"
        fullWidth
        PaperProps={{
          sx: {
            borderRadius: 3,
            boxShadow: '0 8px 32px rgba(0,0,0,0.12)',
          }
        }}
      >
        <DialogTitle sx={{ pb: 2, bgcolor: '#f8fafc', borderBottom: '1px solid #e2e8f0' }}>
          <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
            <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
              <Avatar sx={{ bgcolor: '#38aae1', width: 48, height: 48 }}>
                <PersonAdd />
              </Avatar>
              <Box>
                <Typography variant="h5" sx={{ fontWeight: 700, color: '#25476a' }}>
                  Assign Course Supervisors
                </Typography>
                <Typography variant="body2" color="text.secondary">
                  Assign mentors to supervise specific courses in a class
                </Typography>
              </Box>
            </Box>
            <Button
              onClick={this.handleClose}
              sx={{ minWidth: 'auto', color: '#64748b' }}
            >
              <Close />
            </Button>
          </Box>
        </DialogTitle>

        <DialogContent sx={{ pt: 3 }}>
          {/* Stepper */}
          <Stepper activeStep={activeStep} sx={{ mb: 4 }}>
            {steps.map((label) => (
              <Step key={label}>
                <StepLabel>{label}</StepLabel>
              </Step>
            ))}
          </Stepper>

          {/* Step Content */}
          {activeStep === 0 && this.renderClassSelection()}
          {activeStep === 1 && this.renderCourseSelection()}
          {activeStep === 2 && this.renderMentorSelection()}

          {/* Info Alert */}
          {activeStep === 2 && selectedMentorIds.length > 0 && selectedCourseIds.length > 0 && (
            <Alert severity="info" sx={{ mt: 3 }}>
              <Typography variant="body2">
                Assigning <strong>{selectedMentorIds.length}</strong> mentor(s) to <strong>{selectedCourseIds.length}</strong> course(s) as supervisors
              </Typography>
            </Alert>
          )}
        </DialogContent>

        <DialogActions sx={{ px: 3, py: 2, bgcolor: '#f8fafc', borderTop: '1px solid #e2e8f0' }}>
          <Button
            onClick={this.handleClose}
            sx={{ 
              color: '#64748b',
              textTransform: 'none',
              px: 3,
            }}
          >
            Cancel
          </Button>
          <Box sx={{ flex: 1 }} />
          {activeStep > 0 && (
            <Button
              onClick={this.handleBack}
              startIcon={<ArrowBack />}
              sx={{ 
                textTransform: 'none',
                px: 3,
              }}
            >
              Back
            </Button>
          )}
          {activeStep < 2 ? (
            <Button
              onClick={this.handleNext}
              variant="contained"
              disabled={
                (activeStep === 0 && selectedClassId === null) ||
                (activeStep === 1 && selectedCourseIds.length === 0)
              }
              endIcon={<ArrowForward />}
              sx={{
                bgcolor: '#38aae1',
                '&:hover': { bgcolor: '#2e8bc4' },
                textTransform: 'none',
                px: 3,
              }}
            >
              Next
            </Button>
          ) : (
            <Button
              onClick={this.handleAssign}
              variant="contained"
              disabled={selectedMentorIds.length === 0}
              startIcon={<PersonAdd />}
              sx={{
                bgcolor: '#38aae1',
                '&:hover': { bgcolor: '#2e8bc4' },
                textTransform: 'none',
                px: 3,
                boxShadow: '0 4px 12px rgba(56, 170, 225, 0.3)',
              }}
            >
              Assign {selectedMentorIds.length > 0 && `(${selectedMentorIds.length})`}
            </Button>
          )}
        </DialogActions>
      </Dialog>
    );
  }
}
