import React, { Component } from 'react';
import {
  Dialog,
  DialogContent,
  Button,
  Box,
  Typography,
  TextField,
  Grid,
  IconButton,
  Card,
  CardContent,
  FormControl,
  Select,
  MenuItem,
  Chip,
  LinearProgress,
  Avatar,
  Stepper,
  Step,
  StepLabel,
  StepConnector,
  stepConnectorClasses,
  Tooltip,
  Fade,
  Collapse,
  Tabs,
  Tab,
} from '@mui/material';
import {
  Close,
  Delete,
  Add,
  Save,
  CheckCircle,
  PlayCircleOutline,
  Assignment,
  SportsEsports,
  Edit,
  RadioButtonUnchecked,
  ExpandMore,
  ExpandLess,
  VideoLibrary,
  MenuBook,
  Timeline,
  School,
} from '@mui/icons-material';
import { styled } from '@mui/material/styles';
import { VidstackPlayer } from './VidstackPlayer';

interface SessionData {
  id: string;
  title: string;
  duration: number;
  description: string;
  objectives: string[];
  introVideo: { title: string; duration: number; url: string };
  assignmentVideo: { title: string; duration: number; url: string };
  activityVideos: { title: string; duration: number; url: string }[];
  resources: { name: string; type: 'pdf' | 'doc' | 'ppt' | 'youtube'; url: string }[];
  isCompleted: boolean;
}

interface LessonPlan {
  id: string;
  title: string;
  duration: number;
  description: string;
  objectives: string[];
  introVideo: { title: string; duration: number; url: string };
  assignmentVideo: { title: string; duration: number; url: string };
  activityVideos: { title: string; duration: number; url: string }[];
  resources: { name: string; type: 'pdf' | 'doc' | 'ppt' | 'youtube'; url: string }[];
}

interface AllSessionsEditorProps {
  open: boolean;
  lesson: LessonPlan;
  onSave: (lesson: LessonPlan) => void;
  onClose: () => void;
}

interface AllSessionsEditorState {
  sessions: SessionData[];
  currentSessionId: string;
  activeTab: number;
  newObjective: string;
  newResource: { name: string; type: 'pdf' | 'doc' | 'ppt' | 'youtube'; url: string };
  expandedSections: { [key: string]: boolean };
}

const CustomStepConnector = styled(StepConnector)(({ theme }) => ({
  [`&.${stepConnectorClasses.alternativeLabel}`]: {
    top: 10,
    left: 'calc(-50% + 16px)',
    right: 'calc(50% + 16px)',
  },
  [`&.${stepConnectorClasses.active}`]: {
    [`& .${stepConnectorClasses.line}`]: {
      borderColor: '#38aae1',
      borderWidth: 2,
    },
  },
  [`&.${stepConnectorClasses.completed}`]: {
    [`& .${stepConnectorClasses.line}`]: {
      borderColor: '#4caf50',
      borderWidth: 2,
    },
  },
  [`& .${stepConnectorClasses.line}`]: {
    borderColor: '#e0e0e0',
    borderTopWidth: 1,
    borderRadius: 1,
  },
}));

const createDefaultSession = (sessionNumber: number): SessionData => ({
  id: `session-${sessionNumber}`,
  title: `Session ${sessionNumber}`,
  duration: 60,
  description: `Content for session ${sessionNumber}`,
  objectives: [],
  introVideo: { title: '', duration: 0, url: '' },
  assignmentVideo: { title: '', duration: 0, url: '' },
  activityVideos: [],
  resources: [],
  isCompleted: false
});

const convertLessonToSession = (lesson: LessonPlan, sessionNumber: number): SessionData => ({
  id: `session-${sessionNumber}`,
  title: lesson.title || `Session ${sessionNumber}`,
  duration: lesson.duration || 60,
  description: lesson.description || `Content for session ${sessionNumber}`,
  objectives: lesson.objectives || [],
  introVideo: lesson.introVideo || { title: '', duration: 0, url: '' },
  assignmentVideo: lesson.assignmentVideo || { title: '', duration: 0, url: '' },
  activityVideos: lesson.activityVideos || [],
  resources: lesson.resources || [],
  isCompleted: false
});

export class AllSessionsEditor extends Component<AllSessionsEditorProps, AllSessionsEditorState> {
  constructor(props: AllSessionsEditorProps) {
    super(props);
    
    const sessions = Array.from({ length: 12 }, (_, index) => {
      if (index === 0) {
        return convertLessonToSession(props.lesson, index + 1);
      }
      return createDefaultSession(index + 1);
    });

    this.state = {
      sessions,
      currentSessionId: 'session-1',
      activeTab: 0,
      newObjective: '',
      newResource: {
        name: '',
        type: 'pdf',
        url: ''
      },
      expandedSections: {
        overview: true,
        videos: false,
        resources: false
      }
    };
  }

  componentDidUpdate(prevProps: AllSessionsEditorProps) {
    if (prevProps.lesson !== this.props.lesson && this.props.open) {
      const sessions = Array.from({ length: 12 }, (_, index) => {
        if (index === 0) {
          return convertLessonToSession(this.props.lesson, index + 1);
        }
        return createDefaultSession(index + 1);
      });
      
      this.setState({ 
        sessions,
        currentSessionId: 'session-1'
      });
    }
  }

  getCurrentSession = (): SessionData => {
    return this.state.sessions.find(s => s.id === this.state.currentSessionId) || this.state.sessions[0];
  };

  updateCurrentSession = (updates: Partial<SessionData>) => {
    this.setState({
      sessions: this.state.sessions.map(session =>
        session.id === this.state.currentSessionId 
          ? { ...session, ...updates }
          : session
      )
    });
  };

  handleSave = () => {
    const mainSession = this.state.sessions[0];
    const updatedLesson: LessonPlan = {
      id: this.props.lesson.id,
      title: mainSession.title,
      duration: mainSession.duration,
      description: mainSession.description,
      objectives: mainSession.objectives,
      introVideo: mainSession.introVideo,
      assignmentVideo: mainSession.assignmentVideo,
      activityVideos: mainSession.activityVideos,
      resources: mainSession.resources
    };
    this.props.onSave(updatedLesson);
    this.props.onClose();
  };

  switchToSession = (sessionId: string) => {
    this.setState({ currentSessionId: sessionId });
  };

  addObjective = () => {
    const { newObjective } = this.state;
    const currentSession = this.getCurrentSession();
    
    if (newObjective.trim()) {
      this.updateCurrentSession({
        objectives: [...currentSession.objectives, newObjective]
      });
      this.setState({ newObjective: '' });
    }
  };

  removeObjective = (index: number) => {
    const currentSession = this.getCurrentSession();
    this.updateCurrentSession({
      objectives: currentSession.objectives.filter((_, i) => i !== index)
    });
  };

  addActivityVideo = () => {
    const currentSession = this.getCurrentSession();
    this.updateCurrentSession({
      activityVideos: [...currentSession.activityVideos, { title: '', duration: 0, url: '' }]
    });
  };

  removeActivityVideo = (index: number) => {
    const currentSession = this.getCurrentSession();
    this.updateCurrentSession({
      activityVideos: currentSession.activityVideos.filter((_, i) => i !== index)
    });
  };

  updateActivityVideo = (index: number, field: string, value: any) => {
    const currentSession = this.getCurrentSession();
    const newActivities = [...currentSession.activityVideos];
    newActivities[index] = { ...newActivities[index], [field]: value };
    this.updateCurrentSession({
      activityVideos: newActivities
    });
  };

  addResource = () => {
    const { newResource } = this.state;
    const currentSession = this.getCurrentSession();
    
    if (newResource.name.trim()) {
      this.updateCurrentSession({
        resources: [...currentSession.resources, { ...newResource }]
      });
      this.setState({
        newResource: {
          name: '',
          type: 'pdf',
          url: ''
        }
      });
    }
  };

  removeResource = (index: number) => {
    const currentSession = this.getCurrentSession();
    this.updateCurrentSession({
      resources: currentSession.resources.filter((_, i) => i !== index)
    });
  };

  toggleSessionComplete = (sessionId: string) => {
    this.setState({
      sessions: this.state.sessions.map(session =>
        session.id === sessionId 
          ? { ...session, isCompleted: !session.isCompleted }
          : session
      )
    });
  };

  getSessionProgress = () => {
    const completedSessions = this.state.sessions.filter(s => s.isCompleted).length;
    return Math.round((completedSessions / 12) * 100);
  };

  toggleSection = (section: string) => {
    this.setState({
      expandedSections: {
        ...this.state.expandedSections,
        [section]: !this.state.expandedSections[section]
      }
    });
  };

  render() {
    const { open, onClose } = this.props;
    const { sessions, currentSessionId, newObjective, newResource, expandedSections } = this.state;
    const currentSession = this.getCurrentSession();
    const currentSessionNumber = parseInt(currentSessionId.split('-')[1]);
    const completedSessions = sessions.filter(s => s.isCompleted).length;

    return (
      <Dialog 
        open={open} 
        onClose={onClose} 
        maxWidth={false}
        fullWidth
        PaperProps={{
          sx: { 
            borderRadius: 4,
            maxWidth: '95vw',
            maxHeight: '95vh',
            width: '95vw',
            height: '95vh',
            background: 'linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%)',
          }
        }}
      >
        <DialogContent sx={{ p: 0, display: 'flex', flexDirection: 'column', height: '100%' }}>
          {/* Enhanced Header */}
          <Box sx={{ 
            background: 'linear-gradient(135deg, #25476a 0%, #1e3a52 100%)',
            color: 'white',
            p: 4,
            position: 'relative',
            overflow: 'hidden'
          }}>
            <Box sx={{
              position: 'absolute',
              top: 0,
              right: 0,
              width: 200,
              height: 200,
              background: 'radial-gradient(circle, rgba(56, 170, 225, 0.3) 0%, transparent 70%)',
              borderRadius: '50%',
              transform: 'translate(50%, -50%)'
            }} />
            
            <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'flex-start', position: 'relative' }}>
              <Box sx={{ flex: 1 }}>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                  <Avatar sx={{ bgcolor: '#38aae1', mr: 2, width: 48, height: 48 }}>
                    <School />
                  </Avatar>
                  <Box>
                    <Typography variant="h4" sx={{ fontWeight: 700, mb: 0.5 }}>
                      Lesson Plan Management
                    </Typography>
                    <Typography variant="body1" sx={{ opacity: 0.9 }}>
                      Comprehensive session planning and content organization
                    </Typography>
                  </Box>
                </Box>
                
                <Box sx={{ display: 'flex', alignItems: 'center', gap: 2, mt: 3 }}>
                  <Timeline sx={{ color: '#feb139' }} />
                  <Typography variant="body2" sx={{ opacity: 0.9 }}>
                    Currently editing: <strong>{currentSession.title}</strong>
                  </Typography>
                </Box>
              </Box>
              
              <IconButton 
                onClick={onClose} 
                sx={{ 
                  color: 'white', 
                  bgcolor: 'rgba(255,255,255,0.1)',
                  '&:hover': { bgcolor: 'rgba(255,255,255,0.2)' }
                }}
              >
                <Close />
              </IconButton>
            </Box>
          </Box>

          {/* Main Content Layout */}
          <Box sx={{ display: 'flex', flex: 1, overflow: 'hidden' }}>
            {/* Session Navigation - Enhanced Stepper */}
            <Box sx={{ 
              width: 320, 
              bgcolor: 'white',
              borderRight: '1px solid #e2e8f0',
              display: 'flex',
              flexDirection: 'column',
              boxShadow: 'inset -4px 0 8px rgba(0,0,0,0.05)'
            }}>
              <Box sx={{ p: 3, borderBottom: '1px solid #e2e8f0' }}>
                <Typography variant="h6" sx={{ fontWeight: 700, color: '#25476a', mb: 1 }}>
                  Session Timeline
                </Typography>
                <Typography variant="body2" sx={{ color: '#64748b' }}>
                  Click any session to edit its content
                </Typography>
              </Box>
              
              <Box sx={{ flex: 1, overflow: 'auto', p: 2 }}>
                {sessions.map((session, index) => (
                  <Card 
                    key={session.id}
                    onClick={() => this.switchToSession(session.id)}
                    sx={{
                      mb: 2,
                      cursor: 'pointer',
                      border: currentSessionId === session.id ? '2px solid #38aae1' : '1px solid #e2e8f0',
                      bgcolor: currentSessionId === session.id ? 'rgba(56, 170, 225, 0.05)' : 'white',
                      transform: currentSessionId === session.id ? 'scale(1.02)' : 'scale(1)',
                      transition: 'all 0.2s cubic-bezier(0.4, 0, 0.2, 1)',
                      '&:hover': {
                        transform: 'scale(1.02)',
                        boxShadow: '0 8px 25px rgba(0,0,0,0.15)',
                      }
                    }}
                  >
                    <CardContent sx={{ p: 2 }}>
                      <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 1 }}>
                        <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                          <Avatar 
                            sx={{ 
                              width: 32, 
                              height: 32,
                              bgcolor: session.isCompleted ? '#4caf50' : (currentSessionId === session.id ? '#38aae1' : '#e0e0e0'),
                              color: session.isCompleted || currentSessionId === session.id ? 'white' : '#64748b',
                              fontSize: '0.875rem',
                              fontWeight: 600
                            }}
                          >
                            {index + 1}
                          </Avatar>
                          <Typography variant="subtitle2" sx={{ fontWeight: 600, color: '#25476a' }}>
                            Session {index + 1}
                          </Typography>
                        </Box>
                        
                        {session.isCompleted && (
                          <CheckCircle sx={{ color: '#4caf50', fontSize: 20 }} />
                        )}
                      </Box>
                      
                      <Typography 
                        variant="body2" 
                        sx={{ 
                          color: '#64748b',
                          mb: 1,
                          overflow: 'hidden',
                          textOverflow: 'ellipsis',
                          whiteSpace: 'nowrap'
                        }}
                      >
                        {session.title}
                      </Typography>
                      
                      <Box sx={{ display: 'flex', gap: 1, flexWrap: 'wrap' }}>
                        <Chip 
                          size="small" 
                          label={`${session.duration}min`}
                          sx={{ fontSize: '0.75rem', height: 20 }}
                        />
                        {session.objectives.length > 0 && (
                          <Chip 
                            size="small" 
                            label={`${session.objectives.length} goals`}
                            sx={{ fontSize: '0.75rem', height: 20 }}
                          />
                        )}
                      </Box>
                      
                      <Button
                        size="small"
                        onClick={(e) => {
                          e.stopPropagation();
                          this.toggleSessionComplete(session.id);
                        }}
                        startIcon={session.isCompleted ? <RadioButtonUnchecked /> : <CheckCircle />}
                        sx={{
                          mt: 1,
                          fontSize: '0.75rem',
                          textTransform: 'none',
                          color: session.isCompleted ? '#64748b' : '#4caf50',
                          minHeight: 24,
                          '&:hover': {
                            bgcolor: session.isCompleted ? 'rgba(100, 116, 139, 0.1)' : 'rgba(76, 175, 80, 0.1)',
                          }
                        }}
                      >
                        {session.isCompleted ? 'Mark Incomplete' : 'Mark Complete'}
                      </Button>
                    </CardContent>
                  </Card>
                ))}
              </Box>
            </Box>

            {/* Content Area - Enhanced Design */}
            <Box sx={{ flex: 1, display: 'flex', flexDirection: 'column', overflow: 'hidden' }}>
              {/* Session Header Card */}
              <Card sx={{ 
                mx: 4,
                mt: 4,
                mb: 3,
                background: 'linear-gradient(135deg, #38aae1 0%, #2563eb 100%)',
                color: 'white',
                overflow: 'hidden',
                position: 'relative'
              }}>
                <Box sx={{
                  position: 'absolute',
                  top: -50,
                  right: -50,
                  width: 150,
                  height: 150,
                  background: 'radial-gradient(circle, rgba(255,255,255,0.2) 0%, transparent 70%)',
                  borderRadius: '50%'
                }} />
                
                <CardContent sx={{ p: 4, position: 'relative' }}>
                  <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 2 }}>
                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                      <Avatar sx={{ bgcolor: 'rgba(255,255,255,0.2)', width: 56, height: 56 }}>
                        <MenuBook sx={{ fontSize: 28 }} />
                      </Avatar>
                      <Box>
                        <Typography variant="h4" sx={{ fontWeight: 700, mb: 0.5 }}>
                          Session {currentSessionNumber} Configuration
                        </Typography>
                        <Typography variant="body1" sx={{ opacity: 0.9 }}>
                          Configure content and structure for this learning session
                        </Typography>
                      </Box>
                    </Box>
                    
                    <Chip 
                      icon={currentSession.isCompleted ? <CheckCircle /> : <Edit />}
                      label={currentSession.isCompleted ? 'Complete' : 'In Progress'} 
                      sx={{
                        bgcolor: currentSession.isCompleted ? 'rgba(76, 175, 80, 0.2)' : 'rgba(255,255,255,0.2)',
                        color: 'white',
                        border: '1px solid rgba(255,255,255,0.3)',
                        fontWeight: 600
                      }}
                    />
                  </Box>
                </CardContent>
              </Card>

              {/* Tabs Navigation */}
              <Box sx={{ 
                borderBottom: '1px solid #e2e8f0', 
                bgcolor: 'white', 
                mx: 4,
                borderRadius: '12px 12px 0 0',
                boxShadow: '0 2px 8px rgba(0,0,0,0.05)' 
              }}>
                <Tabs 
                  value={this.state.activeTab} 
                  onChange={(e, newValue) => this.setState({ activeTab: newValue })}
                  sx={{ 
                    px: 3,
                    '& .MuiTab-root': {
                      textTransform: 'none',
                      fontWeight: 600,
                      fontSize: '1rem',
                      color: '#64748b',
                      minHeight: 60,
                      '&.Mui-selected': {
                        color: '#38aae1',
                      }
                    },
                    '& .MuiTabs-indicator': {
                      backgroundColor: '#38aae1',
                      height: 3,
                      borderRadius: '3px 3px 0 0'
                    }
                  }}
                >
                  <Tab 
                    label="Session Overview" 
                    icon={<Edit />} 
                    iconPosition="start"
                    sx={{ mr: 2 }}
                  />
                  <Tab 
                    label="Video Content" 
                    icon={<VideoLibrary />} 
                    iconPosition="start"
                    sx={{ mr: 2 }}
                  />
                  <Tab 
                    label="Learning Resources" 
                    icon={<MenuBook />} 
                    iconPosition="start"
                  />
                </Tabs>
              </Box>

              {/* Tab Content */}
              <Box sx={{ flex: 1, overflow: 'auto', bgcolor: 'white', mx: 4, borderRadius: '0 0 12px 12px' }}>

                {/* Tab Content */}
                <Box sx={{ p: 4 }}>
                  {/* Overview Tab */}
                  {this.state.activeTab === 0 && (
                    <Box sx={{ maxWidth: 800 }}>
                      <Grid container spacing={3}>
                        <Grid item xs={12}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                            Lesson Topic
                          </Typography>
                          <TextField
                            fullWidth
                            value={currentSession.title}
                            onChange={(e) => this.updateCurrentSession({ title: e.target.value })}
                            placeholder="Enter lesson topic"
                            size="small"
                            sx={{
                              '& .MuiOutlinedInput-root': {
                                borderRadius: 2,
                              }
                            }}
                          />
                        </Grid>
                        <Grid item xs={12}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                            Lesson Sub Topic
                          </Typography>
                          <TextField
                            fullWidth
                            multiline
                            rows={3}
                            value={currentSession.description}
                            onChange={(e) => this.updateCurrentSession({ description: e.target.value })}
                            placeholder="Describe the specific sub topics covered in this lesson"
                            size="small"
                            sx={{
                              '& .MuiOutlinedInput-root': {
                                borderRadius: 2,
                              }
                            }}
                          />
                        </Grid>
                        <Grid item xs={12} md={4}>
                          <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                            Duration (minutes)
                          </Typography>
                          <TextField
                            fullWidth
                            type="number"
                            value={currentSession.duration}
                            onChange={(e) => this.updateCurrentSession({ duration: parseInt(e.target.value) || 0 })}
                            size="small"
                            sx={{
                              '& .MuiOutlinedInput-root': {
                                borderRadius: 2,
                              }
                            }}
                          />
                        </Grid>
                      </Grid>

                      {/* Session Update Button */}
                      <Box sx={{ mt: 4, mb: 4 }}>
                        <Button
                          variant="contained"
                          size="medium"
                          onClick={() => {
                            // Update session with current values
                            console.log('Updating lesson topic, sub topic, and duration');
                          }}
                          sx={{ 
                            bgcolor: '#feb139',
                            color: '#25476a',
                            textTransform: 'none',
                            borderRadius: 2,
                            fontWeight: 600,
                            px: 3,
                            '&:hover': { 
                              bgcolor: '#e09e2d',
                              boxShadow: '0 4px 12px rgba(254, 177, 57, 0.3)' 
                            }
                          }}
                        >
                          Update Lesson Details
                        </Button>
                      </Box>

                      {/* Desired Outcomes */}
                      <Box sx={{ mt: 4 }}>
                        <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 3 }}>
                          <Typography variant="h6" sx={{ fontWeight: 700, color: '#25476a' }}>
                            Desired Outcomes
                          </Typography>
                          <Button
                            startIcon={<Add />}
                            variant="contained"
                            size="small"
                            onClick={this.addObjective}
                            disabled={!newObjective.trim()}
                            sx={{ 
                              bgcolor: '#38aae1',
                              textTransform: 'none',
                              borderRadius: 2,
                              '&:hover': { bgcolor: '#2e8bc4' }
                            }}
                          >
                            Add Outcome
                          </Button>
                        </Box>
                        
                        <TextField
                          fullWidth
                          placeholder="Enter desired outcome for this session"
                          value={newObjective}
                          onChange={(e) => this.setState({ newObjective: e.target.value })}
                          onKeyPress={(e) => {
                            if (e.key === 'Enter') {
                              this.addObjective();
                            }
                          }}
                          size="small"
                          sx={{ 
                            mb: 3,
                            '& .MuiOutlinedInput-root': {
                              borderRadius: 2,
                            }
                          }}
                        />

                        {currentSession.objectives.map((objective, index) => (
                          <Card 
                            key={index}
                            sx={{ 
                              mb: 2,
                              border: '1px solid #e2e8f0',
                              borderRadius: 2,
                              '&:hover': { boxShadow: '0 4px 12px rgba(0,0,0,0.1)' }
                            }}
                          >
                            <CardContent sx={{ 
                              display: 'flex', 
                              alignItems: 'center', 
                              justifyContent: 'space-between',
                              p: 2,
                              '&:last-child': { pb: 2 }
                            }}>
                              <Typography sx={{ flex: 1 }}>{objective}</Typography>
                              <IconButton 
                                size="small" 
                                onClick={() => this.removeObjective(index)}
                                sx={{ color: '#ef4444' }}
                              >
                                <Delete fontSize="small" />
                              </IconButton>
                            </CardContent>
                          </Card>
                        ))}
                      </Box>
                    </Box>
                  )}

                  {/* Video Content Tab */}
                  {this.state.activeTab === 1 && (
                    <Box sx={{ maxWidth: 800 }}>
                      {/* Introduction Video */}
                      <Card sx={{ mb: 4, border: '1px solid #e2e8f0', borderRadius: 2 }}>
                        <CardContent sx={{ p: 3 }}>
                          <Box sx={{ display: 'flex', alignItems: 'center', mb: 3 }}>
                            <PlayCircleOutline sx={{ color: '#38aae1', mr: 2 }} />
                            <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                              Introduction Video
                            </Typography>
                          </Box>
                          
                          <Grid container spacing={3}>
                            <Grid item xs={12} md={8}>
                              <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                Video Title
                              </Typography>
                              <TextField
                                fullWidth
                                value={currentSession.introVideo.title}
                                onChange={(e) => this.updateCurrentSession({
                                  introVideo: { ...currentSession.introVideo, title: e.target.value }
                                })}
                                placeholder="Introduction video title"
                                size="small"
                                sx={{ '& .MuiOutlinedInput-root': { borderRadius: 2 } }}
                              />
                            </Grid>
                            <Grid item xs={12} md={4}>
                              <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                Duration (minutes)
                              </Typography>
                              <TextField
                                fullWidth
                                type="number"
                                value={currentSession.introVideo.duration}
                                onChange={(e) => this.updateCurrentSession({
                                  introVideo: { ...currentSession.introVideo, duration: parseInt(e.target.value) || 0 }
                                })}
                                size="small"
                                sx={{ '& .MuiOutlinedInput-root': { borderRadius: 2 } }}
                              />
                            </Grid>
                            <Grid item xs={12}>
                              <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                Upload Video
                              </Typography>
                              
                              {/* Video Preview */}
                              {currentSession.introVideo.url && (
                                <Box sx={{ mb: 3 }}>
                                  <Typography variant="body2" sx={{ mb: 2, fontWeight: 600, color: '#25476a' }}>
                                    Video Preview
                                  </Typography>
                                  <VidstackPlayer
                                    src={currentSession.introVideo.url}
                                    title={currentSession.introVideo.title}
                                    aspectRatio="16/9"
                                  />
                                  <Typography variant="body2" sx={{ color: '#38aae1', mt: 1, fontWeight: 600 }}>
                                    Current video: {currentSession.introVideo.url}
                                  </Typography>
                                </Box>
                              )}
                              
                              <Box sx={{ 
                                border: '2px dashed #e2e8f0', 
                                borderRadius: 2, 
                                p: 3, 
                                textAlign: 'center',
                                bgcolor: '#f8fafc',
                                '&:hover': {
                                  borderColor: '#38aae1',
                                  bgcolor: 'rgba(56, 170, 225, 0.05)'
                                },
                                cursor: 'pointer'
                              }}>
                                <input
                                  type="file"
                                  accept="video/*"
                                  style={{ display: 'none' }}
                                  id="intro-video-upload"
                                  onChange={(e) => {
                                    const file = e.target.files?.[0];
                                    if (file) {
                                      this.updateCurrentSession({
                                        introVideo: { ...currentSession.introVideo, url: file.name }
                                      });
                                    }
                                  }}
                                />
                                <label htmlFor="intro-video-upload" style={{ cursor: 'pointer', width: '100%', display: 'block' }}>
                                  <PlayCircleOutline sx={{ fontSize: 48, color: '#64748b', mb: 1 }} />
                                  <Typography variant="body2" sx={{ color: '#64748b', mb: 1 }}>
                                    {currentSession.introVideo.url ? 'Click to replace introduction video' : 'Click to upload introduction video'}
                                  </Typography>
                                  <Typography variant="caption" sx={{ color: '#94a3b8' }}>
                                    MP4, AVI, MOV up to 100MB
                                  </Typography>
                                </label>
                              </Box>
                            </Grid>
                          </Grid>
                        </CardContent>
                      </Card>

                      {/* Assignment Video */}
                      <Card sx={{ mb: 4, border: '1px solid #e2e8f0', borderRadius: 2 }}>
                        <CardContent sx={{ p: 3 }}>
                          <Box sx={{ display: 'flex', alignItems: 'center', mb: 3 }}>
                            <Assignment sx={{ color: '#ff9800', mr: 2 }} />
                            <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                              Assignment Video
                            </Typography>
                          </Box>
                          
                          <Grid container spacing={3}>
                            <Grid item xs={12} md={8}>
                              <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                Video Title
                              </Typography>
                              <TextField
                                fullWidth
                                value={currentSession.assignmentVideo.title}
                                onChange={(e) => this.updateCurrentSession({
                                  assignmentVideo: { ...currentSession.assignmentVideo, title: e.target.value }
                                })}
                                placeholder="Assignment video title"
                                size="small"
                                sx={{ '& .MuiOutlinedInput-root': { borderRadius: 2 } }}
                              />
                            </Grid>
                            <Grid item xs={12} md={4}>
                              <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                Duration (minutes)
                              </Typography>
                              <TextField
                                fullWidth
                                type="number"
                                value={currentSession.assignmentVideo.duration}
                                onChange={(e) => this.updateCurrentSession({
                                  assignmentVideo: { ...currentSession.assignmentVideo, duration: parseInt(e.target.value) || 0 }
                                })}
                                size="small"
                                sx={{ '& .MuiOutlinedInput-root': { borderRadius: 2 } }}
                              />
                            </Grid>
                            <Grid item xs={12}>
                              <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                Upload Video
                              </Typography>
                              
                              {/* Video Preview */}
                              {currentSession.assignmentVideo.url && (
                                <Box sx={{ mb: 3 }}>
                                  <Typography variant="body2" sx={{ mb: 2, fontWeight: 600, color: '#25476a' }}>
                                    Video Preview
                                  </Typography>
                                  <VidstackPlayer
                                    src={currentSession.assignmentVideo.url}
                                    title={currentSession.assignmentVideo.title}
                                    aspectRatio="16/9"
                                  />
                                  <Typography variant="body2" sx={{ color: '#ff9800', mt: 1, fontWeight: 600 }}>
                                    Current video: {currentSession.assignmentVideo.url}
                                  </Typography>
                                </Box>
                              )}
                              
                              <Box sx={{ 
                                border: '2px dashed #e2e8f0', 
                                borderRadius: 2, 
                                p: 3, 
                                textAlign: 'center',
                                bgcolor: '#f8fafc',
                                '&:hover': {
                                  borderColor: '#ff9800',
                                  bgcolor: 'rgba(255, 152, 0, 0.05)'
                                },
                                cursor: 'pointer'
                              }}>
                                <input
                                  type="file"
                                  accept="video/*"
                                  style={{ display: 'none' }}
                                  id="assignment-video-upload"
                                  onChange={(e) => {
                                    const file = e.target.files?.[0];
                                    if (file) {
                                      this.updateCurrentSession({
                                        assignmentVideo: { ...currentSession.assignmentVideo, url: file.name }
                                      });
                                    }
                                  }}
                                />
                                <label htmlFor="assignment-video-upload" style={{ cursor: 'pointer', width: '100%', display: 'block' }}>
                                  <Assignment sx={{ fontSize: 48, color: '#64748b', mb: 1 }} />
                                  <Typography variant="body2" sx={{ color: '#64748b', mb: 1 }}>
                                    {currentSession.assignmentVideo.url ? 'Click to replace assignment video' : 'Click to upload assignment video'}
                                  </Typography>
                                  <Typography variant="caption" sx={{ color: '#94a3b8' }}>
                                    MP4, AVI, MOV up to 100MB
                                  </Typography>
                                </label>
                              </Box>
                            </Grid>
                          </Grid>
                        </CardContent>
                      </Card>

                      {/* Activity Videos */}
                      <Card sx={{ border: '1px solid #e2e8f0', borderRadius: 2 }}>
                        <CardContent sx={{ p: 3 }}>
                          <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 3 }}>
                            <Box sx={{ display: 'flex', alignItems: 'center' }}>
                              <SportsEsports sx={{ color: '#9c27b0', mr: 2 }} />
                              <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                                Activity Videos
                              </Typography>
                            </Box>
                            <Button
                              startIcon={<Add />}
                              variant="contained"
                              onClick={this.addActivityVideo}
                              sx={{ 
                                bgcolor: '#9c27b0',
                                '&:hover': { bgcolor: '#7b1fa2' },
                                textTransform: 'none',
                                borderRadius: 2
                              }}
                            >
                              Add Activity Video
                            </Button>
                          </Box>
                          
                          {currentSession.activityVideos.map((activity, index) => (
                            <Card key={index} sx={{ mb: 3, border: '1px solid #e2e8f0', borderRadius: 2 }}>
                              <CardContent sx={{ p: 3 }}>
                                <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 2 }}>
                                  <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                                    Activity Video {index + 1}
                                  </Typography>
                                  <IconButton 
                                    size="small" 
                                    onClick={() => this.removeActivityVideo(index)}
                                    sx={{ color: '#ef4444' }}
                                  >
                                    <Delete />
                                  </IconButton>
                                </Box>
                                
                                <Grid container spacing={3}>
                                  <Grid item xs={12} md={8}>
                                    <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                      Video Title
                                    </Typography>
                                    <TextField
                                      fullWidth
                                      value={activity.title}
                                      onChange={(e) => this.updateActivityVideo(index, 'title', e.target.value)}
                                      placeholder="Activity video title"
                                      size="small"
                                      sx={{ '& .MuiOutlinedInput-root': { borderRadius: 2 } }}
                                    />
                                  </Grid>
                                  <Grid item xs={12} md={4}>
                                    <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                      Duration (minutes)
                                    </Typography>
                                    <TextField
                                      fullWidth
                                      type="number"
                                      value={activity.duration}
                                      onChange={(e) => this.updateActivityVideo(index, 'duration', parseInt(e.target.value) || 0)}
                                      size="small"
                                      sx={{ '& .MuiOutlinedInput-root': { borderRadius: 2 } }}
                                    />
                                  </Grid>
                                  <Grid item xs={12}>
                                    <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                      Upload Video
                                    </Typography>
                                    <Box sx={{ 
                                      border: '2px dashed #e2e8f0', 
                                      borderRadius: 2, 
                                      p: 3, 
                                      textAlign: 'center',
                                      bgcolor: '#f8fafc',
                                      '&:hover': {
                                        borderColor: '#9c27b0',
                                        bgcolor: 'rgba(156, 39, 176, 0.05)'
                                      },
                                      cursor: 'pointer'
                                    }}>
                                      <input
                                        type="file"
                                        accept="video/*"
                                        style={{ display: 'none' }}
                                        id={`activity-video-upload-${index}`}
                                        onChange={(e) => {
                                          const file = e.target.files?.[0];
                                          if (file) {
                                            this.updateActivityVideo(index, 'url', file.name);
                                          }
                                        }}
                                      />
                                      <label htmlFor={`activity-video-upload-${index}`} style={{ cursor: 'pointer', width: '100%', display: 'block' }}>
                                        <SportsEsports sx={{ fontSize: 48, color: '#64748b', mb: 1 }} />
                                        <Typography variant="body2" sx={{ color: '#64748b', mb: 1 }}>
                                          Click to upload activity video
                                        </Typography>
                                        <Typography variant="caption" sx={{ color: '#94a3b8' }}>
                                          MP4, AVI, MOV up to 100MB
                                        </Typography>
                                        {activity.url && (
                                          <Typography variant="body2" sx={{ color: '#9c27b0', mt: 1, fontWeight: 600 }}>
                                            Selected: {activity.url}
                                          </Typography>
                                        )}
                                      </label>
                                    </Box>
                                  </Grid>
                                </Grid>
                              </CardContent>
                            </Card>
                          ))}
                        </CardContent>
                      </Card>
                    </Box>
                  )}

                  {/* Learning Resources Tab */}
                  {this.state.activeTab === 2 && (
                    <Box sx={{ maxWidth: 800 }}>
                      <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 3 }}>
                        <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a' }}>
                          Resources ({currentSession.resources.length})
                        </Typography>
                        <Button
                          startIcon={<Add />}
                          variant="contained"
                          onClick={this.addResource}
                          disabled={!newResource.name.trim()}
                          sx={{ 
                            bgcolor: '#9c27b0',
                            '&:hover': { bgcolor: '#7b1fa2' },
                            textTransform: 'none',
                            borderRadius: 2
                          }}
                        >
                          Add Resource
                        </Button>
                      </Box>

                      {/* Add New Resource */}
                      <Card sx={{ mb: 4, border: '1px solid #e2e8f0', borderRadius: 2, bgcolor: '#f8fafc' }}>
                        <CardContent sx={{ p: 3 }}>
                          <Typography variant="h6" sx={{ mb: 2, fontWeight: 600, color: '#25476a' }}>
                            Add New Resource
                          </Typography>
                          <Grid container spacing={3}>
                            <Grid item xs={12} md={6}>
                              <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                Resource Name
                              </Typography>
                              <TextField
                                fullWidth
                                value={newResource.name}
                                onChange={(e) => this.setState({
                                  newResource: { ...newResource, name: e.target.value }
                                })}
                                placeholder="Resource name"
                                size="small"
                                sx={{ '& .MuiOutlinedInput-root': { borderRadius: 2, bgcolor: 'white' } }}
                              />
                            </Grid>
                            <Grid item xs={12} md={6}>
                              <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                Resource Type
                              </Typography>
                              <FormControl fullWidth size="small">
                                <Select
                                  value={newResource.type}
                                  onChange={(e) => this.setState({
                                    newResource: { ...newResource, type: e.target.value as 'pdf' | 'doc' | 'ppt' | 'youtube' }
                                  })}
                                  sx={{ borderRadius: 2, bgcolor: 'white' }}
                                >
                                  <MenuItem value="pdf">PDF</MenuItem>
                                  <MenuItem value="doc">Document</MenuItem>
                                  <MenuItem value="ppt">Presentation</MenuItem>
                                  <MenuItem value="youtube">YouTube</MenuItem>
                                </Select>
                              </FormControl>
                            </Grid>
                            <Grid item xs={12}>
                              <Typography variant="body2" sx={{ mb: 1, fontWeight: 600, color: '#25476a' }}>
                                Upload Resource
                              </Typography>
                              <Box sx={{ 
                                border: '2px dashed #e2e8f0', 
                                borderRadius: 2, 
                                p: 3, 
                                textAlign: 'center',
                                bgcolor: 'white',
                                '&:hover': {
                                  borderColor: '#9c27b0',
                                  bgcolor: 'rgba(156, 39, 176, 0.05)'
                                },
                                cursor: 'pointer'
                              }}>
                                <input
                                  type="file"
                                  accept={newResource.type === 'pdf' ? '.pdf' : newResource.type === 'doc' ? '.doc,.docx' : newResource.type === 'ppt' ? '.ppt,.pptx' : 'image/*'}
                                  style={{ display: 'none' }}
                                  id="resource-upload"
                                  onChange={(e) => {
                                    const file = e.target.files?.[0];
                                    if (file) {
                                      this.setState({
                                        newResource: { ...newResource, url: file.name }
                                      });
                                    }
                                  }}
                                />
                                <label htmlFor="resource-upload" style={{ cursor: 'pointer', width: '100%', display: 'block' }}>
                                  <MenuBook sx={{ fontSize: 48, color: '#64748b', mb: 1 }} />
                                  <Typography variant="body2" sx={{ color: '#64748b', mb: 1 }}>
                                    Click to upload {newResource.type === 'youtube' ? 'or enter YouTube URL' : newResource.type.toUpperCase() + ' file'}
                                  </Typography>
                                  <Typography variant="caption" sx={{ color: '#94a3b8' }}>
                                    {newResource.type === 'pdf' && 'PDF files up to 10MB'}
                                    {newResource.type === 'doc' && 'DOC, DOCX files up to 10MB'}
                                    {newResource.type === 'ppt' && 'PPT, PPTX files up to 10MB'}
                                    {newResource.type === 'youtube' && 'Enter YouTube video URL'}
                                  </Typography>
                                  {newResource.url && (
                                    <Typography variant="body2" sx={{ color: '#9c27b0', mt: 1, fontWeight: 600 }}>
                                      Selected: {newResource.url}
                                    </Typography>
                                  )}
                                </label>
                              </Box>
                              {newResource.type === 'youtube' && (
                                <TextField
                                  fullWidth
                                  value={newResource.url}
                                  onChange={(e) => this.setState({
                                    newResource: { ...newResource, url: e.target.value }
                                  })}
                                  placeholder="Enter YouTube URL"
                                  size="small"
                                  sx={{ 
                                    mt: 2,
                                    '& .MuiOutlinedInput-root': { borderRadius: 2, bgcolor: 'white' } 
                                  }}
                                />
                              )}
                            </Grid>
                          </Grid>
                        </CardContent>
                      </Card>

                      {/* Existing Resources */}
                      {currentSession.resources.map((resource, index) => (
                        <Card 
                          key={index}
                          sx={{ 
                            mb: 2,
                            border: '1px solid #e2e8f0',
                            borderRadius: 2,
                            '&:hover': { boxShadow: '0 4px 12px rgba(0,0,0,0.1)' }
                          }}
                        >
                          <CardContent sx={{ 
                            display: 'flex', 
                            alignItems: 'center', 
                            justifyContent: 'space-between',
                            p: 2,
                            '&:last-child': { pb: 2 }
                          }}>
                            <Box>
                              <Typography variant="body1" sx={{ fontWeight: 600, color: '#25476a' }}>
                                {resource.name}
                              </Typography>
                              <Chip 
                                label={resource.type.toUpperCase()}
                                size="small"
                                sx={{ mt: 0.5, fontSize: '0.75rem' }}
                              />
                            </Box>
                            <IconButton 
                              size="small" 
                              onClick={() => this.removeResource(index)}
                              sx={{ color: '#ef4444' }}
                            >
                              <Delete fontSize="small" />
                            </IconButton>
                          </CardContent>
                        </Card>
                      ))}
                    </Box>
                  )}
                </Box>
              </Box>

              {/* Enhanced Footer */}
              <Box sx={{ 
                p: 4, 
                bgcolor: 'white', 
                borderTop: '1px solid #e2e8f0',
                display: 'flex',
                justifyContent: 'space-between',
                alignItems: 'center',
                boxShadow: '0 -4px 20px rgba(0,0,0,0.05)'
              }}>
                <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                  <Typography variant="body2" sx={{ color: '#64748b' }}>
                    Auto-saved • Last saved: just now
                  </Typography>
                </Box>
                
                <Box sx={{ display: 'flex', gap: 2 }}>
                  <Button 
                    onClick={onClose}
                    variant="outlined"
                    sx={{ 
                      color: '#64748b',
                      borderColor: '#e2e8f0',
                      textTransform: 'none',
                      borderRadius: 2,
                      px: 3,
                      '&:hover': { 
                        bgcolor: '#f8fafc',
                        borderColor: '#d1d5db' 
                      }
                    }}
                  >
                    Cancel
                  </Button>
                  <Button 
                    onClick={this.handleSave}
                    variant="contained"
                    startIcon={<Save />}
                    sx={{
                      background: 'linear-gradient(135deg, #38aae1 0%, #2563eb 100%)',
                      textTransform: 'none',
                      fontWeight: 600,
                      borderRadius: 2,
                      px: 4,
                      boxShadow: '0 4px 20px rgba(56, 170, 225, 0.3)',
                      '&:hover': {
                        background: 'linear-gradient(135deg, #2e8bc4 0%, #1d4ed8 100%)',
                        boxShadow: '0 6px 25px rgba(56, 170, 225, 0.4)',
                      },
                    }}
                  >
                    Save All Sessions
                  </Button>
                </Box>
              </Box>
            </Box>
          </Box>
        </DialogContent>
      </Dialog>
    );
  }
}