// Add Stream Dialog Component - Using MUI with Ant Design-inspired styling
import React, { Component } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  TextField,
  Box,
  IconButton,
  Typography,
  Autocomplete,
  Chip,
  CircularProgress,
} from '@mui/material';
import { Close } from '@mui/icons-material';

interface ClassOption {
  label: string;
  class_id: string;
}

interface AddStreamDialogProps {
  open: boolean;
  onClose: () => void;
  onAdd: (streamName: string, classIds: string[]) => void;
  availableClasses?: ClassOption[];
  isLoadingClassOptions?: boolean;
}

interface AddStreamDialogState {
  streamName: string;
  selectedClasses: ClassOption[];
  isLoadingClasses: boolean;
  errors: {
    streamName: string;
    classes: string;
  };
}

export class AddStreamDialog extends Component<AddStreamDialogProps, AddStreamDialogState> {
  constructor(props: AddStreamDialogProps) {
    super(props);
    this.state = {
      streamName: '',
      selectedClasses: [],
      isLoadingClasses: false,
      errors: {
        streamName: '',
        classes: '',
      },
    };
  }

  componentDidUpdate(prevProps: AddStreamDialogProps) {
    if (this.props.open && !prevProps.open) {
      // Reset form when dialog opens
      this.setState({
        streamName: '',
        selectedClasses: [],
        isLoadingClasses: false,
        errors: {
          streamName: '',
          classes: '',
        },
      });
    }
  }

  handleStreamNameChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    this.setState({ 
      streamName: event.target.value,
      errors: { ...this.state.errors, streamName: '' }
    });
  };

  handleClassesChange = (_event: React.SyntheticEvent, value: (string | ClassOption)[]) => {
    // Simulate loading when selecting classes
    this.setState({ isLoadingClasses: true });
    
    setTimeout(() => {
      // Convert string values to ClassOption objects (for freeSolo input)
      const normalizedValue = value.map(item => {
        if (typeof item === 'string') {
          // Create a new ClassOption for custom input
          return {
            label: item,
            class_id: `custom_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`
          };
        }
        return item;
      });
      
      this.setState({ 
        selectedClasses: normalizedValue,
        isLoadingClasses: false,
        errors: { ...this.state.errors, classes: '' }
      });
    }, 500);
  };

  handleSubmit = () => {
    const { streamName, selectedClasses } = this.state;
    const errors = {
      streamName: '',
      classes: '',
    };

    // Validation
    if (!streamName.trim()) {
      errors.streamName = 'Stream name is required';
    }
    if (selectedClasses.length === 0) {
      errors.classes = 'At least one class is required';
    }

    if (errors.streamName || errors.classes) {
      this.setState({ errors });
      return;
    }

    // Extract class_ids from selected classes
    const classIds = selectedClasses.map(classItem => classItem.class_id);
    this.props.onAdd(streamName.trim(), classIds);
    this.setState({
      streamName: '',
      selectedClasses: [],
      errors: {
        streamName: '',
        classes: '',
      },
    });
  };

  handleKeyPress = (event: React.KeyboardEvent) => {
    if (event.key === 'Enter') {
      event.preventDefault();
      this.handleSubmit();
    }
  };

  render() {
    const { open, onClose, availableClasses = [], isLoadingClassOptions = false } = this.props;
    const { streamName, selectedClasses, isLoadingClasses, errors } = this.state;

    return (
      <Dialog 
        open={open} 
        onClose={onClose}
        maxWidth="sm"
        fullWidth
        PaperProps={{
          sx: {
            borderRadius: 3,
            boxShadow: '0 8px 32px rgba(0, 0, 0, 0.12)',
          }
        }}
      >
        <DialogTitle
          sx={{
            backgroundColor: '#25476a',
            color: '#ffffff',
            display: 'flex',
            justifyContent: 'space-between',
            alignItems: 'center',
            py: 2.5,
            px: 3,
          }}
        >
          <Typography variant="h6" sx={{ fontWeight: 600 }}>
            Add New Stream
          </Typography>
          <IconButton
            onClick={onClose}
            size="small"
            sx={{
              color: '#ffffff',
              '&:hover': {
                backgroundColor: 'rgba(255, 255, 255, 0.1)',
              },
            }}
          >
            <Close />
          </IconButton>
        </DialogTitle>

        <DialogContent sx={{ pt: 3, px: 3, pb: 2 }}>
          <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2.5 }}>
            <Box>
              <Typography 
                variant="body2" 
                sx={{ 
                  color: '#25476a', 
                  fontWeight: 600, 
                  mb: 1,
                  fontSize: '0.875rem'
                }}
              >
                Stream Name *
              </Typography>
              
              <TextField
                fullWidth
                placeholder="e.g., Web Development, Mobile Apps, Data Science"
                value={streamName}
                onChange={this.handleStreamNameChange}
                onKeyPress={this.handleKeyPress}
                error={!!errors.streamName}
                helperText={errors.streamName}
                autoFocus
                sx={{
                  '& .MuiOutlinedInput-root': {
                    borderRadius: 2,
                    backgroundColor: '#f8fafc',
                    '& fieldset': {
                      borderColor: errors.streamName ? '#d32f2f' : '#e2e8f0',
                      borderWidth: '2px',
                    },
                    '&:hover fieldset': {
                      borderColor: errors.streamName ? '#d32f2f' : '#38aae1',
                    },
                    '&.Mui-focused fieldset': {
                      borderColor: errors.streamName ? '#d32f2f' : '#38aae1',
                    },
                  },
                }}
              />
            </Box>

            <Box>
              <Typography 
                variant="body2" 
                sx={{ 
                  color: '#25476a', 
                  fontWeight: 600, 
                  mb: 1,
                  fontSize: '0.875rem'
                }}
              >
                Classes *
              </Typography>
              
              <Autocomplete
                multiple
                options={availableClasses}
                value={selectedClasses}
                onChange={this.handleClassesChange}
                loading={isLoadingClasses || isLoadingClassOptions}
                disabled={isLoadingClassOptions}
                getOptionLabel={(option) => typeof option === 'string' ? option : option.label}
                isOptionEqualToValue={(option, value) => {
                  if (typeof option === 'string' || typeof value === 'string') return false;
                  return option.class_id === value.class_id;
                }}
                freeSolo
                filterOptions={(options, state) => {
                  const filtered = options.filter((option) =>
                    option.label.toLowerCase().includes(state.inputValue.toLowerCase())
                  );
                  return filtered;
                }}
                noOptionsText={isLoadingClassOptions ? "Loading classes..." : "No classes found. Type to add custom class."}
                renderTags={(value, getTagProps) =>
                  value.map((option, index) => {
                    const label = typeof option === 'string' ? option : option.label;
                    return (
                      <Chip
                        label={label}
                        {...getTagProps({ index })}
                        key={index}
                        sx={{
                          backgroundColor: '#38aae1',
                          color: '#ffffff',
                          fontWeight: 500,
                          '& .MuiChip-deleteIcon': {
                            color: 'rgba(255, 255, 255, 0.7)',
                            '&:hover': {
                              color: '#ffffff',
                            },
                          },
                        }}
                      />
                    );
                  })
                }
                renderInput={(params) => (
                  <TextField
                    {...params}
                    placeholder={isLoadingClassOptions ? "Loading classes..." : (selectedClasses.length === 0 ? "Search or type class names..." : "")}
                    error={!!errors.classes}
                    helperText={
                      errors.classes || 
                      (isLoadingClassOptions ? "Please wait while we load available classes..." : 
                       isLoadingClasses ? "Loading classes..." : 
                       "Type to search or add custom class")
                    }
                    InputProps={{
                      ...params.InputProps,
                      endAdornment: (
                        <>
                          {(isLoadingClasses || isLoadingClassOptions) ? (
                            <CircularProgress 
                              size={20} 
                              sx={{ 
                                color: '#38aae1',
                                mr: 1,
                              }} 
                            />
                          ) : null}
                          {params.InputProps.endAdornment}
                        </>
                      ),
                    }}
                    sx={{
                      '& .MuiOutlinedInput-root': {
                        borderRadius: 2,
                        backgroundColor: isLoadingClassOptions ? '#f1f5f9' : '#f8fafc',
                        '& fieldset': {
                          borderColor: errors.classes ? '#d32f2f' : '#e2e8f0',
                          borderWidth: '2px',
                        },
                        '&:hover fieldset': {
                          borderColor: errors.classes ? '#d32f2f' : '#38aae1',
                        },
                        '&.Mui-focused fieldset': {
                          borderColor: errors.classes ? '#d32f2f' : '#38aae1',
                        },
                        '&.Mui-disabled': {
                          backgroundColor: '#f1f5f9',
                        },
                      },
                    }}
                  />
                )}
              />
            </Box>

            <Box
              sx={{
                backgroundColor: '#e3f2fd',
                borderLeft: '4px solid #38aae1',
                p: 2,
                borderRadius: 1,
                mt: 1,
              }}
            >
              <Typography 
                variant="body2" 
                sx={{ 
                  color: '#25476a',
                  fontSize: '0.8125rem',
                  lineHeight: 1.6,
                }}
              >
                <strong>Tip:</strong> A stream represents the learning track (e.g., Web Development). 
                You can add multiple classes to each stream (e.g., Beginner Level 1, Intermediate Level 2). 
                Press Enter after typing to add a custom class name.
              </Typography>
            </Box>
          </Box>
        </DialogContent>

        <DialogActions sx={{ px: 3, pb: 3, pt: 2 }}>
          <Button
            onClick={onClose}
            variant="outlined"
            sx={{
              borderColor: '#e2e8f0',
              color: '#64748b',
              borderRadius: 2,
              px: 3,
              textTransform: 'none',
              fontWeight: 600,
              borderWidth: '2px',
              '&:hover': {
                borderColor: '#cbd5e1',
                backgroundColor: '#f8fafc',
                borderWidth: '2px',
              },
            }}
          >
            Cancel
          </Button>
          <Button
            onClick={this.handleSubmit}
            variant="contained"
            sx={{
              backgroundColor: '#38aae1',
              color: '#ffffff',
              borderRadius: 2,
              px: 3,
              textTransform: 'none',
              fontWeight: 600,
              boxShadow: '0 2px 8px rgba(56, 170, 225, 0.3)',
              '&:hover': {
                backgroundColor: '#2c8fc9',
                boxShadow: '0 4px 12px rgba(56, 170, 225, 0.4)',
              },
            }}
          >
            Add Stream
          </Button>
        </DialogActions>
      </Dialog>
    );
  }
}
