import React, { useState } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  Box,
  TextField,
  Button,
  IconButton,
  Typography,
  Select,
  MenuItem,
  FormControl,
  RadioGroup,
  FormControlLabel,
  Radio,
  Grid,
  Paper,
} from '@mui/material';
import {
  Close,
  CloudUpload,
} from '@mui/icons-material';

interface AddMentorDialogProps {
  open: boolean;
  onClose: () => void;
  onSubmit: (mentorData: any) => void;
}

export const AddMentorDialog: React.FC<AddMentorDialogProps> = ({
  open,
  onClose,
  onSubmit,
}) => {
  const [formData, setFormData] = useState({
    name: '',
    emailAddress: '',
    phoneNumber: '',
    location: '',
    image: null as File | null,
    externalTrainer: false,
    supervisor: '',
    supervisorAssigned: '',
  });

  const handleSubmit = () => {
    onSubmit(formData);
    setFormData({
      name: '',
      emailAddress: '',
      phoneNumber: '',
      location: '',
      image: null,
      externalTrainer: false,
      supervisor: '',
      supervisorAssigned: '',
    });
    onClose();
  };

  const handleInputChange = (field: string) => (event: any) => {
    setFormData(prev => ({
      ...prev,
      [field]: event.target.value
    }));
  };

  const handleRadioChange = (event: any) => {
    setFormData(prev => ({
      ...prev,
      externalTrainer: event.target.value === 'true'
    }));
  };

  const handleImageUpload = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (file) {
      setFormData(prev => ({
        ...prev,
        image: file
      }));
    }
  };

  return (
    <Dialog 
      open={open} 
      onClose={onClose}
      maxWidth="sm"
      fullWidth
      PaperProps={{
        sx: {
          borderRadius: 3,
          maxHeight: '90vh',
        }
      }}
    >
      <DialogTitle sx={{ 
        display: 'flex', 
        justifyContent: 'space-between', 
        alignItems: 'center',
        pb: 2,
        borderBottom: '1px solid #e5e7eb'
      }}>
        <Typography component="span" variant="h6" sx={{ color: '#374151', fontWeight: 600 }}>
          Add New Mentor
        </Typography>
        <IconButton onClick={onClose} sx={{ color: '#6b7280' }}>
          <Close />
        </IconButton>
      </DialogTitle>

      <DialogContent sx={{ p: 3 }}>
        <Box sx={{ mt: 1 }}>
          {/* Name */}
          <Box sx={{ mb: 3 }}>
            <Typography variant="body2" sx={{ mb: 1, color: '#374151', fontWeight: 500 }}>
              <span style={{ color: '#ef4444' }}>*</span> Name
            </Typography>
            <TextField
              fullWidth
              placeholder=""
              value={formData.name}
              onChange={handleInputChange('name')}
              variant="outlined"
              sx={{
                '& .MuiOutlinedInput-root': {
                  backgroundColor: '#ffffff',
                  '& fieldset': {
                    borderColor: '#d1d5db',
                  },
                  '&:hover fieldset': {
                    borderColor: '#38aae1',
                  },
                  '&.Mui-focused fieldset': {
                    borderColor: '#38aae1',
                  },
                }
              }}
            />
          </Box>

          {/* Email Address */}
          <Box sx={{ mb: 3 }}>
            <Typography variant="body2" sx={{ mb: 1, color: '#374151', fontWeight: 500 }}>
              <span style={{ color: '#ef4444' }}>*</span> Email Adress
            </Typography>
            <TextField
              fullWidth
              placeholder=""
              type="email"
              value={formData.emailAddress}
              onChange={handleInputChange('emailAddress')}
              variant="outlined"
              sx={{
                '& .MuiOutlinedInput-root': {
                  backgroundColor: '#ffffff',
                  '& fieldset': {
                    borderColor: '#d1d5db',
                  },
                  '&:hover fieldset': {
                    borderColor: '#38aae1',
                  },
                  '&.Mui-focused fieldset': {
                    borderColor: '#38aae1',
                  },
                }
              }}
            />
          </Box>

          {/* External Trainer and Phone Number */}
          <Grid container spacing={3} sx={{ mb: 3 }}>
            <Grid item xs={6}>
              <Typography variant="body2" sx={{ mb: 2, color: '#374151', fontWeight: 500 }}>
                External Trainer
              </Typography>
              <RadioGroup
                value={formData.externalTrainer ? 'true' : 'false'}
                onChange={handleRadioChange}
                sx={{ flexDirection: 'row' }}
              >
                <FormControlLabel 
                  value="true" 
                  control={
                    <Radio 
                      sx={{
                        color: '#d1d5db',
                        '&.Mui-checked': {
                          color: '#38aae1',
                        },
                      }}
                    />
                  } 
                  label={
                    <Typography variant="body2" sx={{ color: '#374151' }}>
                      Yes
                    </Typography>
                  }
                />
                <FormControlLabel 
                  value="false" 
                  control={
                    <Radio 
                      sx={{
                        color: '#d1d5db',
                        '&.Mui-checked': {
                          color: '#38aae1',
                        },
                      }}
                    />
                  } 
                  label={
                    <Typography variant="body2" sx={{ color: '#374151' }}>
                      No
                    </Typography>
                  }
                />
              </RadioGroup>
            </Grid>
            <Grid item xs={6}>
              <Typography variant="body2" sx={{ mb: 1, color: '#374151', fontWeight: 500 }}>
                <span style={{ color: '#ef4444' }}>*</span> Phone Number
              </Typography>
              <TextField
                fullWidth
                placeholder=""
                value={formData.phoneNumber}
                onChange={handleInputChange('phoneNumber')}
                variant="outlined"
                sx={{
                  '& .MuiOutlinedInput-root': {
                    backgroundColor: '#ffffff',
                    '& fieldset': {
                      borderColor: '#d1d5db',
                    },
                    '&:hover fieldset': {
                      borderColor: '#38aae1',
                    },
                    '&.Mui-focused fieldset': {
                      borderColor: '#38aae1',
                    },
                  }
                }}
              />
            </Grid>
          </Grid>

          {/* Image and Location */}
          <Grid container spacing={3} sx={{ mb: 3 }}>
            <Grid item xs={6}>
              <Typography variant="body2" sx={{ mb: 1, color: '#374151', fontWeight: 500 }}>
                Image
              </Typography>
              <Paper
                sx={{
                  height: 120,
                  border: '1px solid #d1d5db',
                  borderRadius: 1,
                  display: 'flex',
                  flexDirection: 'column',
                  alignItems: 'center',
                  justifyContent: 'center',
                  backgroundColor: '#ffffff',
                  cursor: 'pointer',
                  '&:hover': {
                    backgroundColor: '#f9fafb',
                    borderColor: '#38aae1',
                  }
                }}
                onClick={() => document.getElementById('mentor-image-upload')?.click()}
              >
                <CloudUpload sx={{ fontSize: 24, color: '#9ca3af', mb: 1 }} />
                <Typography variant="body2" sx={{ color: '#6b7280', fontWeight: 400 }}>
                  Image
                </Typography>
                <input
                  id="mentor-image-upload"
                  type="file"
                  accept="image/*"
                  style={{ display: 'none' }}
                  onChange={handleImageUpload}
                />
              </Paper>
              {formData.image && (
                <Typography variant="caption" sx={{ color: '#6b7280', mt: 1, display: 'block' }}>
                  {formData.image.name}
                </Typography>
              )}
            </Grid>
            <Grid item xs={6}>
              <Typography variant="body2" sx={{ mb: 1, color: '#374151', fontWeight: 500 }}>
                Location
              </Typography>
              <FormControl fullWidth>
                <Select
                  value={formData.location}
                  onChange={handleInputChange('location')}
                  displayEmpty
                  placeholder="Select..."
                  sx={{
                    backgroundColor: '#ffffff',
                    '& .MuiOutlinedInput-notchedOutline': {
                      borderColor: '#d1d5db',
                    },
                    '&:hover .MuiOutlinedInput-notchedOutline': {
                      borderColor: '#38aae1',
                    },
                    '&.Mui-focused .MuiOutlinedInput-notchedOutline': {
                      borderColor: '#38aae1',
                    },
                  }}
                >
                  <MenuItem value="" disabled>
                    <span style={{ color: '#9ca3af' }}>Select...</span>
                  </MenuItem>
                  <MenuItem value="nairobi">Nairobi</MenuItem>
                  <MenuItem value="mombasa">Mombasa</MenuItem>
                  <MenuItem value="kisumu">Kisumu</MenuItem>
                  <MenuItem value="nakuru">Nakuru</MenuItem>
                  <MenuItem value="eldoret">Eldoret</MenuItem>
                </Select>
              </FormControl>
            </Grid>
          </Grid>

          {/* Supervisor and Supervisor Assigned */}
          <Grid container spacing={3} sx={{ mb: 4 }}>
            <Grid item xs={6}>
              <Typography variant="body2" sx={{ mb: 1, color: '#374151', fontWeight: 500 }}>
                <span style={{ color: '#ef4444' }}>*</span> Supervisor
              </Typography>
              <RadioGroup
                value={formData.supervisor}
                onChange={handleInputChange('supervisor')}
                sx={{ flexDirection: 'row' }}
              >
                <FormControlLabel 
                  value="yes" 
                  control={
                    <Radio 
                      sx={{
                        color: '#d1d5db',
                        '&.Mui-checked': {
                          color: '#38aae1',
                        },
                      }}
                    />
                  } 
                  label={
                    <Typography variant="body2" sx={{ color: '#374151' }}>
                      Yes
                    </Typography>
                  }
                />
                <FormControlLabel 
                  value="no" 
                  control={
                    <Radio 
                      sx={{
                        color: '#d1d5db',
                        '&.Mui-checked': {
                          color: '#38aae1',
                        },
                      }}
                    />
                  } 
                  label={
                    <Typography variant="body2" sx={{ color: '#374151' }}>
                      No
                    </Typography>
                  }
                />
              </RadioGroup>
            </Grid>
            <Grid item xs={6}>
              <Typography variant="body2" sx={{ mb: 1, color: '#374151', fontWeight: 500 }}>
                Supervisor Assigned
              </Typography>
              <FormControl fullWidth>
                <Select
                  value={formData.supervisorAssigned}
                  onChange={handleInputChange('supervisorAssigned')}
                  displayEmpty
                  sx={{
                    backgroundColor: '#ffffff',
                    '& .MuiOutlinedInput-notchedOutline': {
                      borderColor: '#d1d5db',
                    },
                    '&:hover .MuiOutlinedInput-notchedOutline': {
                      borderColor: '#38aae1',
                    },
                    '&.Mui-focused .MuiOutlinedInput-notchedOutline': {
                      borderColor: '#38aae1',
                    },
                  }}
                >
                  <MenuItem value="" disabled>
                    <span style={{ color: '#9ca3af' }}>Search to Select</span>
                  </MenuItem>
                  <MenuItem value="john-doe">John Doe</MenuItem>
                  <MenuItem value="jane-smith">Jane Smith</MenuItem>
                  <MenuItem value="mike-johnson">Mike Johnson</MenuItem>
                </Select>
              </FormControl>
            </Grid>
          </Grid>

          {/* Submit Button */}
          <Button
            fullWidth
            variant="contained"
            onClick={handleSubmit}
            sx={{
              backgroundColor: '#60a5fa',
              color: '#ffffff',
              py: 1.5,
              fontSize: '14px',
              fontWeight: 400,
              textTransform: 'uppercase',
              borderRadius: 1,
              boxShadow: 'none',
              '&:hover': {
                backgroundColor: '#3b82f6',
                boxShadow: 'none',
              },
            }}
          >
            CREATE NEW MENTOR
          </Button>
        </Box>
      </DialogContent>
    </Dialog>
  );
};