import React, { useState } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  Box,
  TextField,
  Button,
  IconButton,
  Typography,
  Select,
  MenuItem,
  FormControl,
  Grid,
  Toolbar,
  Divider,
  Paper,
} from '@mui/material';
import {
  Close,
  FormatBold,
  FormatItalic,
  FormatUnderlined,
  Link,
  FormatListBulleted,
  FormatListNumbered,
  TextFields,
  CloudUpload,
} from '@mui/icons-material';

interface AddCourseDialogProps {
  open: boolean;
  onClose: () => void;
  onSubmit: (courseData: any) => void;
}

export const AddCourseDialog: React.FC<AddCourseDialogProps> = ({
  open,
  onClose,
  onSubmit,
}) => {
  const [formData, setFormData] = useState({
    courseName: '',
    image: null as File | null,
    preRequisiteCourses: '',
    category: '',
    numberOfSessions: '',
    courseDescription: '',
  });

  const handleSubmit = () => {
    onSubmit(formData);
    setFormData({
      courseName: '',
      image: null,
      preRequisiteCourses: '',
      category: '',
      numberOfSessions: '',
      courseDescription: '',
    });
    onClose();
  };

  const handleInputChange = (field: string) => (event: any) => {
    setFormData(prev => ({
      ...prev,
      [field]: event.target.value
    }));
  };

  const handleImageUpload = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (file) {
      setFormData(prev => ({
        ...prev,
        image: file
      }));
    }
  };

  return (
    <Dialog 
      open={open} 
      onClose={onClose}
      maxWidth="md"
      fullWidth
      PaperProps={{
        sx: {
          borderRadius: 3,
          maxHeight: '90vh',
        }
      }}
    >
      <DialogTitle sx={{ 
        display: 'flex', 
        justifyContent: 'space-between', 
        alignItems: 'center',
        pb: 2,
        borderBottom: '1px solid #e5e7eb'
      }}>
        <Typography component="span" variant="h6" sx={{ color: '#374151', fontWeight: 500 }}>
          New Course
        </Typography>
        <IconButton onClick={onClose} sx={{ color: '#6b7280' }}>
          <Close />
        </IconButton>
      </DialogTitle>

      <DialogContent sx={{ p: 3 }}>
        <Box sx={{ mt: 1 }}>
          {/* Course Name */}
          <Box sx={{ mb: 3 }}>
            <Typography variant="body2" sx={{ mb: 1, color: '#374151', fontWeight: 500 }}>
              Course <span style={{ color: '#ef4444' }}>*</span>
            </Typography>
            <TextField
              fullWidth
              placeholder="Enter course name"
              value={formData.courseName}
              onChange={handleInputChange('courseName')}
              variant="outlined"
              sx={{
                '& .MuiOutlinedInput-root': {
                  backgroundColor: '#ffffff',
                  '& fieldset': {
                    borderColor: '#d1d5db',
                  },
                  '&:hover fieldset': {
                    borderColor: '#38aae1',
                  },
                  '&.Mui-focused fieldset': {
                    borderColor: '#38aae1',
                  },
                }
              }}
            />
          </Box>

          {/* Image Upload and Pre-requisite Courses */}
          <Grid container spacing={3} sx={{ mb: 3 }}>
            <Grid item xs={6}>
              <Typography variant="body2" sx={{ mb: 1, color: '#374151', fontWeight: 500 }}>
                Image
              </Typography>
              <Paper
                sx={{
                  height: 120,
                  border: '2px dashed #60a5fa',
                  borderRadius: 1,
                  display: 'flex',
                  flexDirection: 'column',
                  alignItems: 'center',
                  justifyContent: 'center',
                  backgroundColor: '#ffffff',
                  cursor: 'pointer',
                  '&:hover': {
                    backgroundColor: '#f9fafb',
                    borderColor: '#3b82f6',
                  }
                }}
                onClick={() => document.getElementById('image-upload')?.click()}
              >
                <CloudUpload sx={{ fontSize: 32, color: '#9ca3af', mb: 1 }} />
                <Typography variant="body2" sx={{ color: '#6b7280', fontWeight: 400 }}>
                  Image
                </Typography>
                <Typography variant="body2" sx={{ color: '#60a5fa', fontWeight: 500, mt: 0.5 }}>
                  Upload Image
                </Typography>
                <input
                  id="image-upload"
                  type="file"
                  accept="image/*"
                  style={{ display: 'none' }}
                  onChange={handleImageUpload}
                />
              </Paper>
              {formData.image && (
                <Typography variant="caption" sx={{ color: '#6b7280', mt: 1, display: 'block' }}>
                  {formData.image.name}
                </Typography>
              )}
            </Grid>
            <Grid item xs={6}>
              <Typography variant="body2" sx={{ mb: 1, color: '#374151', fontWeight: 500 }}>
                Pre-requisite Courses
              </Typography>
              <FormControl fullWidth>
                <Select
                  value={formData.preRequisiteCourses}
                  onChange={handleInputChange('preRequisiteCourses')}
                  displayEmpty
                  sx={{
                    backgroundColor: '#ffffff',
                    '& .MuiOutlinedInput-notchedOutline': {
                      borderColor: '#d1d5db',
                    },
                    '&:hover .MuiOutlinedInput-notchedOutline': {
                      borderColor: '#38aae1',
                    },
                    '&.Mui-focused .MuiOutlinedInput-notchedOutline': {
                      borderColor: '#38aae1',
                    },
                  }}
                >
                  <MenuItem value="" disabled>
                    <span style={{ color: '#9ca3af' }}>Search to Select</span>
                  </MenuItem>
                  <MenuItem value="python-basics">Python Basics</MenuItem>
                  <MenuItem value="web-development">Web Development</MenuItem>
                  <MenuItem value="data-science">Data Science</MenuItem>
                </Select>
              </FormControl>
            </Grid>
          </Grid>

          {/* Category and Number of Sessions */}
          <Grid container spacing={3} sx={{ mb: 3 }}>
            <Grid item xs={6}>
              <Typography variant="body2" sx={{ mb: 1, color: '#374151', fontWeight: 500 }}>
                Category
              </Typography>
              <FormControl fullWidth>
                <Select
                  value={formData.category}
                  onChange={handleInputChange('category')}
                  displayEmpty
                  sx={{
                    backgroundColor: '#ffffff',
                    '& .MuiOutlinedInput-notchedOutline': {
                      borderColor: '#d1d5db',
                    },
                    '&:hover .MuiOutlinedInput-notchedOutline': {
                      borderColor: '#38aae1',
                    },
                    '&.Mui-focused .MuiOutlinedInput-notchedOutline': {
                      borderColor: '#38aae1',
                    },
                  }}
                >
                  <MenuItem value="" disabled>
                    <span style={{ color: '#9ca3af' }}>Category</span>
                  </MenuItem>
                  <MenuItem value="programming">Programming</MenuItem>
                  <MenuItem value="web-development">Web Development</MenuItem>
                  <MenuItem value="mobile-development">Mobile Development</MenuItem>
                  <MenuItem value="computer-science">Computer Science</MenuItem>
                  <MenuItem value="database">Database</MenuItem>
                </Select>
              </FormControl>
            </Grid>
            <Grid item xs={6}>
              <Typography variant="body2" sx={{ mb: 1, color: '#374151', fontWeight: 500 }}>
                No. Sessions
              </Typography>
              <TextField
                fullWidth
                placeholder="Enter number of sessions"
                value={formData.numberOfSessions}
                onChange={handleInputChange('numberOfSessions')}
                variant="outlined"
                sx={{
                  '& .MuiOutlinedInput-root': {
                    backgroundColor: '#ffffff',
                    '& fieldset': {
                      borderColor: '#d1d5db',
                    },
                    '&:hover fieldset': {
                      borderColor: '#38aae1',
                    },
                    '&.Mui-focused fieldset': {
                      borderColor: '#38aae1',
                    },
                  }
                }}
              />
            </Grid>
          </Grid>

          {/* Course Description */}
          <Box sx={{ mb: 4 }}>
            <Typography variant="body2" sx={{ mb: 1, color: '#374151', fontWeight: 500 }}>
              Course Description <span style={{ color: '#ef4444' }}>*</span>
            </Typography>
            
            {/* Rich Text Editor Toolbar */}
            <Box sx={{ 
              border: '1px solid #d1d5db', 
              borderRadius: '4px 4px 0 0',
              backgroundColor: '#ffffff'
            }}>
              <Toolbar sx={{ 
                minHeight: '40px !important', 
                px: 1,
                borderBottom: '1px solid #d1d5db',
                backgroundColor: '#f9fafb'
              }}>
                <Select
                  value="normal"
                  size="small"
                  sx={{ 
                    mr: 1, 
                    minWidth: 70,
                    fontSize: '12px',
                    '& .MuiOutlinedInput-notchedOutline': { border: 'none' }
                  }}
                >
                  <MenuItem value="normal">Normal</MenuItem>
                </Select>
                <Divider orientation="vertical" flexItem sx={{ mx: 0.5 }} />
                <IconButton size="small" sx={{ color: '#6b7280', padding: '4px' }}>
                  <FormatBold fontSize="small" />
                </IconButton>
                <IconButton size="small" sx={{ color: '#6b7280', padding: '4px' }}>
                  <FormatItalic fontSize="small" />
                </IconButton>
                <IconButton size="small" sx={{ color: '#6b7280', padding: '4px' }}>
                  <FormatUnderlined fontSize="small" />
                </IconButton>
                <IconButton size="small" sx={{ color: '#6b7280', padding: '4px' }}>
                  <Link fontSize="small" />
                </IconButton>
                <IconButton size="small" sx={{ color: '#6b7280', padding: '4px' }}>
                  <FormatListBulleted fontSize="small" />
                </IconButton>
                <IconButton size="small" sx={{ color: '#6b7280', padding: '4px' }}>
                  <FormatListNumbered fontSize="small" />
                </IconButton>
                <IconButton size="small" sx={{ color: '#6b7280', padding: '4px' }}>
                  <TextFields fontSize="small" />
                </IconButton>
              </Toolbar>
              <TextField
                fullWidth
                multiline
                rows={4}
                placeholder="Enter course description..."
                value={formData.courseDescription}
                onChange={handleInputChange('courseDescription')}
                variant="outlined"
                sx={{
                  '& .MuiOutlinedInput-root': {
                    backgroundColor: '#ffffff',
                    borderRadius: '0 0 4px 4px',
                    '& fieldset': {
                      border: 'none',
                    },
                  }
                }}
              />
            </Box>
          </Box>

          {/* Submit Button */}
          <Button
            fullWidth
            variant="contained"
            onClick={handleSubmit}
            sx={{
              backgroundColor: '#60a5fa',
              color: '#ffffff',
              py: 1.5,
              fontSize: '14px',
              fontWeight: 500,
              textTransform: 'uppercase',
              borderRadius: 1,
              boxShadow: 'none',
              '&:hover': {
                backgroundColor: '#3b82f6',
                boxShadow: 'none',
              },
            }}
          >
            ADD COURSE
          </Button>
        </Box>
      </DialogContent>
    </Dialog>
  );
};