import React, { useState } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  Box,
  TextField,
  Button,
  IconButton,
  Typography,
  Select,
  MenuItem,
  FormControl,
  InputLabel,
  Switch,
  FormControlLabel,
  Grid,
  Toolbar,
  Divider,
} from '@mui/material';
import {
  Close,
  FormatBold,
  FormatItalic,
  FormatUnderlined,
  Link,
  FormatListBulleted,
  FormatListNumbered,
  TextFields,
} from '@mui/icons-material';

interface AddClassDialogProps {
  open: boolean;
  onClose: () => void;
  onSubmit: (classData: any) => void;
}

export const AddClassDialog: React.FC<AddClassDialogProps> = ({
  open,
  onClose,
  onSubmit,
}) => {
  const [formData, setFormData] = useState({
    className: '',
    expectedOutcome: '',
    classType: '',
    classDescription: '',
    ageFrom: '',
    ageTo: '',
    requiredSessions: '',
    status: true,
  });

  const handleSubmit = () => {
    onSubmit(formData);
    setFormData({
      className: '',
      expectedOutcome: '',
      classType: '',
      classDescription: '',
      ageFrom: '',
      ageTo: '',
      requiredSessions: '',
      status: true,
    });
    onClose();
  };

  const handleInputChange = (field: string) => (event: any) => {
    setFormData(prev => ({
      ...prev,
      [field]: event.target.value
    }));
  };

  const handleSwitchChange = (event: any) => {
    setFormData(prev => ({
      ...prev,
      status: event.target.checked
    }));
  };

  return (
    <Dialog 
      open={open} 
      onClose={onClose}
      maxWidth="md"
      fullWidth
      PaperProps={{
        sx: {
          borderRadius: 3,
          maxHeight: '90vh',
        }
      }}
    >
      <DialogTitle sx={{ 
        display: 'flex', 
        justifyContent: 'space-between', 
        alignItems: 'center',
        pb: 2,
        borderBottom: '1px solid #e0e7ff'
      }}>
        <Typography component="span" variant="h5" sx={{ color: '#64748b', fontWeight: 600 }}>
          Add New Class
        </Typography>
        <IconButton onClick={onClose} sx={{ color: '#64748b' }}>
          <Close />
        </IconButton>
      </DialogTitle>

      <DialogContent sx={{ p: 3 }}>
        <Box sx={{ mt: 1 }}>
          {/* Class Name */}
          <Box sx={{ mb: 3 }}>
            <Typography variant="body2" sx={{ mb: 1, color: '#64748b', fontWeight: 500 }}>
              Class Name <span style={{ color: '#ef4444' }}>*</span>
            </Typography>
            <TextField
              fullWidth
              placeholder="Enter class name"
              value={formData.className}
              onChange={handleInputChange('className')}
              variant="outlined"
              sx={{
                '& .MuiOutlinedInput-root': {
                  backgroundColor: '#f8fafc',
                  '& fieldset': {
                    borderColor: '#e2e8f0',
                  },
                }
              }}
            />
          </Box>

          {/* Expected Outcome and Class Type */}
          <Grid container spacing={2} sx={{ mb: 3 }}>
            <Grid item xs={6}>
              <Typography variant="body2" sx={{ mb: 1, color: '#64748b', fontWeight: 500 }}>
                Expected Outcome <span style={{ color: '#ef4444' }}>*</span>
              </Typography>
              <TextField
                fullWidth
                placeholder="Enter expected outcome"
                value={formData.expectedOutcome}
                onChange={handleInputChange('expectedOutcome')}
                variant="outlined"
                sx={{
                  '& .MuiOutlinedInput-root': {
                    backgroundColor: '#f8fafc',
                    '& fieldset': {
                      borderColor: '#e2e8f0',
                    },
                  }
                }}
              />
            </Grid>
            <Grid item xs={6}>
              <Typography variant="body2" sx={{ mb: 1, color: '#64748b', fontWeight: 500 }}>
                Class Type <span style={{ color: '#ef4444' }}>*</span>
              </Typography>
              <FormControl fullWidth>
                <Select
                  value={formData.classType}
                  onChange={handleInputChange('classType')}
                  displayEmpty
                  sx={{
                    backgroundColor: '#f8fafc',
                    '& .MuiOutlinedInput-notchedOutline': {
                      borderColor: '#e2e8f0',
                    },
                  }}
                >
                  <MenuItem value="" disabled>
                    <span style={{ color: '#94a3b8' }}>Select class type</span>
                  </MenuItem>
                  <MenuItem value="beginner">Beginner</MenuItem>
                  <MenuItem value="intermediate">Intermediate</MenuItem>
                  <MenuItem value="advanced">Advanced</MenuItem>
                </Select>
              </FormControl>
            </Grid>
          </Grid>

          {/* Class Description */}
          <Box sx={{ mb: 3 }}>
            <Typography variant="body2" sx={{ mb: 1, color: '#64748b', fontWeight: 500 }}>
              Class Description <span style={{ color: '#ef4444' }}>*</span>
            </Typography>
            
            {/* Rich Text Editor Toolbar */}
            <Box sx={{ 
              border: '1px solid #e2e8f0', 
              borderRadius: '8px 8px 0 0',
              backgroundColor: '#f8fafc'
            }}>
              <Toolbar sx={{ 
                minHeight: '48px !important', 
                px: 2,
                borderBottom: '1px solid #e2e8f0'
              }}>
                <Select
                  value="normal"
                  size="small"
                  sx={{ 
                    mr: 1, 
                    minWidth: 80,
                    '& .MuiOutlinedInput-notchedOutline': { border: 'none' }
                  }}
                >
                  <MenuItem value="normal">Normal</MenuItem>
                  <MenuItem value="heading1">Heading 1</MenuItem>
                  <MenuItem value="heading2">Heading 2</MenuItem>
                </Select>
                <Divider orientation="vertical" flexItem sx={{ mx: 1 }} />
                <IconButton size="small" sx={{ color: '#64748b' }}>
                  <FormatBold />
                </IconButton>
                <IconButton size="small" sx={{ color: '#64748b' }}>
                  <FormatItalic />
                </IconButton>
                <IconButton size="small" sx={{ color: '#64748b' }}>
                  <FormatUnderlined />
                </IconButton>
                <IconButton size="small" sx={{ color: '#64748b' }}>
                  <Link />
                </IconButton>
                <IconButton size="small" sx={{ color: '#64748b' }}>
                  <FormatListBulleted />
                </IconButton>
                <IconButton size="small" sx={{ color: '#64748b' }}>
                  <FormatListNumbered />
                </IconButton>
                <IconButton size="small" sx={{ color: '#64748b' }}>
                  <TextFields />
                </IconButton>
              </Toolbar>
              <TextField
                fullWidth
                multiline
                rows={4}
                placeholder="Enter class description..."
                value={formData.classDescription}
                onChange={handleInputChange('classDescription')}
                variant="outlined"
                sx={{
                  '& .MuiOutlinedInput-root': {
                    backgroundColor: '#ffffff',
                    borderRadius: '0 0 8px 8px',
                    '& fieldset': {
                      border: 'none',
                    },
                  }
                }}
              />
            </Box>
          </Box>

          {/* Age Range */}
          <Grid container spacing={2} sx={{ mb: 3 }}>
            <Grid item xs={6}>
              <Typography variant="body2" sx={{ mb: 1, color: '#64748b', fontWeight: 500 }}>
                Age From <span style={{ color: '#ef4444' }}>*</span>
              </Typography>
              <TextField
                fullWidth
                placeholder="Enter minimum age"
                value={formData.ageFrom}
                onChange={handleInputChange('ageFrom')}
                type="number"
                variant="outlined"
                sx={{
                  '& .MuiOutlinedInput-root': {
                    backgroundColor: '#f8fafc',
                    '& fieldset': {
                      borderColor: '#e2e8f0',
                    },
                  }
                }}
              />
            </Grid>
            <Grid item xs={6}>
              <Typography variant="body2" sx={{ mb: 1, color: '#64748b', fontWeight: 500 }}>
                Age To <span style={{ color: '#ef4444' }}>*</span>
              </Typography>
              <TextField
                fullWidth
                placeholder="Enter maximum age"
                value={formData.ageTo}
                onChange={handleInputChange('ageTo')}
                type="number"
                variant="outlined"
                sx={{
                  '& .MuiOutlinedInput-root': {
                    backgroundColor: '#f8fafc',
                    '& fieldset': {
                      borderColor: '#e2e8f0',
                    },
                  }
                }}
              />
            </Grid>
          </Grid>

          {/* Required Sessions and Status */}
          <Grid container spacing={2} sx={{ mb: 4 }}>
            <Grid item xs={6}>
              <Typography variant="body2" sx={{ mb: 1, color: '#64748b', fontWeight: 500 }}>
                Required Sessions <span style={{ color: '#ef4444' }}>*</span>
              </Typography>
              <TextField
                fullWidth
                placeholder="Enter number of sessions"
                value={formData.requiredSessions}
                onChange={handleInputChange('requiredSessions')}
                type="number"
                variant="outlined"
                sx={{
                  '& .MuiOutlinedInput-root': {
                    backgroundColor: '#f8fafc',
                    '& fieldset': {
                      borderColor: '#e2e8f0',
                    },
                  }
                }}
              />
            </Grid>
            <Grid item xs={6}>
              <Typography variant="body2" sx={{ mb: 1, color: '#64748b', fontWeight: 500 }}>
                Status <span style={{ color: '#ef4444' }}>*</span>
              </Typography>
              <Box sx={{ display: 'flex', alignItems: 'center', mt: 1 }}>
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.status}
                      onChange={handleSwitchChange}
                      sx={{
                        '& .MuiSwitch-switchBase.Mui-checked': {
                          color: '#38aae1',
                        },
                        '& .MuiSwitch-switchBase.Mui-checked + .MuiSwitch-track': {
                          backgroundColor: '#38aae1',
                        },
                      }}
                    />
                  }
                  label={
                    <Typography sx={{ color: '#64748b', fontWeight: 500 }}>
                      {formData.status ? 'Active' : 'Inactive'}
                    </Typography>
                  }
                />
              </Box>
            </Grid>
          </Grid>

          {/* Submit Button */}
          <Button
            fullWidth
            variant="contained"
            onClick={handleSubmit}
            sx={{
              backgroundColor: '#38aae1',
              color: '#ffffff',
              py: 1.5,
              fontSize: '16px',
              fontWeight: 600,
              textTransform: 'none',
              borderRadius: 2,
              '&:hover': {
                backgroundColor: '#2563eb',
              },
            }}
          >
            Add Class
          </Button>
        </Box>
      </DialogContent>
    </Dialog>
  );
};