import React, { Component, createRef } from 'react';
import { Box, Typography } from '@mui/material';
import { CloudUpload } from '@mui/icons-material';

interface ActivityVideoUploaderProps {
  onUpload: (file: File) => void;
  currentUrl?: string;
  currentTitle?: string;
  activityIndex: number;
}

interface ActivityVideoUploaderState {
  isDragging: boolean;
}

export class ActivityVideoUploader extends Component<ActivityVideoUploaderProps, ActivityVideoUploaderState> {
  private fileInputRef = createRef<HTMLInputElement>();

  constructor(props: ActivityVideoUploaderProps) {
    super(props);
    this.state = {
      isDragging: false
    };
  }

  handleDragEnter = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    this.setState({ isDragging: true });
  };

  handleDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    this.setState({ isDragging: false });
  };

  handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
  };

  handleDrop = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    this.setState({ isDragging: false });

    const files = e.dataTransfer.files;
    if (files && files.length > 0) {
      this.handleFile(files[0]);
    }
  };

  handleFileSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = e.target.files;
    if (files && files.length > 0) {
      this.handleFile(files[0]);
    }
  };

  handleFile = (file: File) => {
    // Validate file type
    if (!file.type.startsWith('video/')) {
      alert('Please upload a video file');
      return;
    }

    // Validate file size (100MB max)
    const maxSize = 100 * 1024 * 1024; // 100MB in bytes
    if (file.size > maxSize) {
      alert('Video file must be smaller than 100MB');
      return;
    }

    this.props.onUpload(file);
  };

  handleClick = () => {
    this.fileInputRef.current?.click();
  };

  render() {
    const { currentUrl, currentTitle } = this.props;
    const { isDragging } = this.state;

    return (
      <Box>
        <Box
          onClick={this.handleClick}
          onDragEnter={this.handleDragEnter}
          onDragOver={this.handleDragOver}
          onDragLeave={this.handleDragLeave}
          onDrop={this.handleDrop}
          sx={{
            border: '2px dashed #38aae1',
            borderRadius: 2,
            p: 4,
            textAlign: 'center',
            cursor: 'pointer',
            bgcolor: isDragging ? '#f0f9ff' : '#ffffff',
            transition: 'all 0.2s',
            '&:hover': {
              bgcolor: '#f0f9ff',
              borderColor: '#25476a'
            }
          }}
        >
          <CloudUpload sx={{ fontSize: 48, color: '#38aae1', mb: 2 }} />
          <Typography variant="h6" sx={{ fontWeight: 600, color: '#25476a', mb: 1 }}>
            {currentUrl ? 'Video Uploaded' : 'Click or drag video to upload'}
          </Typography>
          <Typography variant="body2" sx={{ color: '#64748b' }}>
            {currentUrl ? currentTitle || 'Video file selected' : 'Support for MP4, AVI, MOV, etc. (Max 100MB)'}
          </Typography>
        </Box>
        <input
          ref={this.fileInputRef}
          type="file"
          accept="video/*"
          onChange={this.handleFileSelect}
          style={{ display: 'none' }}
        />
      </Box>
    );
  }
}
