import React, { Component } from 'react';
import {
  ThemeProvider,
  createTheme,
  CssBaseline,
  Box,
  Drawer,
  List,
  Typography,
  ListItem,
  ListItemButton,
  ListItemIcon,
  ListItemText,
} from '@mui/material';
import {
  Dashboard,
  School,
  MenuBook,
  People,
  PersonAdd,
  FamilyRestroom,
  Schedule,
  Payment,
  Settings,
  TrendingUp,
  SelfImprovement,
  AccountCircle
} from '@mui/icons-material';
import { DashboardContent } from './components/DashboardContent';
import { SelfPacedLearning } from './components/SelfPacedLearning';
import { CourseAnalytics } from './components/CourseAnalytics';
import { TechnicalMentors } from './components/TechnicalMentors';
import { Learners } from './components/Learners';
import { Settings as SettingsComponent } from './components/Settings';
import { Courses } from './components/Courses';
import { Classes } from './components/Classes';
import { Profile } from './components/Profile';
import { Timetable } from './components/Timetable';
import { ParentDashboard } from './components/ParentDashboard';
import { ParentBilling } from './components/ParentBilling';

const drawerWidth = 280;

// Custom theme with the specified colors
const theme = createTheme({
  palette: {
    primary: {
      main: '#38aae1',
      contrastText: '#ffffff',
    },
    secondary: {
      main: '#feb139',
      contrastText: '#25476a',
    },
    background: {
      default: '#f8fafc',
      paper: '#ffffff',
    },
    text: {
      primary: '#25476a',
      secondary: '#64748b',
    },
  },
  typography: {
    fontFamily: '"Inter", "Roboto", "Helvetica", "Arial", sans-serif',
  },
  shape: {
    borderRadius: 12,
  },
});

const menuItems = [
  { text: 'Dashboard', icon: <Dashboard />, key: 'dashboard' },
  { text: 'Timetable', icon: <Schedule />, key: 'timetable' },
  { text: 'Technical Mentors', icon: <PersonAdd />, key: 'technical-mentors' },
  { text: 'Learners', icon: <School />, key: 'learners' },
  { text: 'Parents', icon: <FamilyRestroom />, key: 'parents' },
  { text: 'Classes', icon: <MenuBook />, key: 'classes' },
  { text: 'Courses', icon: <MenuBook />, key: 'courses' },
  { text: 'Self Paced', icon: <SelfImprovement />, key: 'self-paced' },
  { text: 'Analytics', icon: <TrendingUp />, key: 'analytics' },
  { text: 'Payment', icon: <Payment />, key: 'payment' },
  { text: 'Profile', icon: <AccountCircle />, key: 'profile' },
  { text: 'Settings', icon: <Settings />, key: 'settings' },
];

interface AppState {
  selectedPage: string;
}

class App extends Component<{}, AppState> {
  constructor(props: {}) {
    super(props);
    this.state = {
      selectedPage: 'dashboard'
    };
  }

  setSelectedPage = (page: string) => {
    this.setState({ selectedPage: page });
  };

  renderContent = () => {
    switch (this.state.selectedPage) {
      case 'dashboard':
        return <DashboardContent />;
      case 'timetable':
        return <Timetable />;
      case 'technical-mentors':
        return <TechnicalMentors />;
      case 'learners':
        return <Learners />;
      case 'parents':
        return <ParentDashboard />;
      case 'payment':
        return <ParentBilling />;
      case 'classes':
        return <Classes />;
      case 'courses':
        return <Courses />;
      case 'self-paced':
        return <SelfPacedLearning />;
      case 'analytics':
        return <CourseAnalytics />;
      case 'profile':
        return <Profile />;
      case 'settings':
        return <SettingsComponent />;
      default:
        return <SelfPacedLearning />;
    }
  };

  render() {
    const { selectedPage } = this.state;

    return (
      <ThemeProvider theme={theme}>
        <CssBaseline />
        <Box sx={{ display: 'flex' }}>
          <Drawer
            variant="permanent"
            sx={{
              width: drawerWidth,
              flexShrink: 0,
              '& .MuiDrawer-paper': {
                width: drawerWidth,
                boxSizing: 'border-box',
                backgroundColor: '#25476a',
                color: '#ffffff',
                borderRight: 'none',
                boxShadow: '4px 0 20px rgba(0,0,0,0.1)',
              },
            }}
          >
            <Box sx={{ p: 3, borderBottom: '1px solid rgba(255,255,255,0.12)' }}>
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                <Typography variant="h5" sx={{ color: 'white', fontWeight: 700 }}>
                  DigiUnzi
                </Typography>
              </Box>
              <Typography variant="body2" sx={{ color: 'rgba(255,255,255,0.8)', fontWeight: 500 }}>
                Learning Management System
              </Typography>
              <Typography variant="caption" sx={{ color: 'rgba(255,255,255,0.6)' }}>
                Superadmin Dashboard
              </Typography>
            </Box>
            
            <List sx={{ pt: 2 }}>
              {menuItems.map((item) => (
                <ListItem key={item.key} disablePadding sx={{ mb: 0.5 }}>
                  <ListItemButton
                    onClick={() => this.setSelectedPage(item.key)}
                    sx={{
                      mx: 1,
                      borderRadius: 2,
                      backgroundColor: selectedPage === item.key ? 'rgba(56, 170, 225, 0.25)' : 'transparent',
                      border: selectedPage === item.key ? '1px solid rgba(56, 170, 225, 0.3)' : '1px solid transparent',
                      mb: 0.5,
                      '&:hover': {
                        backgroundColor: 'rgba(255,255,255,0.1)',
                      },
                    }}
                  >
                    <ListItemIcon sx={{ color: '#ffffff', minWidth: '40px' }}>
                      {item.icon}
                    </ListItemIcon>
                    <ListItemText 
                      primary={item.text} 
                      sx={{
                        '& .MuiListItemText-primary': {
                          color: '#ffffff',
                          fontWeight: 500,
                        }
                      }}
                    />
                  </ListItemButton>
                </ListItem>
              ))}
            </List>
          </Drawer>

          <Box component="main" sx={{ flexGrow: 1, bgcolor: 'background.default', minHeight: '100vh' }}>
            {this.renderContent()}
          </Box>
        </Box>
      </ThemeProvider>
    );
  }
}

export default App;