// PictoBlox integration utilities

// Local storage keys
const STORAGE_KEYS = {
  USER_PROJECTS: 'digifunzi_pictoblox_projects',
  SESSION_PROGRESS: 'digifunzi_session_progress',
  CURRENT_PROJECT: 'digifunzi_current_project',
  AUTO_SAVE: 'digifunzi_auto_save_enabled'
};

export class PictoBloxManager {
  static instance;
  iframe = null;
  isConnected = false;
  messageHandlers = new Map();

  static getInstance() {
    if (!PictoBloxManager.instance) {
      PictoBloxManager.instance = new PictoBloxManager();
    }
    return PictoBloxManager.instance;
  }

  // Initialize connection with PictoBlox iframe
  initialize(iframe) {
    return new Promise((resolve) => {
      this.iframe = iframe;
      
      // Set up message listener for iframe communication
      const messageListener = (event) => {
        if (event.origin !== this.getPictoBloxOrigin()) return;
        
        const { type, data } = event.data;
        const handler = this.messageHandlers.get(type);
        if (handler) {
          handler(data);
        }
      };

      window.addEventListener('message', messageListener);
      
      // Wait for PictoBlox to signal ready
      this.onMessage('PICTOBLOX_READY', () => {
        this.isConnected = true;
        resolve(true);
      });

      // Timeout after 10 seconds
      setTimeout(() => {
        if (!this.isConnected) {
          console.warn('PictoBlox connection timeout');
          resolve(false);
        }
      }, 10000);
    });
  }

  // Register message handlers
  onMessage(type, handler) {
    this.messageHandlers.set(type, handler);
  }

  // Send message to PictoBlox iframe
  sendMessage(type, data) {
    if (!this.iframe || !this.isConnected) {
      console.warn('PictoBlox not connected');
      return;
    }

    this.iframe.contentWindow?.postMessage(
      { type, data },
      this.getPictoBloxOrigin()
    );
  }

  getPictoBloxOrigin() {
    return 'https://pictoblox.ai'; // Official PictoBlox domain
  }

  // Load a project into PictoBlox
  loadProject(project) {
    return new Promise((resolve) => {
      if (!this.isConnected) {
        resolve(false);
        return;
      }

      this.onMessage('PROJECT_LOADED', (success) => {
        resolve(success);
      });

      this.sendMessage('LOAD_PROJECT', {
        projectData: project.projectData,
        blocks: project.blocks,
        sprites: project.sprites,
        sounds: project.sounds,
        variables: project.variables
      });
    });
  }

  // Save current project from PictoBlox
  saveProject() {
    return new Promise((resolve) => {
      if (!this.isConnected) {
        resolve(null);
        return;
      }

      this.onMessage('PROJECT_DATA', (projectData) => {
        const project = {
          id: `project_${Date.now()}`,
          name: projectData.name || 'Untitled Project',
          description: projectData.description || '',
          projectData: projectData.serialized,
          blocks: projectData.blocks,
          sprites: projectData.sprites,
          sounds: projectData.sounds,
          variables: projectData.variables,
          updatedAt: new Date().toISOString()
        };
        resolve(project);
      });

      this.sendMessage('GET_PROJECT_DATA');
    });
  }

  // Set lesson context in PictoBlox
  setLessonContext(session) {
    this.sendMessage('SET_LESSON_CONTEXT', {
      sessionId: session.sessionId,
      sessionTitle: session.sessionTitle,
      ageGroup: session.ageGroup,
      objectives: session.lessonObjectives,
      requiredBlocks: session.requiredBlocks,
      difficulty: session.difficulty
    });
  }

  // Enable/disable specific blocks based on lesson
  configureBlocks(allowedBlocks, disabledBlocks = []) {
    this.sendMessage('CONFIGURE_BLOCKS', {
      allowed: allowedBlocks,
      disabled: disabledBlocks
    });
  }
}

// Local storage management
export class PictoBloxStorage {
  
  // Save project to local storage
  static saveProject(project) {
    try {
      const projects = this.getUserProjects();
      const existingIndex = projects.findIndex(p => p.id === project.id);
      
      if (existingIndex >= 0) {
        projects[existingIndex] = { ...project, updatedAt: new Date().toISOString() };
      } else {
        projects.push({ ...project, createdAt: new Date().toISOString() });
      }
      
      localStorage.setItem(STORAGE_KEYS.USER_PROJECTS, JSON.stringify(projects));
    } catch (error) {
      console.error('Failed to save project to localStorage:', error);
    }
  }

  // Get all user projects
  static getUserProjects() {
    try {
      const stored = localStorage.getItem(STORAGE_KEYS.USER_PROJECTS);
      return stored ? JSON.parse(stored) : [];
    } catch (error) {
      console.error('Failed to load projects from localStorage:', error);
      return [];
    }
  }

  // Get projects for specific session
  static getSessionProjects(sessionId) {
    return this.getUserProjects().filter(p => p.sessionId === sessionId);
  }

  // Save current project
  static saveCurrentProject(project) {
    try {
      localStorage.setItem(STORAGE_KEYS.CURRENT_PROJECT, JSON.stringify(project));
    } catch (error) {
      console.error('Failed to save current project:', error);
    }
  }

  // Get current project
  static getCurrentProject() {
    try {
      const stored = localStorage.getItem(STORAGE_KEYS.CURRENT_PROJECT);
      return stored ? JSON.parse(stored) : null;
    } catch (error) {
      console.error('Failed to load current project:', error);
      return null;
    }
  }

  // Export project as downloadable file
  static exportProject(project) {
    try {
      const dataStr = JSON.stringify(project, null, 2);
      const dataBlob = new Blob([dataStr], { type: 'application/json' });
      
      const link = document.createElement('a');
      link.href = URL.createObjectURL(dataBlob);
      link.download = `${project.name.replace(/[^a-z0-9]/gi, '_').toLowerCase()}.digifunzi`;
      link.click();
      
      URL.revokeObjectURL(link.href);
    } catch (error) {
      console.error('Failed to export project:', error);
    }
  }

  // Import project from file
  static importProject(file) {
    return new Promise((resolve, reject) => {
      const reader = new FileReader();
      
      reader.onload = (e) => {
        try {
          const project = JSON.parse(e.target?.result);
          project.id = `imported_${Date.now()}`;
          project.updatedAt = new Date().toISOString();
          resolve(project);
        } catch (error) {
          reject(new Error('Invalid project file format'));
        }
      };
      
      reader.onerror = () => reject(new Error('Failed to read file'));
      reader.readAsText(file);
    });
  }

  // Clear all stored data (for debugging/reset)
  static clearAll() {
    Object.values(STORAGE_KEYS).forEach(key => {
      localStorage.removeItem(key);
    });
  }
}

// Generate session-specific starter projects
export function generateSessionStarters(sessionId, sessionTitle, ageGroup) {
  const baseProject = {
    sessionId,
    ageGroup,
    createdAt: new Date().toISOString()
  };

  const starters = [];

  // Always include empty project
  starters.push({
    ...baseProject,
    id: `session_${sessionId}_empty`,
    name: 'Empty Project',
    description: 'Start with a blank canvas',
    blocks: [],
    sprites: [],
    sounds: [],
    variables: {}
  });

  // Session-specific starters based on title content
  if (sessionTitle.toLowerCase().includes('animation')) {
    starters.push({
      ...baseProject,
      id: `session_${sessionId}_walking`,
      name: 'Walking Character',
      description: 'Pre-configured sprite with walking animation blocks',
      blocks: [
        { type: 'event_whenflagclicked' },
        { type: 'looks_nextcostume' },
        { type: 'control_wait', inputs: { DURATION: 0.1 } },
        { type: 'control_forever' }
      ],
      sprites: [{ name: 'Character', costumes: ['walk1', 'walk2'] }],
      sounds: [],
      variables: {}
    });
  }

  if (sessionTitle.toLowerCase().includes('ai') || sessionTitle.toLowerCase().includes('recognition')) {
    starters.push({
      ...baseProject,
      id: `session_${sessionId}_ai`,
      name: 'AI Recognition Starter',
      description: 'Template with AI blocks and camera setup',
      blocks: [
        { type: 'ai_when_sees_object' },
        { type: 'looks_say' },
        { type: 'ai_camera_on' }
      ],
      sprites: [{ name: 'AI Assistant' }],
      sounds: [],
      variables: { confidence_threshold: 0.8 }
    });
  }

  // Age-specific additional starters
  if (ageGroup === 'crib') {
    starters.push({
      ...baseProject,
      id: `session_${sessionId}_colorful`,
      name: 'Colorful Fun',
      description: 'Bright colors and simple animations',
      blocks: [
        { type: 'looks_changecoloreffectby', inputs: { CHANGE: 25 } },
        { type: 'motion_turnright', inputs: { DEGREES: 15 } }
      ],
      sprites: [{ name: 'Rainbow Cat' }],
      sounds: ['happy_sound.wav'],
      variables: { color_speed: 1 }
    });
  } else if (ageGroup === 'clicker') {
    starters.push({
      ...baseProject,
      id: `session_${sessionId}_game`,
      name: 'Simple Game',
      description: 'Basic game mechanics template',
      blocks: [
        { type: 'event_whenkeypressed' },
        { type: 'motion_changexby', inputs: { DX: 10 } },
        { type: 'control_if' }
      ],
      sprites: [{ name: 'Player' }, { name: 'Target' }],
      sounds: ['point.wav', 'game_over.wav'],
      variables: { score: 0, lives: 3 }
    });
  } else {
    starters.push({
      ...baseProject,
      id: `session_${sessionId}_advanced`,
      name: 'Advanced Template',
      description: 'Complex project with multiple features',
      blocks: [
        { type: 'procedures_definition' },
        { type: 'data_variable' },
        { type: 'control_repeat' }
      ],
      sprites: [{ name: 'Main Character' }],
      sounds: [],
      variables: { algorithm_step: 1, optimization_level: 'medium' }
    });
  }

  return starters;
}

// Device detection for responsive features
export function getDeviceType() {
  const width = window.innerWidth;
  
  if (width < 768) return 'mobile';
  if (width < 1024) return 'tablet';
  return 'desktop';
}

// Check if device supports fullscreen
export function supportsFullscreen() {
  try {
    const elem = document.documentElement;
    const hasApi = !!(
      elem.requestFullscreen ||
      elem.webkitRequestFullscreen ||
      elem.mozRequestFullScreen ||
      elem.msRequestFullscreen
    );
    
    // Check if fullscreen is allowed by permissions policy
    if ('permissions' in navigator && hasApi) {
      // For modern browsers, check permissions policy
      try {
        return document.fullscreenEnabled !== false;
      } catch {
        return hasApi;
      }
    }
    
    return hasApi;
  } catch (error) {
    console.warn('Error checking fullscreen support:', error);
    return false;
  }
}

// Enter fullscreen mode
export function enterFullscreen(element) {
  return new Promise((resolve, reject) => {
    try {
      const elem = element;
      
      // Check if fullscreen is actually supported and allowed
      if (!supportsFullscreen()) {
        reject(new Error('Fullscreen not supported or not allowed'));
        return;
      }
      
      const requestMethod = 
        elem.requestFullscreen ||
        elem.webkitRequestFullscreen ||
        elem.mozRequestFullScreen ||
        elem.msRequestFullscreen;
      
      if (requestMethod) {
        const result = requestMethod.call(elem);
        
        // Handle both promise-based and callback-based APIs
        if (result && typeof result.then === 'function') {
          result
            .then(() => resolve())
            .catch((error) => {
              console.warn('Fullscreen request failed:', error);
              reject(new Error(`Fullscreen blocked: ${error.message || 'Permission denied'}`));
            });
        } else {
          // For older browsers without promise support
          setTimeout(() => {
            if (document.fullscreenElement || document.webkitFullscreenElement) {
              resolve();
            } else {
              reject(new Error('Fullscreen request failed'));
            }
          }, 100);
        }
      } else {
        reject(new Error('Fullscreen not supported'));
      }
    } catch (error) {
      console.warn('Fullscreen request error:', error);
      reject(new Error(`Fullscreen failed: ${error.message}`));
    }
  });
}

// Exit fullscreen mode
export function exitFullscreen() {
  return new Promise((resolve, reject) => {
    try {
      const doc = document;
      
      // Check if we're actually in fullscreen
      if (!document.fullscreenElement && !doc.webkitFullscreenElement && !doc.mozFullScreenElement) {
        resolve(); // Already not in fullscreen
        return;
      }
      
      const exitMethod = 
        doc.exitFullscreen ||
        doc.webkitExitFullscreen ||
        doc.mozCancelFullScreen ||
        doc.msExitFullscreen;
      
      if (exitMethod) {
        const result = exitMethod.call(doc);
        
        // Handle both promise-based and callback-based APIs
        if (result && typeof result.then === 'function') {
          result
            .then(() => resolve())
            .catch((error) => {
              console.warn('Exit fullscreen failed:', error);
              reject(new Error(`Exit fullscreen failed: ${error.message || 'Unknown error'}`));
            });
        } else {
          // For older browsers without promise support
          setTimeout(() => {
            if (!document.fullscreenElement && !doc.webkitFullscreenElement) {
              resolve();
            } else {
              reject(new Error('Exit fullscreen failed'));
            }
          }, 100);
        }
      } else {
        reject(new Error('Exit fullscreen not supported'));
      }
    } catch (error) {
      console.warn('Exit fullscreen error:', error);
      reject(new Error(`Exit fullscreen failed: ${error.message}`));
    }
  });
}