import { useState, useCallback } from 'react';

export const useVideoTracking = () => {
  const [videoSessions, setVideoSessions] = useState(new Map());

  const initializeSession = useCallback((sessionId) => {
    if (!videoSessions.has(sessionId)) {
      const newSession = {
        sessionId,
        videos: {
          mainConcepts: {
            id: `session-${sessionId}-main`,
            title: 'Main Concepts',
            duration: 780, // 13 minutes default
            watchedTime: 0,
            isCompleted: false,
            lastWatchedPosition: 0
          },
          assignment: {
            id: `session-${sessionId}-assignment`,
            title: 'Assignment Video',
            duration: 300, // 5 minutes default
            watchedTime: 0,
            isCompleted: false,
            lastWatchedPosition: 0
          },
          activity1: {
            id: `session-${sessionId}-activity1`,
            title: 'Activity 1',
            duration: 600, // 10 minutes default
            watchedTime: 0,
            isCompleted: false,
            lastWatchedPosition: 0
          },
          courseVideo: {
            id: `session-${sessionId}-course`,
            title: 'Course Video',
            duration: 900, // 15 minutes default
            watchedTime: 0,
            isCompleted: false,
            lastWatchedPosition: 0
          }
        }
      };

      setVideoSessions(prev => new Map(prev).set(sessionId, newSession));
    }
  }, [videoSessions]);

  const updateVideoProgress = useCallback((
    sessionId,
    videoType,
    currentTime,
    duration
  ) => {
    setVideoSessions(prev => {
      const updated = new Map(prev);
      const session = updated.get(sessionId);
      
      if (session) {
        const video = session.videos[videoType];
        const watchedTime = Math.max(video.watchedTime, currentTime);
        const isCompleted = watchedTime >= duration * 0.9; // 90% completion
        
        session.videos[videoType] = {
          ...video,
          duration,
          watchedTime,
          isCompleted,
          lastWatchedPosition: currentTime
        };
        
        updated.set(sessionId, session);
      }
      
      return updated;
    });
  }, []);

  const getVideoData = useCallback((sessionId, videoType) => {
    const session = videoSessions.get(sessionId);
    return session ? session.videos[videoType] : null;
  }, [videoSessions]);

  const getSessionProgress = useCallback((sessionId) => {
    const session = videoSessions.get(sessionId);
    if (!session) return 0;

    const videos = Object.values(session.videos);
    const totalDuration = videos.reduce((sum, video) => sum + video.duration, 0);
    const totalWatched = videos.reduce((sum, video) => sum + video.watchedTime, 0);
    
    return totalDuration > 0 ? Math.min(100, (totalWatched / totalDuration) * 100) : 0;
  }, [videoSessions]);

  const isSessionComplete = useCallback((sessionId) => {
    const session = videoSessions.get(sessionId);
    if (!session) return false;

    return Object.values(session.videos).every(video => video.isCompleted);
  }, [videoSessions]);

  const getTotalWatchTime = useCallback((sessionId) => {
    const session = videoSessions.get(sessionId);
    if (!session) return 0;

    return Object.values(session.videos).reduce((sum, video) => sum + video.watchedTime, 0);
  }, [videoSessions]);

  const formatTime = useCallback((seconds) => {
    const minutes = Math.floor(seconds / 60);
    const remainingSeconds = Math.floor(seconds % 60);
    return `${minutes}m ${remainingSeconds}s`;
  }, []);

  const getVideoWatchTime = useCallback((sessionId, videoId) => {
    const session = videoSessions.get(sessionId);
    if (!session) return 0;
    
    const videos = Object.values(session.videos);
    const video = videos.find(v => v.id.includes(videoId));
    return video ? video.watchedTime : 0;
  }, [videoSessions]);

  const getVideoProgress = useCallback((sessionId, videoId) => {
    const session = videoSessions.get(sessionId);
    if (!session) return 0;
    
    const videos = Object.values(session.videos);
    const video = videos.find(v => v.id.includes(videoId));
    if (!video || video.duration === 0) return 0;
    
    return Math.min(100, (video.watchedTime / video.duration) * 100);
  }, [videoSessions]);

  const getVideoPosition = useCallback((sessionId, videoId) => {
    const session = videoSessions.get(sessionId);
    if (!session) return 0;
    
    const videos = Object.values(session.videos);
    const video = videos.find(v => v.id.includes(videoId));
    return video ? video.lastWatchedPosition : 0;
  }, [videoSessions]);

  return {
    initializeSession,
    updateVideoProgress,
    getVideoData,
    getSessionProgress,
    isSessionComplete,
    getTotalWatchTime,
    formatTime,
    getVideoWatchTime,
    getVideoProgress,
    getVideoPosition,
    videoSessions
  };
};