import { useState } from 'react';
import { View } from '../types';
import { useVideoTracking } from './useVideoTracking';

export const useSessionManagement = () => {
  const [currentView, setCurrentView] = useState<View>('course-detail');
  const [currentSession, setCurrentSession] = useState<number>(1);
  const [completedSessions, setCompletedSessions] = useState<Set<number>>(new Set([]));
  const videoTracking = useVideoTracking();

  const handleStartCourse = () => {
    setCurrentView('session-detail');
    setCurrentSession(1);
    videoTracking.initializeSession(1);
  };

  const handleSessionClick = (sessionId: number) => {
    setCurrentSession(sessionId);
    setCurrentView('session-detail');
    videoTracking.initializeSession(sessionId);
  };

  const handleBackToCourse = () => {
    setCurrentView('course-detail');
  };

  const handlePreviousSession = () => {
    if (currentSession > 1) {
      setCurrentSession(currentSession - 1);
    }
  };

  const handleNextSession = (totalSessions: number) => {
    if (currentSession < totalSessions && completedSessions.has(currentSession)) {
      setCurrentSession(currentSession + 1);
    }
  };

  const handleSessionComplete = () => {
    const newCompleted = new Set(completedSessions);
    newCompleted.add(currentSession);
    setCompletedSessions(newCompleted);
  };

  const handleVideoProgress = (sessionId: number, videoId: string, progress: number) => {
    // Map video IDs to video types for the tracking system
    const videoTypeMap: Record<string, string> = {
      'introduction': 'courseVideo',
      'main-concepts': 'mainConcepts', 
      'activity-1': 'activity1',
      'assignment': 'assignment',
      'course-demo': 'courseVideo'
    };
    
    const videoType = videoTypeMap[videoId] || 'courseVideo';
    const duration = 300; // Default duration, this should be passed from video element
    const currentTime = (progress / 100) * duration;
    
    videoTracking.updateVideoProgress(sessionId, videoType as any, currentTime, duration);
  };

  return {
    currentView,
    currentSession,
    completedSessions,
    handleStartCourse,
    handleSessionClick,
    handleBackToCourse,
    handlePreviousSession,
    handleNextSession,
    handleSessionComplete,
    handleVideoProgress,
    videoTracking
  };
};