import { useState, useEffect, useCallback } from 'react';
import { PictoBloxStorage } from '../utils/pictoBloxUtils';

export function usePictoBloxIntegration(config = {}) {
  const [projects, setProjects] = useState([]);
  const [syncStatus, setSyncStatus] = useState('idle');
  const [lastSyncTime, setLastSyncTime] = useState(null);
  const [offlineMode, setOfflineMode] = useState(!navigator.onLine);

  // Monitor online/offline status
  useEffect(() => {
    const handleOnline = () => {
      setOfflineMode(false);
      // Auto-sync when coming back online
      if (projects.some(p => p.syncStatus === 'local')) {
        syncProjectsToLMS();
      }
    };

    const handleOffline = () => {
      setOfflineMode(true);
    };

    window.addEventListener('online', handleOnline);
    window.addEventListener('offline', handleOffline);

    return () => {
      window.removeEventListener('online', handleOnline);
      window.removeEventListener('offline', handleOffline);
    };
  }, [projects]);

  // Load projects from local storage on init
  useEffect(() => {
    loadProjectsFromStorage();
  }, []);

  const loadProjectsFromStorage = useCallback(() => {
    const localProjects = PictoBloxStorage.getUserProjects();
    const lmsProjects = localProjects.map(project => ({
      id: project.id,
      name: project.name,
      description: project.description,
      projectData: project.projectData || '',
      sessionId: project.sessionId || 1,
      userId: config.userId || 'anonymous',
      courseId: config.courseId || 'default',
      createdAt: project.createdAt || new Date().toISOString(),
      updatedAt: project.updatedAt || new Date().toISOString(),
      syncStatus: 'local' // All locally stored projects start as unsynced
    }));
    
    setProjects(lmsProjects);
  }, [config.userId, config.courseId]);

  // Save project locally and optionally to LMS
  const saveProject = useCallback(async (project) => {
    try {
      // Save to local storage first
      PictoBloxStorage.saveProject(project);

      const lmsProject = {
        id: project.id,
        name: project.name,
        description: project.description,
        projectData: project.projectData || '',
        sessionId: project.sessionId || 1,
        userId: config.userId || 'anonymous',
        courseId: config.courseId || 'default',
        createdAt: project.createdAt || new Date().toISOString(),
        updatedAt: new Date().toISOString(),
        syncStatus: offlineMode ? 'local' : 'syncing'
      };

      // Update local projects list
      setProjects(prev => {
        const existingIndex = prev.findIndex(p => p.id === project.id);
        if (existingIndex >= 0) {
          const updated = [...prev];
          updated[existingIndex] = lmsProject;
          return updated;
        }
        return [...prev, lmsProject];
      });

      // Try to sync to LMS if online and configured
      if (!offlineMode && config.apiEndpoint) {
        try {
          await syncProjectToLMS(lmsProject);
        } catch (error) {
          console.warn('Failed to sync project to LMS, saved locally:', error);
          // Update sync status to indicate error
          setProjects(prev => prev.map(p => 
            p.id === project.id ? { ...p, syncStatus: 'error' } : p
          ));
        }
      }

      return true;
    } catch (error) {
      console.error('Failed to save project:', error);
      return false;
    }
  }, [config, offlineMode]);

  // Sync single project to LMS
  const syncProjectToLMS = useCallback(async (project) => {
    if (!config.apiEndpoint) {
      throw new Error('No LMS API endpoint configured');
    }

    const response = await fetch(`${config.apiEndpoint}/projects`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        ...(config.authToken && { 'Authorization': `Bearer ${config.authToken}` })
      },
      body: JSON.stringify({
        id: project.id,
        name: project.name,
        description: project.description,
        projectData: project.projectData,
        sessionId: project.sessionId,
        userId: project.userId,
        courseId: project.courseId,
        metadata: {
          clientTimestamp: project.updatedAt,
          platform: 'digifunzi-web'
        }
      })
    });

    if (!response.ok) {
      throw new Error(`LMS sync failed: ${response.status} ${response.statusText}`);
    }

    // Update sync status
    setProjects(prev => prev.map(p => 
      p.id === project.id ? { ...p, syncStatus: 'synced' } : p
    ));
  }, [config.apiEndpoint, config.authToken]);

  // Sync all local projects to LMS
  const syncProjectsToLMS = useCallback(async () => {
    if (offlineMode || !config.apiEndpoint) {
      return;
    }

    setSyncStatus('syncing');
    
    try {
      const localProjects = projects.filter(p => p.syncStatus === 'local' || p.syncStatus === 'error');
      
      for (const project of localProjects) {
        try {
          await syncProjectToLMS(project);
        } catch (error) {
          console.error(`Failed to sync project ${project.id}:`, error);
          // Continue with other projects
        }
      }
      
      setLastSyncTime(new Date());
      setSyncStatus('idle');
    } catch (error) {
      console.error('Batch sync failed:', error);
      setSyncStatus('error');
    }
  }, [projects, offlineMode, config.apiEndpoint, syncProjectToLMS]);

  // Load projects from LMS
  const loadProjectsFromLMS = useCallback(async () => {
    if (!config.apiEndpoint || !config.userId) {
      return [];
    }

    try {
      const response = await fetch(
        `${config.apiEndpoint}/projects?userId=${config.userId}&courseId=${config.courseId}`,
        {
          headers: {
            ...(config.authToken && { 'Authorization': `Bearer ${config.authToken}` })
          }
        }
      );

      if (!response.ok) {
        throw new Error(`Failed to load projects from LMS: ${response.status}`);
      }

      const lmsProjects = await response.json();
      
      // Convert LMS format to PictoBloxProject format
      const projects = lmsProjects.map((lmsProject) => ({
        id: lmsProject.id,
        name: lmsProject.name,
        description: lmsProject.description,
        projectData: lmsProject.projectData,
        sessionId: lmsProject.sessionId,
        ageGroup: lmsProject.ageGroup,
        createdAt: lmsProject.createdAt,
        updatedAt: lmsProject.updatedAt
      }));

      // Save to local storage
      projects.forEach(project => {
        PictoBloxStorage.saveProject(project);
      });

      return projects;
    } catch (error) {
      console.error('Failed to load projects from LMS:', error);
      return [];
    }
  }, [config]);

  // Delete project
  const deleteProject = useCallback(async (projectId) => {
    try {
      // Remove from local storage
      const localProjects = PictoBloxStorage.getUserProjects();
      const filteredProjects = localProjects.filter(p => p.id !== projectId);
      localStorage.setItem('digifunzi_pictoblox_projects', JSON.stringify(filteredProjects));

      // Remove from state
      setProjects(prev => prev.filter(p => p.id !== projectId));

      // Try to delete from LMS if configured
      if (!offlineMode && config.apiEndpoint) {
        try {
          await fetch(`${config.apiEndpoint}/projects/${projectId}`, {
            method: 'DELETE',
            headers: {
              ...(config.authToken && { 'Authorization': `Bearer ${config.authToken}` })
            }
          });
        } catch (error) {
          console.warn('Failed to delete project from LMS:', error);
        }
      }

      return true;
    } catch (error) {
      console.error('Failed to delete project:', error);
      return false;
    }
  }, [config, offlineMode]);

  // Get sync statistics
  const getSyncStats = useCallback(() => {
    const total = projects.length;
    const synced = projects.filter(p => p.syncStatus === 'synced').length;
    const local = projects.filter(p => p.syncStatus === 'local').length;
    const errors = projects.filter(p => p.syncStatus === 'error').length;

    return {
      total,
      synced,
      local,
      errors,
      syncPercentage: total > 0 ? Math.round((synced / total) * 100) : 100
    };
  }, [projects]);

  return {
    projects,
    syncStatus,
    lastSyncTime,
    offlineMode,
    
    // Actions
    saveProject,
    syncProjectsToLMS,
    loadProjectsFromLMS,
    deleteProject,
    loadProjectsFromStorage,
    
    // Utils
    getSyncStats
  };
}