import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from "./ui/dialog";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Card, CardContent } from "./ui/card";
import { Star, MessageCircle, Phone, Mail, Clock } from "lucide-react";
import { ImageWithFallback } from "./figma/ImageWithFallback";

interface TechnicalMentor {
  id: number;
  name: string;
  expertise: string[];
  rating: number;
  availability: 'available' | 'busy' | 'offline';
  hourlyRate: number;
  totalSessions: number;
  image: string;
  email: string;
  phone: string;
  responseTime: string;
}

interface SupportModalProps {
  isOpen: boolean;
  onClose: () => void;
}

export function SupportModal({ isOpen, onClose }: SupportModalProps) {
  const mentors: TechnicalMentor[] = [
    {
      id: 1,
      name: "Sarah Johnson",
      expertise: ["Scratch Programming", "Animation", "Game Development"],
      rating: 4.9,
      availability: "available",
      hourlyRate: 25,
      totalSessions: 150,
      image: "https://images.unsplash.com/photo-1661675639921-c425af954585?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxwcm9mZXNzaW9uYWwlMjB0ZWFjaGVyJTIwbWVudG9yfGVufDF8fHx8MTc1NjcyNzU0OXww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      email: "sarah.johnson@digifunzi.com",
      phone: "+254 123 456 789",
      responseTime: "Usually responds within 1 hour"
    },
    {
      id: 2,
      name: "David Kim",
      expertise: ["Advanced Coding", "AI & ML", "Robotics"],
      rating: 4.8,
      availability: "busy",
      hourlyRate: 30,
      totalSessions: 120,
      image: "https://images.unsplash.com/photo-1543282949-ffbf6a0f263c?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxjb2RpbmclMjBpbnN0cnVjdG9yJTIwbWVudG9yfGVufDF8fHx8MTc1NjcyNzU1M3ww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      email: "david.kim@digifunzi.com",
      phone: "+254 123 456 790",
      responseTime: "Usually responds within 2 hours"
    },
    {
      id: 3,
      name: "Maria Garcia",
      expertise: ["Creative Coding", "Digital Art", "Interactive Design"],
      rating: 4.7,
      availability: "available",
      hourlyRate: 28,
      totalSessions: 98,
      image: "https://images.unsplash.com/photo-1661675639921-c425af954585?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxwcm9mZXNzaW9uYWwlMjB0ZWFjaGVyJTIwbWVudG9yfGVufDF8fHx8MTc1NjcyNzU0OXww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      email: "maria.garcia@digifunzi.com",
      phone: "+254 123 456 791",
      responseTime: "Usually responds within 30 minutes"
    }
  ];

  const getAvailabilityBadge = (availability: string) => {
    switch (availability) {
      case 'available':
        return <Badge className="bg-green-100 text-green-700">Available</Badge>;
      case 'busy':
        return <Badge className="bg-yellow-100 text-yellow-700">Busy</Badge>;
      case 'offline':
        return <Badge className="bg-gray-100 text-gray-700">Offline</Badge>;
      default:
        return null;
    }
  };

  const renderStars = (rating: number) => {
    return Array.from({ length: 5 }, (_, i) => (
      <Star
        key={i}
        className={`w-4 h-4 ${i < Math.floor(rating) ? 'fill-yellow-400 text-yellow-400' : 'text-gray-300'}`}
      />
    ));
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-4xl max-h-[80vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="text-xl">Get Technical Support</DialogTitle>
          <DialogDescription>
            Connect with our expert technical mentors for personalized help
          </DialogDescription>
        </DialogHeader>

        <div className="grid gap-6 mt-4">
          {mentors.map((mentor) => (
            <Card key={mentor.id} className="overflow-hidden">
              <CardContent className="p-6">
                <div className="flex items-start gap-4">
                  <div className="flex-shrink-0">
                    <div className="w-16 h-16 rounded-full overflow-hidden">
                      <ImageWithFallback
                        src={mentor.image}
                        alt={mentor.name}
                        className="w-full h-full object-cover"
                      />
                    </div>
                  </div>

                  <div className="flex-grow">
                    <div className="flex items-start justify-between mb-2">
                      <div>
                        <h3 className="font-medium text-lg">{mentor.name}</h3>
                        <div className="flex items-center gap-2 mt-1">
                          <div className="flex items-center gap-1">
                            {renderStars(mentor.rating)}
                            <span className="text-sm text-muted-foreground ml-1">
                              {mentor.rating} ({mentor.totalSessions} sessions)
                            </span>
                          </div>
                        </div>
                      </div>
                      <div className="text-right">
                        {getAvailabilityBadge(mentor.availability)}
                        <div className="text-sm font-medium mt-1">
                          KES {mentor.hourlyRate}/hour
                        </div>
                      </div>
                    </div>

                    <div className="space-y-3">
                      <div>
                        <h4 className="text-sm font-medium mb-1">Expertise</h4>
                        <div className="flex flex-wrap gap-1">
                          {mentor.expertise.map((skill, index) => (
                            <Badge key={index} variant="secondary" className="text-xs">
                              {skill}
                            </Badge>
                          ))}
                        </div>
                      </div>

                      <div className="space-y-2 text-sm">
                        <div className="flex items-center gap-2">
                          <Mail className="w-4 h-4 text-muted-foreground flex-shrink-0" />
                          <span className="text-muted-foreground break-all">{mentor.email}</span>
                        </div>
                        <div className="flex items-center gap-2">
                          <Phone className="w-4 h-4 text-muted-foreground flex-shrink-0" />
                          <span className="text-muted-foreground">{mentor.phone}</span>
                        </div>
                        <div className="flex items-center gap-2">
                          <Clock className="w-4 h-4 text-muted-foreground flex-shrink-0" />
                          <span className="text-muted-foreground">{mentor.responseTime}</span>
                        </div>
                      </div>

                      <div className="flex gap-2 pt-2">
                        <Button 
                          size="sm" 
                          className="bg-blue-600 hover:bg-blue-700"
                          disabled={mentor.availability === 'offline'}
                        >
                          <MessageCircle className="w-4 h-4 mr-1" />
                          Start Chat
                        </Button>
                        <Button 
                          size="sm" 
                          variant="outline"
                          disabled={mentor.availability !== 'available'}
                        >
                          <Phone className="w-4 h-4 mr-1" />
                          Schedule Call
                        </Button>
                      </div>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        <div className="mt-6 p-4 bg-blue-50 rounded-lg">
          <h4 className="font-medium mb-2">Need immediate help?</h4>
          <p className="text-sm text-muted-foreground mb-3">
            For urgent technical issues, you can also reach out to our general support team.
          </p>
          <div className="flex gap-2">
            <Button size="sm" variant="outline">
              <Mail className="w-4 h-4 mr-1" />
              support@digifunzi.com
            </Button>
            <Button size="sm" variant="outline">
              <Phone className="w-4 h-4 mr-1" />
              +254 100 000 000
            </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}