import { Card, CardContent } from "./ui/card";
import { Badge } from "./ui/badge";
import { Button } from "./ui/button";
import { PlayCircle, Lock, CheckCircle, Clock } from "lucide-react";

export function SessionsList({ sessions, onSessionClick, ageGroup = 'quest' }) {
  const getStatusIcon = (status, isActive) => {
    if (isActive) return <PlayCircle className="w-5 h-5 text-blue-600" />;
    if (status === 'completed') return <CheckCircle className="w-5 h-5 text-green-600" />;
    if (status === 'locked') return <Lock className="w-5 h-5 text-gray-400" />;
    return <PlayCircle className="w-5 h-5 text-gray-600" />;
  };

  const getStatusBadge = (status, isActive) => {
    if (isActive) return <Badge className="bg-blue-100 text-blue-700">Now Playing</Badge>;
    if (status === 'completed') return <Badge className="bg-green-100 text-green-700">Completed</Badge>;
    if (status === 'locked') return <Badge className="bg-gray-100 text-gray-700">Locked</Badge>;
    return null;
  };

  const getMentorAvailabilityColor = (availability) => {
    switch (availability) {
      case 'high':
        return 'text-green-600';
      case 'medium':
        return 'text-yellow-600';
      case 'low':
        return 'text-red-600';
      default:
        return 'text-gray-600';
    }
  };

  return (
    <div className="max-w-7xl mx-auto px-4 py-8">
      <div className="space-y-6">
        <h2 className="text-2xl">Course Sessions</h2>
        
        <div className="max-h-[600px] overflow-y-auto space-y-3 pr-2 scrollbar-thin scrollbar-thumb-gray-300 scrollbar-track-gray-100">
          {sessions.map((session) => (
            <Card 
              key={session.id}
              className={`cursor-pointer transition-all hover:shadow-md ${
                session.isActive ? 'ring-2 ring-blue-500 bg-blue-50' : ''
              } ${session.status === 'locked' ? 'opacity-60' : ''}`}
              onClick={() => session.status !== 'locked' && onSessionClick(session.id)}
            >
              <CardContent className="p-4">
                <div className="flex items-center gap-4">
                  <div className="flex-shrink-0">
                    <div className={`w-10 h-10 rounded-full flex items-center justify-center ${
                      session.isActive ? 'bg-blue-100' : 'bg-gray-100'
                    }`}>
                      <span className="text-sm font-medium">
                        {session.id}
                      </span>
                    </div>
                  </div>
                  
                  <div className="flex-grow">
                    <div className="flex items-center gap-2 mb-1">
                      <h3 className="font-medium">{session.title}</h3>
                      {getStatusBadge(session.status, session.isActive)}
                    </div>
                    <p className="text-sm text-muted-foreground mb-2">{session.description}</p>
                    <div className="flex items-center gap-4 text-xs text-muted-foreground">
                      <div className="flex items-center gap-1">
                        <Clock className="w-3 h-3" />
                        <span>{session.duration}</span>
                      </div>
                      <div>
                        <span className="font-medium">KES {session.price.toLocaleString()}</span>
                      </div>
                      <div className={`font-medium ${getMentorAvailabilityColor(session.mentorAvailability)}`}>
                        {session.mentorAvailability} mentor availability
                      </div>
                    </div>
                  </div>
                  
                  <div className="flex-shrink-0">
                    {getStatusIcon(session.status, session.isActive)}
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    </div>
  );
}