import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { Progress } from "./ui/progress";
import { Play, Clock, CheckSquare, BookOpen, Users, ChevronLeft, ChevronRight, Code2 } from "lucide-react";
import { ImageWithFallback } from "./figma/ImageWithFallback";
import { AssignmentCard } from "./AssignmentCard";
import { ResourcesCard } from "./ResourcesCard";
import { SupportModal } from "./SupportModal";
import { PictoBloxEmbed } from "./PictoBloxEmbed";

export function SessionDetail({
  sessionTitle,
  sessionNumber,
  totalSessions,
  progress,
  sessionTime,
  videoUrl,
  onPrevious,
  onNext,
  onHome,
  isCompleted,
  onSessionComplete,
  onVideoProgress,
  videoTracking,
  ageGroup = 'quest'
}) {
  const [currentVideo, setCurrentVideo] = useState("introduction");
  const [currentVideoUrl, setCurrentVideoUrl] = useState(videoUrl);
  const [currentVideoTitle, setCurrentVideoTitle] = useState("Introduction");
  const [assignmentStatus, setAssignmentStatus] = useState(
    sessionNumber === 1 ? 'issued' : 'not-issued'
  );
  const [isSupportModalOpen, setIsSupportModalOpen] = useState(false);
  const [sessionProgress, setSessionProgress] = useState(0);
  const [completedVideos, setCompletedVideos] = useState(new Set());
  const [isPictoBloxMinimized, setIsPictoBloxMinimized] = useState(true);
  const [showPictoBlox, setShowPictoBlox] = useState(false);

  const sessionVideos = [
    { 
      id: "introduction", 
      title: "Introduction", 
      duration: "5:30", 
      url: videoUrl,
      status: currentVideo === "introduction" ? "now-playing" : completedVideos.has("introduction") ? "completed" : "upcoming"
    },
    { 
      id: "main-concepts", 
      title: "Main Concepts", 
      duration: "4:15", 
      url: "https://app.digifunzi.com/videos/Lesson%201%2C%20Main%20Concepts-%20What%20is%20coding_.mp4",
      status: currentVideo === "main-concepts" ? "now-playing" : completedVideos.has("main-concepts") ? "completed" : "upcoming"
    },
    { 
      id: "activity-1", 
      title: "Activity 1", 
      duration: "3:45", 
      url: "https://app.digifunzi.com/videos/Lesson%201%2C%20Activity%201.mp4",
      status: currentVideo === "activity-1" ? "now-playing" : completedVideos.has("activity-1") ? "completed" : "upcoming"
    },
    { 
      id: "assignment", 
      title: "Assignment", 
      duration: "2:30", 
      url: "https://app.digifunzi.com/videos/Lesson%201%2C%20Assignment.mp4",
      status: currentVideo === "assignment" ? "now-playing" : completedVideos.has("assignment") ? "completed" : "upcoming"
    },
    { 
      id: "course-demo", 
      title: "Course Demo", 
      duration: "6:20", 
      url: "https://app.digifunzi.com/videos/Activity%201%20-%20Walking%20Animation.mp4",
      status: currentVideo === "course-demo" ? "now-playing" : completedVideos.has("course-demo") ? "completed" : "upcoming"
    }
  ];

  const handleVideoSelect = (video) => {
    setCurrentVideo(video.id);
    setCurrentVideoUrl(video.url);
    setCurrentVideoTitle(video.title);
    
    // Issue assignment after watching assignment video
    if (video.id === "assignment" && assignmentStatus === 'not-issued') {
      setTimeout(() => {
        setAssignmentStatus('issued');
      }, 1000);
    }
  };

  const handleVideoComplete = (videoId) => {
    const newCompleted = new Set(completedVideos);
    newCompleted.add(videoId);
    setCompletedVideos(newCompleted);
    
    // Update session progress
    const progressPercentage = (newCompleted.size / sessionVideos.length) * 100;
    setSessionProgress(progressPercentage);
    
    // Track video progress with the video tracking system
    if (onVideoProgress) {
      onVideoProgress(sessionNumber, videoId, 100);
    }
    
    // Mark session as completed if all videos are watched and assignment is graded
    if (newCompleted.size === sessionVideos.length && assignmentStatus === 'graded') {
      onSessionComplete();
    }
  };

  const handleUploadAssignment = () => {
    // Simulate assignment upload
    setAssignmentStatus('submitted');
    
    // Simulate grading after 3 seconds
    setTimeout(() => {
      setAssignmentStatus('graded');
    }, 3000);
  };

  const checklist = [
    { item: "Switch Quarky on", completed: false },
    { item: "Connect Quarky to PictoBlox using Bluetooth or Serial port", completed: false },
    { item: "Select AI model to use", completed: false },
    { item: "Write your code", completed: false },
    { item: "Test Functionality", completed: false }
  ];

  // Get session-specific preloaded projects
  const getPreloadedProjects = () => {
    const sessionProjects = [
      {
        id: `session-${sessionNumber}-starter`,
        name: `Session ${sessionNumber} Starter`,
        description: `Pre-configured project for ${sessionTitle}`,
        projectData: `session_${sessionNumber}_starter_project`
      }
    ];

    // Add specific projects based on session content
    if (sessionTitle.toLowerCase().includes('animation')) {
      sessionProjects.push({
        id: 'walking-template',
        name: 'Walking Animation Template',
        description: 'Template with sprite and basic walking setup',
        projectData: 'walking_animation_template'
      });
    }

    if (sessionTitle.toLowerCase().includes('ai')) {
      sessionProjects.push({
        id: 'ai-template',
        name: 'AI Recognition Template',
        description: 'Template with AI blocks pre-configured',
        projectData: 'ai_recognition_template'
      });
    }

    return sessionProjects;
  };

  // Age-specific styling
  const getAgeStyles = () => {
    switch (ageGroup) {
      case 'crib':
        return {
          bgColor: 'bg-gradient-to-br from-pink-50 to-purple-50',
          headerBg: 'bg-gradient-to-r from-pink-400 to-purple-500',
          progressColors: 'from-pink-400 to-purple-500',
          buttonStyle: 'bg-gradient-to-r from-pink-400 to-purple-500 hover:from-pink-500 hover:to-purple-600',
          homeButtonText: '🏠 Back to Fun!',
          nextButtonText: '🎉 Next Adventure!',
          previousButtonText: '← Back',
          sessionText: 'Adventure',
          cardStyle: 'rounded-2xl shadow-lg',
          playingText: '🎬 Playing!',
          completedText: '✅ Done!'
        };
      case 'clicker':
        return {
          bgColor: 'bg-gradient-to-br from-blue-50 to-green-50',
          headerBg: 'bg-gradient-to-r from-blue-500 to-green-500',
          progressColors: 'from-blue-400 to-green-500',
          buttonStyle: 'bg-gradient-to-r from-blue-500 to-green-500 hover:from-blue-600 hover:to-green-600',
          homeButtonText: '🎮 Course Home',
          nextButtonText: '🚀 Next Challenge!',
          previousButtonText: '← Previous',
          sessionText: 'Challenge',
          cardStyle: 'rounded-xl shadow-md',
          playingText: '▶️ Playing',
          completedText: '🎯 Complete'
        };
      default:
        return {
          bgColor: 'bg-gray-50',
          headerBg: 'bg-white',
          progressColors: 'from-indigo-500 to-purple-600',
          buttonStyle: 'bg-blue-600 hover:bg-blue-700',
          homeButtonText: 'Course Home',
          nextButtonText: 'Next Session',
          previousButtonText: 'Previous',
          sessionText: 'Session',
          cardStyle: 'rounded-lg',
          playingText: 'Playing',
          completedText: 'Done'
        };
    }
  };

  const styles = getAgeStyles();

  return (
    <div className={`min-h-screen ${styles.bgColor}`}>
      {/* Header */}
      <div className={`${styles.headerBg} ${ageGroup !== 'quest' ? 'text-white' : 'bg-white border-b'}`}>
        <div className="max-w-7xl mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-4">
              <Button 
                variant={ageGroup === 'quest' ? 'outline' : 'secondary'} 
                size="sm" 
                onClick={onHome}
                className={ageGroup !== 'quest' ? 'bg-white/20 text-white border-white/30 hover:bg-white/30' : ''}
              >
                <ChevronLeft className="w-4 h-4 mr-1" />
                {styles.homeButtonText}
              </Button>
              <div>
                <h1 className={`text-xl ${ageGroup === 'crib' ? 'font-bold' : ''}`}>
                  {ageGroup === 'crib' ? '🌟 Coding Adventures!' : 
                   ageGroup === 'clicker' ? '🎮 Coding Academy' : 
                   'Coding and Animations Course'}
                </h1>
                <p className={`text-sm ${ageGroup !== 'quest' ? 'text-white/80' : 'text-muted-foreground'}`}>
                  {styles.sessionText} {sessionNumber} of {totalSessions}
                </p>
              </div>
            </div>
            <div className="flex items-center gap-4">
              <div className="text-right">
                <div className={`text-sm ${ageGroup !== 'quest' ? 'text-white' : 'text-muted-foreground'}`}>{progress}%</div>
                <div className={`text-sm ${ageGroup !== 'quest' ? 'text-white/80' : 'text-muted-foreground'}`}>Overall Progress</div>
              </div>
              <div className="text-right">
                <div className={`text-sm ${ageGroup !== 'quest' ? 'text-white' : ''}`}>{sessionTime}</div>
                <div className={`text-sm ${ageGroup !== 'quest' ? 'text-white/80' : 'text-muted-foreground'}`}>
                  {ageGroup === 'crib' ? 'Play Time' : ageGroup === 'clicker' ? 'Learning Time' : 'Session Time'}
                </div>
              </div>
              <Button 
                variant={ageGroup === 'quest' ? 'outline' : 'secondary'}
                size="sm"
                onClick={() => {
                  setShowPictoBlox(!showPictoBlox);
                  if (!showPictoBlox) {
                    setIsPictoBloxMinimized(false);
                  }
                }}
                className={ageGroup !== 'quest' ? 'bg-white/20 text-white border-white/30 hover:bg-white/30' : ''}
              >
                <Code2 className="w-4 h-4 mr-1" />
                {showPictoBlox ? 'Hide' : 'Open'} PictoBlox
              </Button>
              {ageGroup === 'quest' && (
                <Button variant="outline" size="sm">
                  Jump to Session {sessionNumber} ▼
                </Button>
              )}
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 py-6">
        <div className={`grid gap-6 ${showPictoBlox && !isPictoBloxMinimized ? 'lg:grid-cols-1' : 'lg:grid-cols-3'}`}>
          {/* Main Content */}
          <div className={`space-y-6 ${showPictoBlox && !isPictoBloxMinimized ? '' : 'lg:col-span-2'}`}>
            {/* Course Title */}
            <div>
              <h1 className="text-3xl mb-2">{sessionTitle}</h1>
              <p className="text-muted-foreground">Currently Playing: {currentVideoTitle}</p>
            </div>

            {/* Video Player */}
            <div className="bg-black rounded-lg overflow-hidden aspect-video relative">
              {currentVideoUrl.startsWith('http') ? (
                <video 
                  src={currentVideoUrl} 
                  controls 
                  className="w-full h-full"
                  onEnded={() => handleVideoComplete(currentVideo)}
                  onTimeUpdate={(e) => {
                    const video = e.target;
                    const progress = (video.currentTime / video.duration) * 100;
                    if (onVideoProgress && !isNaN(progress)) {
                      onVideoProgress(sessionNumber, currentVideo, progress);
                    }
                  }}
                >
                  Your browser does not support the video tag.
                </video>
              ) : (
                <>
                  <ImageWithFallback
                    src={currentVideoUrl}
                    alt="Course video"
                    className="w-full h-full object-cover"
                  />
                  <div className="absolute inset-0 bg-black/50 flex items-center justify-center">
                    <div className="bg-white/20 backdrop-blur-sm rounded-lg p-8">
                      <div className="text-white text-center">
                        <div className="mb-4">
                          <div className="w-16 h-16 bg-white/20 rounded-full flex items-center justify-center mx-auto mb-2">
                            <Play className="w-8 h-8 text-white" />
                          </div>
                        </div>
                        <div className="text-sm">Click to play video</div>
                      </div>
                    </div>
                  </div>
                </>
              )}
            </div>

            {/* Progress Bar */}
            <div className={`bg-white ${styles.cardStyle} p-4 ${ageGroup === 'crib' ? 'border-4 border-pink-100' : ageGroup === 'clicker' ? 'border-2 border-blue-100' : ''}`}>
              <div className="flex items-center justify-between mb-2">
                <span className={`text-sm ${ageGroup === 'crib' ? 'font-bold text-pink-600' : ageGroup === 'clicker' ? 'font-semibold text-blue-600' : ''}`}>
                  {ageGroup === 'crib' ? '🎯 My Progress' : 
                   ageGroup === 'clicker' ? '📊 Challenge Progress' : 
                   'Session Progress'}
                </span>
                <span className="text-sm">{Math.round(sessionProgress)}% • {completedVideos.size}/{sessionVideos.length} videos</span>
              </div>
              <div className="relative">
                <div className="w-full bg-gray-200 rounded-full h-3 overflow-hidden">
                  <div 
                    className={`h-full bg-gradient-to-r ${styles.progressColors} transition-all duration-500`}
                    style={{ width: `${sessionProgress}%` }}
                  >
                    {sessionProgress > 0 && (
                      <div className="absolute inset-0 bg-white/20 animate-pulse" />
                    )}
                  </div>
                </div>
                {sessionProgress > 10 && ageGroup === 'crib' && (
                  <div className="absolute right-2 top-1/2 -translate-y-1/2">
                    <span className="text-lg">⭐</span>
                  </div>
                )}
              </div>
            </div>

            {/* PictoBlox Embed */}
            {showPictoBlox && !isPictoBloxMinimized && (
              <PictoBloxEmbed
                isMinimized={isPictoBloxMinimized}
                onToggleMinimize={() => setIsPictoBloxMinimized(!isPictoBloxMinimized)}
                ageGroup={ageGroup}
                sessionNumber={sessionNumber}
                sessionTitle={sessionTitle}
                preloadedProjects={getPreloadedProjects()}
                onProjectSave={(project) => {
                  console.log('Project saved:', project);
                  // Here you could send the project to your LMS API
                }}
              />
            )}

            {/* Session Checklist */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <CheckSquare className="w-5 h-5" />
                  Session Checklist
                </CardTitle>
                <p className="text-sm text-muted-foreground">
                  Complete these steps for this session
                </p>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  {checklist.map((item, index) => (
                    <div key={index} className="flex items-center gap-3">
                      <div className={`w-4 h-4 border-2 rounded ${
                        item.completed ? 'bg-green-500 border-green-500' : 'border-gray-300'
                      }`}>
                        {item.completed && (
                          <div className="w-full h-full flex items-center justify-center">
                            <div className="w-2 h-2 bg-white rounded-full"></div>
                          </div>
                        )}
                      </div>
                      <span className={`text-sm ${
                        item.completed ? 'line-through text-muted-foreground' : ''
                      }`}>
                        {item.item}
                      </span>
                    </div>
                  ))}
                </div>
                <div className="mt-4 text-xs text-muted-foreground">
                  Check-list Progress: 0/5
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Sidebar */}
          <div className={`space-y-6 ${showPictoBlox && !isPictoBloxMinimized ? 'hidden lg:block' : ''}`}>
            {/* Navigation */}
            <div className="flex gap-2">
              <Button 
                variant={ageGroup === 'quest' ? 'outline' : 'secondary'} 
                size="sm" 
                onClick={onPrevious} 
                className={`flex-1 ${ageGroup === 'crib' ? 'rounded-2xl border-2 border-pink-200 text-pink-600 hover:bg-pink-50' : 
                  ageGroup === 'clicker' ? 'rounded-xl border-2 border-blue-200 text-blue-600 hover:bg-blue-50' : ''}`}
              >
                {styles.previousButtonText}
              </Button>
              <Button 
                size="sm" 
                onClick={onNext} 
                className={`flex-1 ${styles.buttonStyle} ${ageGroup === 'crib' ? 'rounded-2xl font-bold' : ageGroup === 'clicker' ? 'rounded-xl font-semibold' : ''}`}
                disabled={!isCompleted && completedVideos.size < sessionVideos.length}
              >
                {styles.nextButtonText}
              </Button>
            </div>
            
            {!isCompleted && completedVideos.size < sessionVideos.length && (
              <div className={`text-xs text-center p-3 rounded-lg ${
                ageGroup === 'crib' ? 'bg-yellow-100 text-yellow-700 border-2 border-yellow-200 font-medium' :
                ageGroup === 'clicker' ? 'bg-blue-100 text-blue-700 border border-blue-200' :
                'text-muted-foreground bg-yellow-50'
              }`}>
                {ageGroup === 'crib' ? '🎯 Finish all videos to unlock the next adventure!' :
                 ageGroup === 'clicker' ? '⚡ Complete all videos and assignments to unlock the next challenge!' :
                 'Complete all videos and assignment to unlock next session'}
              </div>
            )}

            {/* Tabs */}
            <Tabs defaultValue="videos" className="w-full">
              <TabsList className={`grid w-full ${showPictoBlox ? 'grid-cols-4' : 'grid-cols-3'}`}>
                <TabsTrigger value="videos">Videos</TabsTrigger>
                <TabsTrigger value="assignments">Assignments</TabsTrigger>
                <TabsTrigger value="resources">Resources</TabsTrigger>
                {showPictoBlox && <TabsTrigger value="pictoblox">PictoBlox</TabsTrigger>}
              </TabsList>
              
              <TabsContent value="videos" className="mt-4">
                <Card>
                  <CardHeader>
                    <CardTitle className="text-base">Session Videos</CardTitle>
                    <p className="text-sm text-muted-foreground">
                      Click "Play" to load a video in the main player
                    </p>
                  </CardHeader>
                  <CardContent className="space-y-3">
                    {sessionVideos.map((video, index) => {
                      const watchTime = videoTracking?.getVideoWatchTime?.(sessionNumber, video.id) || 0;
                      const watchProgress = videoTracking?.getVideoProgress?.(sessionNumber, video.id) || 0;
                      const lastPosition = videoTracking?.getVideoPosition?.(sessionNumber, video.id) || 0;
                      
                      return (
                        <div 
                          key={index} 
                          className={`flex flex-col gap-3 p-3 rounded-lg border cursor-pointer transition-colors ${
                            video.status === 'now-playing' ? 'bg-blue-50 border-blue-200' : 'hover:bg-gray-50'
                          }`}
                          onClick={() => handleVideoSelect(video)}
                        >
                          <div className="flex items-center gap-3">
                            <div className={`w-8 h-8 rounded-full flex items-center justify-center ${
                              video.status === 'now-playing' ? 'bg-blue-100' : 
                              video.status === 'completed' ? 'bg-green-100' : 'bg-gray-100'
                            }`}>
                              <Play className={`w-4 h-4 ${
                                video.status === 'now-playing' ? 'text-blue-600' : 
                                video.status === 'completed' ? 'text-green-600' : 'text-gray-600'
                              }`} />
                            </div>
                            <div className="flex-grow">
                              <div className="font-medium text-sm">{video.title}</div>
                              <div className="text-xs text-muted-foreground flex items-center gap-2">
                                <span>{video.duration}</span>
                                {video.status === 'completed' && (
                                  <span className="text-green-600">• 100% watched</span>
                                )}
                                {video.status === 'now-playing' && (
                                  <span className="text-blue-600">• In progress</span>
                                )}
                                {watchTime > 0 && (
                                  <span>• {videoTracking?.formatTime?.(watchTime) || '0m'} watched</span>
                                )}
                              </div>
                            </div>
                            <div className="flex items-center gap-2">
                              {video.status === 'now-playing' && (
                                <Badge className={`text-xs ${
                                  ageGroup === 'crib' ? 'bg-pink-100 text-pink-700 border-pink-200' :
                                  ageGroup === 'clicker' ? 'bg-blue-100 text-blue-700 border-blue-200' :
                                  'bg-blue-100 text-blue-700'
                                }`}>
                                  {styles.playingText}
                                </Badge>
                              )}
                              {video.status === 'completed' && (
                                <Badge className={`text-xs ${
                                  ageGroup === 'crib' ? 'bg-green-100 text-green-700 border-green-200' :
                                  ageGroup === 'clicker' ? 'bg-green-100 text-green-700 border-green-200' :
                                  'bg-green-100 text-green-700'
                                }`}>
                                  {styles.completedText}
                                </Badge>
                              )}
                              <Button 
                                size="sm" 
                                variant="outline"
                                className={ageGroup === 'crib' ? 'rounded-xl border-2 border-purple-200 text-purple-600 hover:bg-purple-50' :
                                  ageGroup === 'clicker' ? 'rounded-lg border-2 border-blue-200 text-blue-600 hover:bg-blue-50' : ''}
                              >
                                {video.status === 'now-playing' ? 
                                  (ageGroup === 'crib' ? '🎬 Playing' : ageGroup === 'clicker' ? '▶️ Playing' : 'Playing') : 
                                  (ageGroup === 'crib' ? '🎬 Play' : ageGroup === 'clicker' ? '▶️ Play' : 'Play')}
                              </Button>
                            </div>
                          </div>
                          
                          {watchProgress > 0 && watchProgress < 100 && (
                            <div className="mt-2">
                              <div className="w-full bg-gray-200 rounded-full h-1.5">
                                <div 
                                  className="bg-blue-600 h-1.5 rounded-full transition-all" 
                                  style={{ width: `${watchProgress}%` }}
                                ></div>
                              </div>
                              <div className="text-xs text-muted-foreground mt-1 flex justify-between">
                                <span>{Math.round(watchProgress)}% completed</span>
                                <span>Last position: {videoTracking?.formatTime?.(lastPosition) || '0:00'}</span>
                              </div>
                            </div>
                          )}
                        </div>
                      );
                    })}
                  </CardContent>
                </Card>
              </TabsContent>
              
              <TabsContent value="assignments" className="mt-4">
                <AssignmentCard 
                  assignmentStatus={assignmentStatus}
                  onUploadAssignment={handleUploadAssignment}
                />
              </TabsContent>
              
              <TabsContent value="resources" className="mt-4">
                <ResourcesCard />
              </TabsContent>

              {showPictoBlox && (
                <TabsContent value="pictoblox" className="mt-4">
                  <Card>
                    <CardHeader>
                      <CardTitle className="text-base flex items-center gap-2">
                        <Code2 className="w-5 h-5" />
                        PictoBlox Controls
                      </CardTitle>
                      <p className="text-sm text-muted-foreground">
                        Manage your PictoBlox workspace
                      </p>
                    </CardHeader>
                    <CardContent className="space-y-4">
                      <div className="flex flex-col gap-2">
                        <Button 
                          size="sm" 
                          onClick={() => setIsPictoBloxMinimized(!isPictoBloxMinimized)}
                          className={`${styles.buttonStyle} ${ageGroup === 'crib' ? 'rounded-2xl font-bold' : ageGroup === 'clicker' ? 'rounded-xl font-semibold' : ''}`}
                        >
                          {isPictoBloxMinimized ? 'Maximize PictoBlox' : 'Minimize PictoBlox'}
                        </Button>
                        <Button 
                          size="sm" 
                          variant="outline"
                          onClick={() => setShowPictoBlox(false)}
                          className={ageGroup === 'crib' ? 'rounded-2xl border-2 border-gray-200' : ageGroup === 'clicker' ? 'rounded-xl border-2 border-gray-200' : ''}
                        >
                          Close PictoBlox
                        </Button>
                      </div>
                      <div className="text-xs text-muted-foreground">
                        <p>🔧 Use the controls above to manage your PictoBlox coding environment.</p>
                        <p className="mt-1">💾 Your projects are automatically saved as you work.</p>
                      </div>
                    </CardContent>
                  </Card>
                </TabsContent>
              )}
            </Tabs>

            {/* Get Help */}
            <Card>
              <CardContent className="pt-6 text-center">
                <Users className="w-8 h-8 mx-auto mb-2 text-muted-foreground" />
                <h3 className="font-medium mb-2">Get Help</h3>
                <p className="text-sm text-muted-foreground mb-4">
                  Need assistance with this lesson?
                </p>
                <Button 
                  variant="outline" 
                  size="sm" 
                  className="w-full"
                  onClick={() => setIsSupportModalOpen(true)}
                >
                  Contact Support
                </Button>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>

      <SupportModal 
        isOpen={isSupportModalOpen}
        onClose={() => setIsSupportModalOpen(false)}
      />
    </div>
  );
}