import { Card, CardContent, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { ExternalLink, FileText, Play, Presentation, File } from "lucide-react";

interface Resource {
  id: number;
  title: string;
  type: 'youtube' | 'pdf' | 'google-doc' | 'google-slides' | 'external-link';
  url: string;
  description?: string;
}

export function ResourcesCard() {
  const resources: Resource[] = [
    {
      id: 1,
      title: "Scratch Programming Basics",
      type: "youtube",
      url: "https://www.youtube.com/watch?v=example1",
      description: "A comprehensive guide to Scratch programming fundamentals"
    },
    {
      id: 2,
      title: "Animation Principles Guide",
      type: "pdf",
      url: "https://example.com/animation-guide.pdf",
      description: "Essential principles for creating smooth animations"
    },
    {
      id: 3,
      title: "Coding Best Practices",
      type: "google-doc",
      url: "https://docs.google.com/document/d/example",
      description: "Best practices for writing clean, maintainable code"
    },
    {
      id: 4,
      title: "Session 1 Slides",
      type: "google-slides",
      url: "https://docs.google.com/presentation/d/example",
      description: "Presentation slides from today's session"
    },
    {
      id: 5,
      title: "Scratch Official Documentation",
      type: "external-link",
      url: "https://scratch.mit.edu/help/",
      description: "Official Scratch programming documentation"
    }
  ];

  const getResourceIcon = (type: string) => {
    switch (type) {
      case 'youtube':
        return <Play className="w-5 h-5 text-red-500" />;
      case 'pdf':
        return <FileText className="w-5 h-5 text-red-600" />;
      case 'google-doc':
        return <FileText className="w-5 h-5 text-blue-600" />;
      case 'google-slides':
        return <Presentation className="w-5 h-5 text-orange-500" />;
      case 'external-link':
        return <ExternalLink className="w-5 h-5 text-gray-600" />;
      default:
        return <File className="w-5 h-5 text-gray-500" />;
    }
  };

  const getResourceTypeLabel = (type: string) => {
    switch (type) {
      case 'youtube':
        return 'YouTube Video';
      case 'pdf':
        return 'PDF Document';
      case 'google-doc':
        return 'Google Doc';
      case 'google-slides':
        return 'Google Slides';
      case 'external-link':
        return 'External Link';
      default:
        return 'File';
    }
  };

  if (resources.length === 0) {
    return (
      <Card>
        <CardContent className="pt-6 text-center">
          <div className="text-muted-foreground">
            <File className="w-8 h-8 mx-auto mb-2" />
            <p className="text-sm">No resources available</p>
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="text-base">Session Resources</CardTitle>
        <p className="text-sm text-muted-foreground">
          Additional materials to help you learn
        </p>
      </CardHeader>
      <CardContent className="space-y-3">
        {resources.map((resource) => (
          <div key={resource.id} className="flex items-start gap-3 p-3 rounded-lg border hover:bg-gray-50 transition-colors">
            <div className="flex-shrink-0 mt-0.5">
              {getResourceIcon(resource.type)}
            </div>
            <div className="flex-grow min-w-0">
              <div className="flex items-start justify-between gap-2">
                <div className="min-w-0 flex-grow">
                  <h4 className="font-medium text-sm truncate">{resource.title}</h4>
                  <p className="text-xs text-muted-foreground">{getResourceTypeLabel(resource.type)}</p>
                  {resource.description && (
                    <p className="text-xs text-muted-foreground mt-1 line-clamp-2">
                      {resource.description}
                    </p>
                  )}
                </div>
                <Button 
                  size="sm" 
                  variant="outline" 
                  className="flex-shrink-0"
                  onClick={() => window.open(resource.url, '_blank')}
                >
                  <ExternalLink className="w-3 h-3 mr-1" />
                  Open
                </Button>
              </div>
            </div>
          </div>
        ))}
      </CardContent>
    </Card>
  );
}