import { Card, CardContent, CardHeader, CardTitle } from "./ui/card";
import { Progress } from "./ui/progress";
import { Badge } from "./ui/badge";
import { Clock, CheckCircle, Star, Trophy, Target, Zap } from "lucide-react";

interface ProgressSectionProps {
  overallProgress: number;
  timeSpent: string;
  sessionsCompleted: number;
  totalSessions: number;
  ageGroup?: 'crib' | 'clicker' | 'quest';
}

export function ProgressSection({
  overallProgress,
  timeSpent,
  sessionsCompleted,
  totalSessions,
  ageGroup = 'quest'
}: ProgressSectionProps) {
  // Age-specific styling
  const getAgeStyles = () => {
    switch (ageGroup) {
      case 'crib':
        return {
          cardBg: 'crib-stat-box',
          titleColor: 'crib-text-primary',
          progressColors: 'crib-text-primary',
          badgeColors: {
            ready: 'crib-badge',
            started: 'bg-purple-100 text-purple-700 border-purple-200',
            progress: 'bg-yellow-100 text-yellow-700 border-yellow-200',
            complete: 'bg-green-100 text-green-700 border-green-200'
          },
          iconColors: 'text-pink-500',
          statIcons: [Star, Trophy, Target]
        };
      case 'clicker':
        return {
          cardBg: 'bg-gradient-to-br from-blue-50 to-green-50',
          titleColor: 'text-blue-600',
          progressColors: 'from-blue-400 to-green-500',
          badgeColors: {
            ready: 'bg-blue-100 text-blue-700 border-blue-200',
            started: 'bg-cyan-100 text-cyan-700 border-cyan-200',
            progress: 'bg-orange-100 text-orange-700 border-orange-200',
            complete: 'bg-green-100 text-green-700 border-green-200'
          },
          iconColors: 'text-blue-500',
          statIcons: [Zap, Target, Trophy]
        };
      default:
        return {
          cardBg: 'bg-gradient-to-br from-indigo-50 to-purple-50',
          titleColor: 'text-indigo-600',
          progressColors: 'from-indigo-500 to-purple-600',
          badgeColors: {
            ready: 'bg-gray-100 text-gray-700 border-gray-200',
            started: 'bg-indigo-100 text-indigo-700 border-indigo-200',
            progress: 'bg-orange-100 text-orange-700 border-orange-200',
            complete: 'bg-green-100 text-green-700 border-green-200'
          },
          iconColors: 'text-indigo-500',
          statIcons: [Clock, CheckCircle, Trophy]
        };
    }
  };

  const styles = getAgeStyles();
  const [TimeIcon, CompletionIcon, MentorIcon] = styles.statIcons;

  const getBadgeStyle = () => {
    if (overallProgress === 0) return styles.badgeColors.ready;
    if (overallProgress < 50) return styles.badgeColors.started;
    if (overallProgress < 100) return styles.badgeColors.progress;
    return styles.badgeColors.complete;
  };

  const getBadgeText = () => {
    if (ageGroup === 'crib') {
      return overallProgress === 0 ? 'Let\'s Start!' :
             overallProgress < 50 ? 'Great Start!' :
             overallProgress < 100 ? 'You\'re Amazing!' :
             'Superstar!';
    } else if (ageGroup === 'clicker') {
      return overallProgress === 0 ? 'Ready to Code!' :
             overallProgress < 50 ? 'Getting Started!' :
             overallProgress < 100 ? 'Making Progress!' :
             'Code Master!';
    } else {
      return overallProgress === 0 ? 'Ready to Start' :
             overallProgress < 50 ? 'Just Getting Started' :
             overallProgress < 100 ? 'Making Progress' :
             'Course Complete';
    }
  };

  const getProgressText = () => {
    if (ageGroup === 'crib') {
      return 'My Learning Adventure';
    } else if (ageGroup === 'clicker') {
      return 'Coding Journey';
    }
    return 'Overall Progress';
  };

  return (
    <div className="max-w-7xl mx-auto px-4 py-8">
      <Card className={`${styles.cardBg} border-0 shadow-lg`}>
        <CardHeader className="pb-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-3">
              <div className={`w-10 h-10 rounded-full bg-gradient-to-r ${styles.progressColors} flex items-center justify-center`}>
                <Trophy className="w-5 h-5 text-white" />
              </div>
              <CardTitle className={`text-xl ${styles.titleColor}`}>
                {ageGroup === 'crib' ? 'My Progress' : ageGroup === 'clicker' ? 'Learning Dashboard' : 'Progress Overview'}
              </CardTitle>
            </div>
            <Badge variant="outline" className={`${getBadgeStyle()} font-medium`}>
              {getBadgeText()}
            </Badge>
          </div>
        </CardHeader>
        
        <CardContent className="space-y-6">
          {/* Enhanced Progress Bar */}
          <div className="space-y-3">
            <div className="flex items-center justify-between">
              <span className={`${styles.titleColor} font-medium`}>{getProgressText()}</span>
              <div className="flex items-center gap-2">
                <span className={`font-bold ${styles.titleColor}`}>{overallProgress}%</span>
                {overallProgress > 0 && (
                  <div className={`px-2 py-1 rounded-full bg-gradient-to-r ${styles.progressColors} text-white text-xs`}>
                    {Math.ceil(overallProgress / 10)} ⭐
                  </div>
                )}
              </div>
            </div>
            <div className="relative">
              <div className="w-full bg-gray-200 rounded-full h-4 overflow-hidden">
                <div 
                  className={`h-full bg-gradient-to-r ${styles.progressColors} transition-all duration-500 ease-out relative`}
                  style={{ width: `${overallProgress}%` }}
                >
                  {overallProgress > 0 && (
                    <div className="absolute inset-0 bg-white/20 animate-pulse" />
                  )}
                </div>
              </div>
              {overallProgress > 10 && (
                <div className="absolute right-2 top-1/2 -translate-y-1/2">
                  <Star className="w-3 h-3 text-white fill-white" />
                </div>
              )}
            </div>
          </div>

          {/* Enhanced Stats Cards */}
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            {/* Time Spent */}
            <div className="bg-white/70 backdrop-blur-sm rounded-lg p-4 border border-white/50">
              <div className="flex items-center gap-3">
                <div className={`w-10 h-10 rounded-full bg-gradient-to-r ${styles.progressColors} flex items-center justify-center`}>
                  <TimeIcon className="w-5 h-5 text-white" />
                </div>
                <div>
                  <div className="text-sm text-gray-600">
                    {ageGroup === 'crib' ? 'Play Time' : ageGroup === 'clicker' ? 'Learning Time' : 'Time Spent'}
                  </div>
                  <div className={`font-bold ${styles.titleColor}`}>{timeSpent}</div>
                </div>
              </div>
            </div>

            {/* Sessions Completed */}
            <div className="bg-white/70 backdrop-blur-sm rounded-lg p-4 border border-white/50">
              <div className="flex items-center gap-3">
                <div className={`w-10 h-10 rounded-full bg-gradient-to-r ${styles.progressColors} flex items-center justify-center`}>
                  <CompletionIcon className="w-5 h-5 text-white" />
                </div>
                <div>
                  <div className="text-sm text-gray-600">
                    {ageGroup === 'crib' ? 'Adventures' : ageGroup === 'clicker' ? 'Challenges' : 'Sessions'}
                  </div>
                  <div className={`font-bold ${styles.titleColor}`}>
                    {sessionsCompleted} / {totalSessions}
                  </div>
                </div>
              </div>
            </div>

            {/* Mentor Time / Achievement */}
            <div className="bg-white/70 backdrop-blur-sm rounded-lg p-4 border border-white/50">
              <div className="flex items-center gap-3">
                <div className={`w-10 h-10 rounded-full bg-gradient-to-r ${styles.progressColors} flex items-center justify-center`}>
                  <MentorIcon className="w-5 h-5 text-white" />
                </div>
                <div>
                  <div className="text-sm text-gray-600">
                    {ageGroup === 'crib' ? 'Help Time' : ageGroup === 'clicker' ? 'Mentor Help' : 'Mentor Time'}
                  </div>
                  <div className={`font-bold ${styles.titleColor}`}>
                    {ageGroup === 'crib' ? `${sessionsCompleted * 15}m` : `${sessionsCompleted * 30}m`}
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Achievement Badges for Crib */}
          {ageGroup === 'crib' && sessionsCompleted > 0 && (
            <div className="bg-white/70 backdrop-blur-sm rounded-lg p-4 border border-white/50">
              <div className="flex items-center gap-2 mb-3">
                <Star className="w-5 h-5 text-yellow-500" />
                <span className="font-medium text-pink-600">My Badges</span>
              </div>
              <div className="flex flex-wrap gap-2">
                {sessionsCompleted >= 1 && <Badge className="bg-yellow-100 text-yellow-700 border-yellow-200">🎉 First Step</Badge>}
                {sessionsCompleted >= 3 && <Badge className="bg-blue-100 text-blue-700 border-blue-200">🚀 Explorer</Badge>}
                {sessionsCompleted >= 5 && <Badge className="bg-purple-100 text-purple-700 border-purple-200">⭐ Star Coder</Badge>}
                {overallProgress >= 100 && <Badge className="bg-green-100 text-green-700 border-green-200">🏆 Champion</Badge>}
              </div>
            </div>
          )}

          {/* Level Progress for Clicker */}
          {ageGroup === 'clicker' && overallProgress > 0 && (
            <div className="bg-white/70 backdrop-blur-sm rounded-lg p-4 border border-white/50">
              <div className="flex items-center justify-between mb-2">
                <span className="font-medium text-blue-600">Coding Level</span>
                <span className="text-sm text-gray-600">
                  Level {Math.floor(overallProgress / 25) + 1} / 4
                </span>
              </div>
              <div className="flex gap-1">
                {[1, 2, 3, 4].map((level) => (
                  <div
                    key={level}
                    className={`flex-1 h-2 rounded ${
                      Math.floor(overallProgress / 25) + 1 >= level
                        ? `bg-gradient-to-r ${styles.progressColors}`
                        : 'bg-gray-200'
                    }`}
                  />
                ))}
              </div>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}