import { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Badge } from './ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from './ui/tabs';
import { 
  CheckCircle2, 
  XCircle, 
  AlertCircle, 
  Smartphone, 
  Tablet, 
  Monitor,
  Wifi,
  WifiOff,
  PlayCircle,
  Download,
  Upload
} from 'lucide-react';
import { PictoBloxEmbed } from './PictoBloxEmbed';
import { usePictoBloxIntegration } from '../hooks/usePictoBloxIntegration';
import { 
  getDeviceType, 
  supportsFullscreen, 
  PictoBloxStorage,
  generateSessionStarters 
} from '../utils/pictoBloxUtils';

export function PictoBloxTestSuite() {
  const [testResults, setTestResults] = useState([]);
  const [isRunningTests, setIsRunningTests] = useState(false);
  const [selectedAgeGroup, setSelectedAgeGroup] = useState('quest');
  const [showEmbed, setShowEmbed] = useState(false);
  const [isMinimized, setIsMinimized] = useState(false);

  const pictoBloxIntegration = usePictoBloxIntegration({
    apiEndpoint: 'https://api.digifunzi.com/v1', // Mock endpoint
    userId: 'test-user-123',
    courseId: 'pictoblox-test-course'
  });

  useEffect(() => {
    // Run initial compatibility tests
    runCompatibilityTests();
  }, []);

  const runCompatibilityTests = async () => {
    setIsRunningTests(true);
    const results = [];

    // Device Detection Test
    const deviceType = getDeviceType();
    results.push({
      name: 'Device Detection',
      status: 'pass',
      message: `Detected device type: ${deviceType}`,
      details: `Screen width: ${window.innerWidth}px, Height: ${window.innerHeight}px`
    });

    // Fullscreen Support Test
    const fullscreenSupported = supportsFullscreen();
    results.push({
      name: 'Fullscreen Support',
      status: fullscreenSupported ? 'pass' : 'warning',
      message: fullscreenSupported ? 'Fullscreen API supported' : 'Fullscreen API not supported',
      details: fullscreenSupported ? 'Users can use fullscreen mode' : 'Fullscreen mode will be disabled'
    });

    // Local Storage Test
    try {
      const testData = { test: 'data' };
      localStorage.setItem('digifunzi_test', JSON.stringify(testData));
      const retrieved = JSON.parse(localStorage.getItem('digifunzi_test') || '{}');
      localStorage.removeItem('digifunzi_test');
      
      results.push({
        name: 'Local Storage',
        status: retrieved.test === 'data' ? 'pass' : 'fail',
        message: retrieved.test === 'data' ? 'Local storage working' : 'Local storage failed',
        details: 'Required for saving projects offline'
      });
    } catch (error) {
      results.push({
        name: 'Local Storage',
        status: 'fail',
        message: 'Local storage error',
        details: error instanceof Error ? error.message : 'Unknown error'
      });
    }

    // Network Connectivity Test
    results.push({
      name: 'Network Status',
      status: navigator.onLine ? 'pass' : 'warning',
      message: navigator.onLine ? 'Online' : 'Offline',
      details: navigator.onLine ? 'Can sync with LMS' : 'Working in offline mode'
    });

    // iframe Support Test
    const iframe = document.createElement('iframe');
    iframe.style.display = 'none';
    iframe.src = 'about:blank';
    document.body.appendChild(iframe);
    
    setTimeout(() => {
      const iframeSupported = iframe.contentWindow !== null;
      document.body.removeChild(iframe);
      
      results.push({
        name: 'iframe Support',
        status: iframeSupported ? 'pass' : 'fail',
        message: iframeSupported ? 'iframe elements supported' : 'iframe elements blocked',
        details: iframeSupported ? 'PictoBlox can be embedded' : 'Check security settings'
      });
    }, 100);

    // WebGL Support Test (for 3D features in PictoBlox)
    try {
      const canvas = document.createElement('canvas');
      const gl = canvas.getContext('webgl') || canvas.getContext('experimental-webgl');
      results.push({
        name: 'WebGL Support',
        status: gl ? 'pass' : 'warning',
        message: gl ? 'WebGL supported' : 'WebGL not supported',
        details: gl ? '3D features available' : 'Some advanced features may not work'
      });
    } catch (error) {
      results.push({
        name: 'WebGL Support',
        status: 'warning',
        message: 'WebGL check failed',
        details: 'Unable to determine WebGL support'
      });
    }

    // Camera/Microphone Permissions Test
    if (navigator.mediaDevices && navigator.mediaDevices.getUserMedia) {
      results.push({
        name: 'Media Devices API',
        status: 'pass',
        message: 'Camera/microphone access available',
        details: 'AI features requiring camera input will work'
      });
    } else {
      results.push({
        name: 'Media Devices API',
        status: 'warning',
        message: 'Camera/microphone access limited',
        details: 'Some AI features may not work'
      });
    }

    // PostMessage API Test (for iframe communication)
    results.push({
      name: 'PostMessage API',
      status: typeof window.postMessage === 'function' ? 'pass' : 'fail',
      message: typeof window.postMessage === 'function' ? 'PostMessage supported' : 'PostMessage not supported',
      details: 'Required for PictoBlox communication'
    });

    // Touch Support Test
    const touchSupported = 'ontouchstart' in window || navigator.maxTouchPoints > 0;
    results.push({
      name: 'Touch Support',
      status: touchSupported ? 'pass' : 'pass',
      message: touchSupported ? 'Touch events supported' : 'Mouse-only interface',
      details: touchSupported ? 'Optimized for mobile/tablet' : 'Desktop interface'
    });

    setTestResults(results);
    setIsRunningTests(false);
  };

  const runProjectTests = async () => {
    const results = [...testResults];

    // Project Generation Test
    try {
      const starters = generateSessionStarters(1, 'Test Animation Session', selectedAgeGroup);
      results.push({
        name: 'Project Generation',
        status: starters.length > 0 ? 'pass' : 'fail',
        message: `Generated ${starters.length} starter projects`,
        details: `Age group: ${selectedAgeGroup}`
      });
    } catch (error) {
      results.push({
        name: 'Project Generation',
        status: 'fail',
        message: 'Failed to generate projects',
        details: error instanceof Error ? error.message : 'Unknown error'
      });
    }

    // Local Storage Project Save Test
    try {
      const testProject = {
        id: 'test-project-123',
        name: 'Test Project',
        description: 'Test project for compatibility',
        sessionId: 1,
        ageGroup: selectedAgeGroup,
        blocks: [],
        sprites: [],
        sounds: [],
        variables: {},
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString()
      };

      PictoBloxStorage.saveProject(testProject);
      const retrieved = PictoBloxStorage.getUserProjects().find(p => p.id === 'test-project-123');
      
      results.push({
        name: 'Project Storage',
        status: retrieved ? 'pass' : 'fail',
        message: retrieved ? 'Project save/load working' : 'Project storage failed',
        details: 'Local project persistence'
      });

      // Clean up test project
      if (retrieved) {
        const allProjects = PictoBloxStorage.getUserProjects();
        const filtered = allProjects.filter(p => p.id !== 'test-project-123');
        localStorage.setItem('digifunzi_pictoblox_projects', JSON.stringify(filtered));
      }
    } catch (error) {
      results.push({
        name: 'Project Storage',
        status: 'fail',
        message: 'Project storage error',
        details: error instanceof Error ? error.message : 'Unknown error'
      });
    }

    setTestResults(results);
  };

  const getStatusIcon = (status) => {
    switch (status) {
      case 'pass': return <CheckCircle2 className="w-4 h-4 text-green-500" />;
      case 'fail': return <XCircle className="w-4 h-4 text-red-500" />;
      case 'warning': return <AlertCircle className="w-4 h-4 text-yellow-500" />;
      default: return <AlertCircle className="w-4 h-4 text-gray-400" />;
    }
  };

  const getDeviceIcon = () => {
    const deviceType = getDeviceType();
    switch (deviceType) {
      case 'mobile': return <Smartphone className="w-4 h-4" />;
      case 'tablet': return <Tablet className="w-4 h-4" />;
      default: return <Monitor className="w-4 h-4" />;
    }
  };

  const getTestSummary = () => {
    const passed = testResults.filter(r => r.status === 'pass').length;
    const failed = testResults.filter(r => r.status === 'fail').length;
    const warnings = testResults.filter(r => r.status === 'warning').length;
    
    return { passed, failed, warnings, total: testResults.length };
  };

  const summary = getTestSummary();

  return (
    <div className="max-w-6xl mx-auto p-6 space-y-6">
      <div className="text-center mb-8">
        <h1 className="text-3xl mb-2">PictoBlox Integration Test Suite</h1>
        <p className="text-muted-foreground">
          Comprehensive testing for PictoBlox web integration across devices and platforms
        </p>
      </div>

      {/* Test Summary */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            {getDeviceIcon()}
            Test Summary
            <Badge variant={summary.failed > 0 ? 'destructive' : summary.warnings > 0 ? 'secondary' : 'default'}>
              {summary.failed > 0 ? 'Issues Found' : summary.warnings > 0 ? 'Warnings' : 'All Good'}
            </Badge>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-4 gap-4 text-center">
            <div>
              <div className="text-2xl font-bold text-green-500">{summary.passed}</div>
              <div className="text-sm text-muted-foreground">Passed</div>
            </div>
            <div>
              <div className="text-2xl font-bold text-red-500">{summary.failed}</div>
              <div className="text-sm text-muted-foreground">Failed</div>
            </div>
            <div>
              <div className="text-2xl font-bold text-yellow-500">{summary.warnings}</div>
              <div className="text-sm text-muted-foreground">Warnings</div>
            </div>
            <div>
              <div className="text-2xl font-bold">{summary.total}</div>
              <div className="text-sm text-muted-foreground">Total</div>
            </div>
          </div>
          
          <div className="mt-4 flex items-center gap-2 text-sm text-muted-foreground">
            {navigator.onLine ? <Wifi className="w-4 h-4" /> : <WifiOff className="w-4 h-4" />}
            {navigator.onLine ? 'Online' : 'Offline'} • 
            Device: {getDeviceType()} • 
            Screen: {window.innerWidth}×{window.innerHeight}
          </div>
        </CardContent>
      </Card>

      <Tabs defaultValue="tests" className="w-full">
        <TabsList className="grid w-full grid-cols-3">
          <TabsTrigger value="tests">Compatibility Tests</TabsTrigger>
          <TabsTrigger value="integration">Live Integration</TabsTrigger>
          <TabsTrigger value="performance">Performance</TabsTrigger>
        </TabsList>

        <TabsContent value="tests" className="space-y-4">
          <div className="flex justify-between items-center">
            <h3 className="text-lg font-medium">Compatibility Test Results</h3>
            <div className="flex gap-2">
              <Button 
                onClick={runCompatibilityTests} 
                disabled={isRunningTests}
                variant="outline"
              >
                {isRunningTests ? 'Running...' : 'Rerun Tests'}
              </Button>
              <Button onClick={runProjectTests}>
                Test Projects
              </Button>
            </div>
          </div>

          <div className="grid gap-3">
            {testResults.map((result, index) => (
              <Card key={index}>
                <CardContent className="pt-4">
                  <div className="flex items-start justify-between">
                    <div className="flex items-center gap-3">
                      {getStatusIcon(result.status)}
                      <div>
                        <div className="font-medium">{result.name}</div>
                        <div className="text-sm text-muted-foreground">{result.message}</div>
                        {result.details && (
                          <div className="text-xs text-muted-foreground mt-1">{result.details}</div>
                        )}
                      </div>
                    </div>
                    <Badge variant={
                      result.status === 'pass' ? 'default' :
                      result.status === 'fail' ? 'destructive' : 'secondary'
                    }>
                      {result.status}
                    </Badge>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </TabsContent>

        <TabsContent value="integration" className="space-y-4">
          <div className="flex justify-between items-center">
            <h3 className="text-lg font-medium">Live PictoBlox Integration Test</h3>
            <div className="flex gap-2 items-center">
              <select 
                value={selectedAgeGroup} 
                onChange={(e) => setSelectedAgeGroup(e.target.value)}
                className="px-3 py-1 border rounded"
              >
                <option value="crib">Crib (5-7)</option>
                <option value="clicker">Clicker (8-12)</option>
                <option value="quest">Quest (13-18)</option>
              </select>
              <Button onClick={() => setShowEmbed(!showEmbed)}>
                <PlayCircle className="w-4 h-4 mr-1" />
                {showEmbed ? 'Hide' : 'Show'} PictoBlox
              </Button>
            </div>
          </div>

          {showEmbed && (
            <PictoBloxEmbed
              isMinimized={isMinimized}
              onToggleMinimize={() => setIsMinimized(!isMinimized)}
              ageGroup={selectedAgeGroup}
              sessionNumber={1}
              sessionTitle="Test Integration Session"
              preloadedProjects={generateSessionStarters(1, 'Test Session', selectedAgeGroup)}
              onProjectSave={(project) => {
                console.log('Test project saved:', project);
                pictoBloxIntegration.saveProject(project);
              }}
            />
          )}

          <Card>
            <CardHeader>
              <CardTitle>Integration Status</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-2 text-sm">
                <div>Sync Status: <Badge>{pictoBloxIntegration.syncStatus}</Badge></div>
                <div>Offline Mode: <Badge variant={pictoBloxIntegration.offlineMode ? 'secondary' : 'default'}>
                  {pictoBloxIntegration.offlineMode ? 'Offline' : 'Online'}
                </Badge></div>
                <div>Projects: {pictoBloxIntegration.projects.length} total</div>
                {pictoBloxIntegration.lastSyncTime && (
                  <div>Last Sync: {pictoBloxIntegration.lastSyncTime.toLocaleTimeString()}</div>
                )}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="performance" className="space-y-4">
          <h3 className="text-lg font-medium">Performance Metrics</h3>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <Card>
              <CardHeader>
                <CardTitle className="text-base">Device Information</CardTitle>
              </CardHeader>
              <CardContent className="space-y-2 text-sm">
                <div>Device Type: {getDeviceType()}</div>
                <div>Screen Resolution: {window.innerWidth} × {window.innerHeight}</div>
                <div>Pixel Ratio: {window.devicePixelRatio}</div>
                <div>User Agent: {navigator.userAgent.substring(0, 60)}...</div>
                <div>Platform: {navigator.platform}</div>
                <div>Language: {navigator.language}</div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle className="text-base">Browser Capabilities</CardTitle>
              </CardHeader>
              <CardContent className="space-y-2 text-sm">
                <div>WebGL: {(() => {
                  try {
                    const canvas = document.createElement('canvas');
                    return !!(canvas.getContext('webgl') || canvas.getContext('experimental-webgl'));
                  } catch {
                    return false;
                  }
                })() ? 'Supported' : 'Not Supported'}</div>
                <div>Local Storage: {typeof(Storage) !== "undefined" ? 'Available' : 'Not Available'}</div>
                <div>Touch Events: {'ontouchstart' in window ? 'Supported' : 'Not Supported'}</div>
                <div>Fullscreen API: {supportsFullscreen() ? 'Supported' : 'Not Supported'}</div>
                <div>Network Status: {navigator.onLine ? 'Online' : 'Offline'}</div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>
      </Tabs>
    </div>
  );
}