import { useState, useEffect, useRef } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Badge } from './ui/badge';
import { 
  Maximize2, 
  Minimize2, 
  RotateCcw, 
  Save, 
  Upload, 
  FileText, 
  Download,
  Maximize,
  Minimize,
  Smartphone,
  Tablet,
  Monitor,
  Wifi,
  WifiOff
} from 'lucide-react';
import { 
  PictoBloxProject,
  PictoBloxManager,
  PictoBloxStorage,
  generateSessionStarters,
  getDeviceType,
  supportsFullscreen,
  enterFullscreen,
  exitFullscreen
} from '../utils/pictoBloxUtils';

interface PictoBloxEmbedProps {
  isMinimized: boolean;
  onToggleMinimize: () => void;
  ageGroup?: 'crib' | 'clicker' | 'quest';
  sessionNumber?: number;
  sessionTitle?: string;
  preloadedProjects?: PictoBloxProject[];
  onProjectSave?: (project: PictoBloxProject) => void;
}

export function PictoBloxEmbed({ 
  isMinimized, 
  onToggleMinimize, 
  ageGroup = 'quest',
  sessionNumber = 1,
  sessionTitle = '',
  preloadedProjects = [],
  onProjectSave
}: PictoBloxEmbedProps) {
  const [selectedProject, setSelectedProject] = useState<PictoBloxProject | null>(null);
  const [iframeLoaded, setIframeLoaded] = useState(false);
  const [projectSaved, setProjectSaved] = useState(false);
  const [isFullscreen, setIsFullscreen] = useState(false);
  const [isConnected, setIsConnected] = useState(false);
  const [deviceType, setDeviceType] = useState<'mobile' | 'tablet' | 'desktop'>('desktop');
  const [autoSaveEnabled, setAutoSaveEnabled] = useState(true);
  const [lastSaved, setLastSaved] = useState<Date | null>(null);
  
  const iframeRef = useRef<HTMLIFrameElement>(null);
  const containerRef = useRef<HTMLDivElement>(null);
  const pictoBloxManager = PictoBloxManager.getInstance();

  // Initialize component
  useEffect(() => {
    setDeviceType(getDeviceType());
    
    // Load saved project from localStorage
    const saved = PictoBloxStorage.getCurrentProject();
    if (saved && saved.sessionId === sessionNumber) {
      setSelectedProject(saved);
    }

    // Handle fullscreen change events
    const handleFullscreenChange = () => {
      setIsFullscreen(!!document.fullscreenElement);
    };

    document.addEventListener('fullscreenchange', handleFullscreenChange);
    document.addEventListener('webkitfullscreenchange', handleFullscreenChange);
    document.addEventListener('mozfullscreenchange', handleFullscreenChange);

    return () => {
      document.removeEventListener('fullscreenchange', handleFullscreenChange);
      document.removeEventListener('webkitfullscreenchange', handleFullscreenChange);
      document.removeEventListener('mozfullscreenchange', handleFullscreenChange);
    };
  }, [sessionNumber]);

  // Initialize PictoBlox connection when iframe loads
  useEffect(() => {
    if (iframeLoaded && iframeRef.current) {
      pictoBloxManager.initialize(iframeRef.current).then((connected) => {
        setIsConnected(connected);
        
        if (connected) {
          // Configure PictoBlox for this session
          pictoBloxManager.setLessonContext({
            sessionId: sessionNumber,
            sessionTitle: sessionTitle,
            ageGroup,
            lessonObjectives: [],
            difficulty: ageGroup === 'crib' ? 'beginner' : ageGroup === 'clicker' ? 'intermediate' : 'advanced'
          });

          // Load selected project if available
          if (selectedProject) {
            pictoBloxManager.loadProject(selectedProject);
          }
        }
      });
    }
  }, [iframeLoaded, sessionNumber, sessionTitle, ageGroup, selectedProject]);

  // Auto-save functionality
  useEffect(() => {
    if (!autoSaveEnabled || !isConnected || !selectedProject) return;

    const autoSaveInterval = setInterval(async () => {
      const project = await pictoBloxManager.saveProject();
      if (project) {
        project.sessionId = sessionNumber;
        project.ageGroup = ageGroup;
        PictoBloxStorage.saveCurrentProject(project);
        PictoBloxStorage.saveProject(project);
        setLastSaved(new Date());
      }
    }, 30000); // Auto-save every 30 seconds

    return () => clearInterval(autoSaveInterval);
  }, [autoSaveEnabled, isConnected, selectedProject, sessionNumber, ageGroup]);

  // Generate all available projects
  const getAllProjects = (): PictoBloxProject[] => {
    const sessionStarters = generateSessionStarters(sessionNumber, sessionTitle, ageGroup);
    const userProjects = PictoBloxStorage.getSessionProjects(sessionNumber);
    return [...sessionStarters, ...preloadedProjects, ...userProjects];
  };

  const allProjects = getAllProjects();

  // Age-specific styling
  const getAgeStyles = () => {
    switch (ageGroup) {
      case 'crib':
        return {
          headerBg: 'bg-gradient-to-r from-pink-400 to-purple-500',
          buttonStyle: 'bg-gradient-to-r from-pink-400 to-purple-500 hover:from-pink-500 hover:to-purple-600',
          cardStyle: 'rounded-2xl shadow-lg border-4 border-pink-100',
          minimizedCard: 'rounded-2xl border-4 border-pink-200 bg-gradient-to-r from-pink-50 to-purple-50',
          iconColor: 'text-pink-600',
          title: '🎨 PictoBlox Playground'
        };
      case 'clicker':
        return {
          headerBg: 'bg-gradient-to-r from-blue-500 to-green-500',
          buttonStyle: 'bg-gradient-to-r from-blue-500 to-green-500 hover:from-blue-600 hover:to-green-600',
          cardStyle: 'rounded-xl shadow-md border-2 border-blue-100',
          minimizedCard: 'rounded-xl border-2 border-blue-200 bg-gradient-to-r from-blue-50 to-green-50',
          iconColor: 'text-blue-600',
          title: '🚀 PictoBlox Workshop'
        };
      default:
        return {
          headerBg: 'bg-gradient-to-r from-indigo-500 to-purple-600',
          buttonStyle: 'bg-blue-600 hover:bg-blue-700',
          cardStyle: 'rounded-lg shadow-sm',
          minimizedCard: 'rounded-lg border bg-gray-50',
          iconColor: 'text-indigo-600',
          title: 'PictoBlox IDE'
        };
    }
  };

  const styles = getAgeStyles();

  const handleProjectLoad = async (project: PictoBloxProject) => {
    setSelectedProject(project);
    PictoBloxStorage.saveCurrentProject(project);
    
    if (isConnected) {
      const success = await pictoBloxManager.loadProject(project);
      if (!success) {
        console.error('Failed to load project into PictoBlox');
      }
    }
  };

  const handleSaveProject = async () => {
    if (!isConnected || !selectedProject) return;

    const project = await pictoBloxManager.saveProject();
    if (project) {
      project.sessionId = sessionNumber;
      project.ageGroup = ageGroup;
      project.name = selectedProject.name;
      project.description = selectedProject.description;
      
      PictoBloxStorage.saveProject(project);
      PictoBloxStorage.saveCurrentProject(project);
      setSelectedProject(project);
      setProjectSaved(true);
      setLastSaved(new Date());
      
      // Callback to parent component
      if (onProjectSave) {
        onProjectSave(project);
      }
      
      setTimeout(() => setProjectSaved(false), 2000);
    }
  };

  const handleExportProject = () => {
    if (selectedProject) {
      PictoBloxStorage.exportProject(selectedProject);
    }
  };

  const handleImportProject = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;

    PictoBloxStorage.importProject(file)
      .then((project) => {
        project.sessionId = sessionNumber;
        project.ageGroup = ageGroup;
        PictoBloxStorage.saveProject(project);
        handleProjectLoad(project);
      })
      .catch((error) => {
        console.error('Failed to import project:', error);
        alert('Failed to import project. Please check the file format.');
      });
  };

  const handleResetProject = () => {
    const emptyProject = allProjects.find(p => p.name === 'Empty Project');
    if (emptyProject) {
      handleProjectLoad(emptyProject);
    }
  };

  const handleFullscreenToggle = async () => {
    if (!containerRef.current) return;

    try {
      if (isFullscreen) {
        await exitFullscreen();
      } else {
        await enterFullscreen(containerRef.current);
      }
    } catch (error) {
      console.warn('Fullscreen toggle failed:', error);
      
      // Show user-friendly error message based on age group
      const message = ageGroup === 'crib' ? 
        '🎭 Oops! Full screen needs permission. Try clicking the fullscreen button in your browser!' :
        ageGroup === 'clicker' ?
        '⚡ Fullscreen blocked! Your browser needs permission to go fullscreen.' :
        'Fullscreen unavailable. Please check your browser settings or try a different browser.';
      
      // You can replace this with a proper toast notification if you have one
      alert(message);
    }
  };

  const formatLastSaved = (date: Date): string => {
    const now = new Date();
    const diff = now.getTime() - date.getTime();
    const minutes = Math.floor(diff / 60000);
    
    if (minutes < 1) return 'Just now';
    if (minutes < 60) return `${minutes}m ago`;
    
    const hours = Math.floor(minutes / 60);
    if (hours < 24) return `${hours}h ago`;
    
    return date.toLocaleDateString();
  };

  const getDeviceIcon = () => {
    switch (deviceType) {
      case 'mobile': return <Smartphone className="w-3 h-3" />;
      case 'tablet': return <Tablet className="w-3 h-3" />;
      default: return <Monitor className="w-3 h-3" />;
    }
  };

  if (isMinimized) {
    return (
      <Card className={`fixed bottom-4 right-4 w-80 z-50 ${styles.minimizedCard}`}>
        <CardHeader className="pb-3">
          <div className="flex items-center justify-between">
            <CardTitle className="text-sm flex items-center gap-2">
              <div className={`w-3 h-3 rounded-full ${isConnected ? 'bg-green-500 animate-pulse' : 'bg-red-500'}`}></div>
              {styles.title}
              {selectedProject && (
                <Badge variant="secondary" className="text-xs">
                  {selectedProject.name}
                </Badge>
              )}
            </CardTitle>
            <div className="flex items-center gap-1">
              {getDeviceIcon()}
              <Button 
                size="sm" 
                variant="ghost" 
                onClick={onToggleMinimize}
                className={ageGroup === 'crib' ? 'hover:bg-pink-100' : ageGroup === 'clicker' ? 'hover:bg-blue-100' : ''}
              >
                <Maximize2 className="w-4 h-4" />
              </Button>
            </div>
          </div>
        </CardHeader>
        <CardContent className="pt-0">
          <div className="text-xs text-muted-foreground mb-2 flex items-center gap-2">
            {isConnected ? <Wifi className="w-3 h-3" /> : <WifiOff className="w-3 h-3" />}
            {isConnected ? (
              ageGroup === 'crib' ? '🎨 Creating amazing projects!' : 
              ageGroup === 'clicker' ? '🔧 Building cool stuff!' : 
              'PictoBlox is running smoothly'
            ) : 'Connecting to PictoBlox...'}
          </div>
          {lastSaved && (
            <div className="text-xs text-muted-foreground mb-2">
              Last saved: {formatLastSaved(lastSaved)}
            </div>
          )}
          <div className="flex gap-1">
            <Button 
              size="sm" 
              variant="outline" 
              className="flex-1" 
              onClick={handleSaveProject}
              disabled={!isConnected}
            >
              <Save className="w-3 h-3 mr-1" />
              {projectSaved ? 'Saved!' : 'Save'}
            </Button>
            <Button 
              size="sm" 
              variant="outline" 
              onClick={handleResetProject}
              disabled={!isConnected}
            >
              <RotateCcw className="w-3 h-3" />
            </Button>
            {supportsFullscreen() && (
              <Button 
                size="sm" 
                variant="outline" 
                onClick={handleFullscreenToggle}
              >
                <Maximize className="w-3 h-3" />
              </Button>
            )}
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <div ref={containerRef} className={`w-full ${isFullscreen ? 'fixed inset-0 z-50 bg-black' : ''}`}>
      <Card className={`w-full h-full ${styles.cardStyle} ${isFullscreen ? 'rounded-none border-0' : ''}`}>
        <CardHeader className={`${styles.headerBg} text-white ${isFullscreen ? 'rounded-none' : ''}`}>
          <div className="flex items-center justify-between">
            <CardTitle className="flex items-center gap-2">
              <div className={`w-4 h-4 rounded-full ${isConnected ? 'bg-green-400 animate-pulse' : 'bg-red-400'}`}></div>
              {styles.title}
              {selectedProject && (
                <Badge variant="secondary" className="bg-white/20 text-white border-white/30">
                  {selectedProject.name}
                </Badge>
              )}
              <div className="flex items-center gap-1 text-xs opacity-75">
                {getDeviceIcon()}
                {isConnected ? <Wifi className="w-3 h-3" /> : <WifiOff className="w-3 h-3" />}
              </div>
            </CardTitle>
            <div className="flex items-center gap-2">
              {lastSaved && (
                <Badge variant="secondary" className="bg-white/20 text-white border-white/30 text-xs">
                  Saved {formatLastSaved(lastSaved)}
                </Badge>
              )}
              {projectSaved && (
                <Badge className="bg-green-500 text-white animate-bounce">
                  Saved!
                </Badge>
              )}
              
              {/* Project Actions */}
              <Button 
                size="sm" 
                variant="ghost" 
                onClick={handleSaveProject}
                disabled={!isConnected}
                className="text-white hover:bg-white/20"
                title="Save Project"
              >
                <Save className="w-4 h-4" />
              </Button>
              
              <Button 
                size="sm" 
                variant="ghost" 
                onClick={handleExportProject}
                disabled={!selectedProject}
                className="text-white hover:bg-white/20"
                title="Export Project"
              >
                <Download className="w-4 h-4" />
              </Button>

              <input
                type="file"
                accept=".digifunzi,.json"
                onChange={handleImportProject}
                className="hidden"
                id="import-project"
              />
              <Button 
                size="sm" 
                variant="ghost" 
                onClick={() => document.getElementById('import-project')?.click()}
                className="text-white hover:bg-white/20"
                title="Import Project"
              >
                <Upload className="w-4 h-4" />
              </Button>

              {/* Fullscreen Toggle */}
              {supportsFullscreen() && (
                <Button 
                  size="sm" 
                  variant="ghost" 
                  onClick={handleFullscreenToggle}
                  className="text-white hover:bg-white/20"
                  title={isFullscreen ? "Exit Fullscreen" : "Enter Fullscreen"}
                >
                  {isFullscreen ? <Minimize className="w-4 h-4" /> : <Maximize className="w-4 h-4" />}
                </Button>
              )}
              
              {/* Minimize Toggle */}
              {!isFullscreen && (
                <Button 
                  size="sm" 
                  variant="ghost" 
                  onClick={onToggleMinimize}
                  className="text-white hover:bg-white/20"
                  title="Minimize"
                >
                  <Minimize2 className="w-4 h-4" />
                </Button>
              )}
            </div>
          </div>
        </CardHeader>
      <CardContent className="p-0">
        {/* Project Selection */}
        {!selectedProject && (
          <div className="p-6">
            <h3 className="text-lg mb-4 flex items-center gap-2">
              <FileText className={`w-5 h-5 ${styles.iconColor}`} />
              Choose a Project to Start
            </h3>
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-3">
              {allProjects.map((project) => (
                <Card 
                  key={project.id} 
                  className={`cursor-pointer hover:shadow-md transition-all ${
                    ageGroup === 'crib' ? 'hover:border-pink-200 rounded-xl' :
                    ageGroup === 'clicker' ? 'hover:border-blue-200 rounded-lg' :
                    'hover:border-indigo-200'
                  }`}
                  onClick={() => handleProjectLoad(project)}
                >
                  <CardContent className="p-4">
                    <h4 className="font-medium text-sm mb-1">{project.name}</h4>
                    <p className="text-xs text-muted-foreground">{project.description}</p>
                  </CardContent>
                </Card>
              ))}
            </div>
            <div className="mt-4 p-4 rounded-lg bg-muted/50">
              <div className="flex items-center gap-2 text-sm text-muted-foreground">
                <Upload className="w-4 h-4" />
                You can also upload your own project files or start from scratch
              </div>
            </div>
          </div>
        )}

        {/* PictoBlox IDE Iframe */}
        {selectedProject && (
          <div className="relative">
            {!iframeLoaded && (
              <div className="absolute inset-0 bg-gray-100 flex items-center justify-center z-10">
                <div className="text-center">
                  <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-indigo-600 mx-auto mb-4"></div>
                  <p className="text-sm text-muted-foreground mb-2">
                    {ageGroup === 'crib' ? '🎨 Loading your playground...' :
                     ageGroup === 'clicker' ? '🚀 Preparing your workshop...' :
                     'Loading PictoBlox IDE...'}
                  </p>
                  <div className="text-xs text-muted-foreground">
                    Connecting to PictoBlox servers...
                  </div>
                </div>
              </div>
            )}
            
            {/* Actual PictoBlox iframe */}
            <iframe
              ref={iframeRef}
              src="https://pictoblox.ai/"
              className={`w-full border-0 ${isFullscreen ? 'h-screen' : 'h-96 md:h-[500px] lg:h-[600px]'}`}
              onLoad={() => setIframeLoaded(true)}
              title="PictoBlox IDE"
              allow="camera; microphone; web-share; fullscreen *; clipboard-read; clipboard-write; autoplay; encrypted-media"
              sandbox="allow-same-origin allow-scripts allow-forms allow-popups allow-presentation allow-fullscreen"
              loading="lazy"
            />
            
            {/* Connection Status Overlay */}
            {iframeLoaded && !isConnected && (
              <div className="absolute top-4 right-4 bg-red-500 text-white px-3 py-1 rounded-lg text-xs flex items-center gap-2">
                <WifiOff className="w-3 h-3" />
                Connection Lost
              </div>
            )}
            
            {/* Quick Actions Bar - Only show when not in fullscreen */}
            {!isFullscreen && (
              <div className="absolute bottom-4 left-4 right-4 bg-white/90 backdrop-blur-sm rounded-lg p-2 shadow-lg">
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-2">
                    <Button 
                      size="sm" 
                      variant="outline" 
                      onClick={handleResetProject}
                      disabled={!isConnected}
                    >
                      <RotateCcw className="w-3 h-3 mr-1" />
                      Reset
                    </Button>
                    <Button 
                      size="sm" 
                      variant="outline" 
                      onClick={handleSaveProject}
                      disabled={!isConnected}
                    >
                      <Save className="w-3 h-3 mr-1" />
                      {projectSaved ? 'Saved!' : 'Save'}
                    </Button>
                    <Button 
                      size="sm" 
                      variant="outline" 
                      onClick={() => setAutoSaveEnabled(!autoSaveEnabled)}
                      className={autoSaveEnabled ? 'bg-green-50 border-green-200' : ''}
                    >
                      Auto-save {autoSaveEnabled ? 'ON' : 'OFF'}
                    </Button>
                  </div>
                  <div className="text-xs text-muted-foreground flex items-center gap-2">
                    {getDeviceIcon()}
                    Session {sessionNumber} • {selectedProject.name}
                  </div>
                </div>
              </div>
            )}
          </div>
        )}
      </CardContent>
      </Card>
    </div>
  );
}