import { useState } from 'react';
import { Header } from './Header';
import { Header5to7 } from './Header5to7';
import { Header7to12 } from './Header7to12';
import { CourseCard } from './CourseCard';
import { AgeSelection } from './AgeSelection';
import { Button } from './ui/button';
import { Badge } from './ui/badge';
import { Sparkles, Target, Award, ChevronRight } from 'lucide-react';
import { coursesData } from '../constants/coursesData';

interface CoursesPageProps {
  onSelectCourse: (courseId: string) => void;
  selectedAgeGroup?: string;
}

export function CoursesPage({ onSelectCourse, selectedAgeGroup }: CoursesPageProps) {
  const [selectedAge, setSelectedAge] = useState<string | null>(selectedAgeGroup || null);
  const [showAgeSelection, setShowAgeSelection] = useState(false);

  const filteredCourses = selectedAge 
    ? coursesData.filter(course => {
        if (selectedAge === '5-7') return course.id === 'pictoblox-5-7';
        if (selectedAge === '7-12') return course.id === 'pictoblox-7-12';
        if (selectedAge === '13+') return course.id === 'pictoblox-general';
        return true;
      })
    : coursesData;

  const renderHeader = () => {
    if (selectedAge === '5-7') return <Header5to7 />;
    if (selectedAge === '7-12') return <Header7to12 />;
    return <Header />;
  };

  return (
    <div className="min-h-screen bg-gray-50">
      {renderHeader()}
      
      {/* Hero Section */}
      <div className="relative bg-gradient-to-br from-blue-50 via-indigo-50 to-purple-50 py-16">
        <div className="container mx-auto px-4">
          <div className="text-center max-w-4xl mx-auto">
            <div className="flex justify-center items-center gap-2 mb-4">
              <Sparkles className="w-6 h-6 text-indigo-600" />
              <Badge variant="secondary" className="bg-indigo-100 text-indigo-700">
                Choose Your Learning Path
              </Badge>
            </div>
            
            <h1 className="text-4xl md:text-5xl font-bold text-gray-900 mb-6">
              Start Your Coding Journey with{' '}
              <span className="bg-gradient-to-r from-indigo-600 to-purple-600 bg-clip-text text-transparent">
                DigiFunzi
              </span>
            </h1>
            
            <p className="text-xl text-gray-600 mb-8 leading-relaxed">
              Learn programming through interactive games, creative projects, and hands-on activities. 
              Choose the perfect course designed for your age and skill level.
            </p>

            {/* Age Selection CTA */}
            <div className="flex justify-center items-center gap-4 mb-8">
              <Button
                onClick={() => setShowAgeSelection(!showAgeSelection)}
                variant={selectedAge ? "outline" : "default"}
                size="lg"
                className="gap-2"
              >
                <Target className="w-5 h-5" />
                {selectedAge ? `Age: ${selectedAge}` : 'Find My Perfect Course'}
                <ChevronRight className="w-4 h-4" />
              </Button>
              
              {selectedAge && (
                <Button
                  onClick={() => setSelectedAge(null)}
                  variant="ghost"
                  size="lg"
                >
                  Show All Courses
                </Button>
              )}
            </div>

            {/* Age Selection Component */}
            {showAgeSelection && (
              <div className="bg-white rounded-lg p-6 shadow-lg max-w-md mx-auto mb-8">
                <AgeSelection
                  selectedAge={selectedAge}
                  onAgeSelect={(age) => {
                    setSelectedAge(age);
                    setShowAgeSelection(false);
                  }}
                />
              </div>
            )}

            {/* Stats */}
            <div className="grid grid-cols-1 md:grid-cols-3 gap-8 max-w-2xl mx-auto">
              <div className="text-center">
                <div className="text-3xl font-bold text-indigo-600 mb-2">10K+</div>
                <div className="text-gray-600">Students Enrolled</div>
              </div>
              <div className="text-center">
                <div className="text-3xl font-bold text-purple-600 mb-2">95%</div>
                <div className="text-gray-600">Completion Rate</div>
              </div>
              <div className="text-center">
                <div className="text-3xl font-bold text-green-600 mb-2">4.9★</div>
                <div className="text-gray-600">Student Rating</div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Courses Grid */}
      <div className="container mx-auto px-4 py-16">
        <div className="flex justify-between items-center mb-8">
          <div>
            <h2 className="text-3xl font-bold text-gray-900 mb-2">
              {selectedAge ? `Courses for Ages ${selectedAge}` : 'Available Courses'}
            </h2>
            <p className="text-gray-600">
              Choose the course that matches your learning goals and experience level
            </p>
          </div>
          
          {selectedAge && (
            <Badge variant="outline" className="text-sm">
              {filteredCourses.length} course{filteredCourses.length !== 1 ? 's' : ''} found
            </Badge>
          )}
        </div>

        <div className={`grid gap-8 ${
          selectedAge === '5-7' ? 'grid-cols-1 lg:grid-cols-2' : 
          'grid-cols-1 md:grid-cols-2 lg:grid-cols-3'
        }`}>
          {filteredCourses.map((course) => {
            let ageGroup: 'crib' | 'clicker' | 'quest' = 'quest';
            if (course.id === 'pictoblox-5-7') ageGroup = 'crib';
            else if (course.id === 'pictoblox-7-12') ageGroup = 'clicker';
            
            return (
              <CourseCard
                key={course.id}
                course={course}
                onSelectCourse={onSelectCourse}
                ageGroup={ageGroup}
              />
            );
          })}
        </div>

        {filteredCourses.length === 0 && (
          <div className="text-center py-16">
            <div className="text-gray-400 mb-4">
              <Award className="w-16 h-16 mx-auto" />
            </div>
            <h3 className="text-xl font-semibold text-gray-900 mb-2">
              No courses found for this age group
            </h3>
            <p className="text-gray-600 mb-6">
              Try selecting a different age range or view all available courses
            </p>
            <Button onClick={() => setSelectedAge(null)} variant="outline">
              Show All Courses
            </Button>
          </div>
        )}
      </div>

      {/* Features Section */}
      <div className="bg-white py-16">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Why Choose DigiFunzi?
            </h2>
            <p className="text-gray-600 max-w-2xl mx-auto">
              Our innovative approach to coding education combines fun, creativity, and practical learning
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            <div className="text-center p-6">
              <div className="w-16 h-16 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-full flex items-center justify-center mx-auto mb-4">
                <Sparkles className="w-8 h-8 text-white" />
              </div>
              <h3 className="text-xl font-semibold mb-2">Interactive Learning</h3>
              <p className="text-gray-600">
                Hands-on projects and games that make coding concepts easy to understand and remember
              </p>
            </div>

            <div className="text-center p-6">
              <div className="w-16 h-16 bg-gradient-to-br from-purple-500 to-pink-600 rounded-full flex items-center justify-center mx-auto mb-4">
                <Target className="w-8 h-8 text-white" />
              </div>
              <h3 className="text-xl font-semibold mb-2">Age-Appropriate Content</h3>
              <p className="text-gray-600">
                Carefully designed curriculum that matches your child's developmental stage and interests
              </p>
            </div>

            <div className="text-center p-6">
              <div className="w-16 h-16 bg-gradient-to-br from-green-500 to-emerald-600 rounded-full flex items-center justify-center mx-auto mb-4">
                <Award className="w-8 h-8 text-white" />
              </div>
              <h3 className="text-xl font-semibold mb-2">Expert Support</h3>
              <p className="text-gray-600">
                Technical mentors and progress tracking to ensure your child succeeds in their coding journey
              </p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}