import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "./ui/card";
import { Progress } from "./ui/progress";
import { Clock, Users, Award, Play, CheckCircle, ArrowLeft } from "lucide-react";
import { ImageWithFallback } from "./figma/ImageWithFallback";

interface CourseHeaderProps {
  title: string;
  description: string;
  price: string;
  sessions: number;
  duration: string;
  students: string;
  image: string;
  onStartCourse: () => void;
  onBackToCourses?: () => void;
  overallProgress?: number;
  timeSpent?: string;
  sessionsCompleted?: number;
  totalSessions?: number;
  hasStarted?: boolean;
  ageGroup?: 'crib' | 'clicker' | 'quest';
}

export function CourseHeader({
  title,
  description,
  price,
  sessions,
  duration,
  students,
  image,
  onStartCourse,
  onBackToCourses,
  overallProgress = 0,
  timeSpent = "0m",
  sessionsCompleted = 0,
  totalSessions = 0,
  hasStarted = false,
  ageGroup = 'quest'
}: CourseHeaderProps) {
  return (
    <div className="bg-white border-b border-gray-200">
      <div className="max-w-7xl mx-auto px-4 py-8">
        {onBackToCourses && (
          <div className="mb-6">
            <Button 
              variant="ghost" 
              onClick={onBackToCourses}
              className="gap-2 text-muted-foreground hover:text-foreground"
            >
              <ArrowLeft className="w-4 h-4" />
              Back to Courses
            </Button>
          </div>
        )}
        <div className="grid lg:grid-cols-2 gap-8 items-center">
          <div className="space-y-6">
            <div className="space-y-4">
              <Badge variant="secondary" className="w-fit">
                ANIMATIONS
              </Badge>
              <h1 className="text-4xl">{title}</h1>
              <p className="text-muted-foreground leading-relaxed">
                {description}
              </p>
            </div>
            
            <div className="grid grid-cols-2 gap-4">
              <div className="bg-blue-50 p-4 rounded-lg text-center">
                <div className="text-2xl text-blue-600 mb-1">KES</div>
                <div className="text-3xl">{price}</div>
                <div className="text-sm text-muted-foreground">Price</div>
              </div>
              <div className="bg-green-50 p-4 rounded-lg text-center">
                <div className="text-2xl text-green-600 mb-1">{sessions}</div>
                <div className="text-lg">Sessions</div>
                <div className="text-sm text-muted-foreground">Duration</div>
              </div>
            </div>

            <div className="flex items-center gap-6 text-sm text-muted-foreground">
              <div className="flex items-center gap-2">
                <Clock className="w-4 h-4" />
                <span>{duration}</span>
              </div>
              <div className="flex items-center gap-2">
                <Users className="w-4 h-4" />
                <span>{students}</span>
              </div>
            </div>

            <Button onClick={onStartCourse} className="w-full bg-blue-600 hover:bg-blue-700">
              {hasStarted ? 'Continue Course' : 'Start Course'}
            </Button>

            {hasStarted && (
              <Card className="mt-6">
                <CardHeader>
                  <div className="flex items-center justify-between">
                    <CardTitle className="text-xl">My Progress</CardTitle>
                    <Badge variant="secondary" className="bg-blue-100 text-blue-700">
                      Just Getting Started
                    </Badge>
                  </div>
                </CardHeader>
                <CardContent className="space-y-6">
                  <div>
                    <div className="flex items-center justify-between mb-2">
                      <span className="text-sm">Overall Progress</span>
                      <span className="text-sm">{overallProgress}%</span>
                    </div>
                    <Progress value={overallProgress} className="h-2" />
                  </div>

                  <div className="grid grid-cols-2 gap-6">
                    <div className="flex items-center gap-3">
                      <Clock className="w-5 h-5 text-muted-foreground" />
                      <div>
                        <div className="text-sm text-muted-foreground">Time Spent</div>
                        <div className="font-medium">{timeSpent}</div>
                      </div>
                    </div>

                    <div className="flex items-center gap-3">
                      <CheckCircle className="w-5 h-5 text-muted-foreground" />
                      <div>
                        <div className="text-sm text-muted-foreground">Sessions</div>
                        <div className="font-medium">{sessionsCompleted} completed</div>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            )}
          </div>

          <div className="relative">
            <div className="aspect-video rounded-lg overflow-hidden bg-black relative">
              {image.endsWith('.mp4') ? (
                <video 
                  src={image} 
                  controls 
                  className="w-full h-full object-cover"
                  poster=""
                >
                  Your browser does not support the video tag.
                </video>
              ) : (
                <>
                  <ImageWithFallback
                    src={image}
                    alt={title}
                    className="w-full h-full object-cover"
                  />
                  <div className="absolute inset-0 bg-black/20 flex items-center justify-center">
                    <div className="bg-white/90 rounded-full p-4">
                      <Play className="w-8 h-8 text-blue-600" />
                    </div>
                  </div>
                </>
              )}
            </div>
            <div className="mt-3 text-center text-sm text-muted-foreground">
              Course Introduction Video
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}