import { Card } from './ui/card';
import { Badge } from './ui/badge';
import { Button } from './ui/button';
import { Clock, Users, BookOpen, Star } from 'lucide-react';
import type { Course } from '../constants/coursesData';
import { ImageWithFallback } from './figma/ImageWithFallback';

interface CourseCardProps {
  course: Course;
  onSelectCourse: (courseId: string) => void;
  ageGroup?: 'crib' | 'clicker' | 'quest';
}

export function CourseCard({ course, onSelectCourse, ageGroup = 'quest' }: CourseCardProps) {
  // Age-specific styling
  const getAgeStyles = () => {
    switch (ageGroup) {
      case 'crib':
        return {
          cardStyle: 'rounded-3xl shadow-lg hover:shadow-2xl hover:scale-[1.05] border-4 border-pink-100',
          buttonText: '🎉 Let\'s Play!',
          buttonStyle: 'bg-gradient-to-r from-pink-400 to-purple-500 hover:from-pink-500 hover:to-purple-600 rounded-2xl text-white font-bold text-lg px-8 py-3',
          headerIcon: '🌟',
          priceStyle: 'text-3xl font-bold bg-gradient-to-r from-pink-500 to-purple-600 bg-clip-text text-transparent',
          featuresTitle: '🎯 Fun Things to Learn:'
        };
      case 'clicker':
        return {
          cardStyle: 'rounded-2xl shadow-lg hover:shadow-xl hover:scale-[1.03] border-2 border-blue-100',
          buttonText: '🚀 Start Coding!',
          buttonStyle: 'bg-gradient-to-r from-blue-500 to-green-500 hover:from-blue-600 hover:to-green-600 rounded-xl text-white font-semibold px-6 py-2.5',
          headerIcon: '⚡',
          priceStyle: 'text-2xl font-bold bg-gradient-to-r from-blue-500 to-green-600 bg-clip-text text-transparent',
          featuresTitle: '🎮 What You\'ll Build:'
        };
      default:
        return {
          cardStyle: 'rounded-xl shadow-md hover:shadow-lg hover:scale-[1.02]',
          buttonText: 'Start Learning',
          buttonStyle: 'bg-gradient-to-r from-indigo-500 to-purple-600 hover:from-indigo-600 hover:to-purple-700 rounded-lg text-white font-medium px-6 py-2',
          headerIcon: '🎯',
          priceStyle: 'text-2xl font-semibold text-primary',
          featuresTitle: 'What you\'ll learn:'
        };
    }
  };

  const styles = getAgeStyles();

  const getStatsLayout = () => {
    if (ageGroup === 'crib') {
      return (
        <div className="grid grid-cols-2 gap-3 mb-4">
          <div className="bg-pink-50 rounded-2xl p-3 text-center">
            <div className="text-2xl mb-1">🎮</div>
            <div className="text-sm font-medium text-pink-700">{course.sessions} Adventures</div>
          </div>
          <div className="bg-purple-50 rounded-2xl p-3 text-center">
            <div className="text-2xl mb-1">⏰</div>
            <div className="text-sm font-medium text-purple-700">{course.duration}</div>
          </div>
        </div>
      );
    } else if (ageGroup === 'clicker') {
      return (
        <div className="grid grid-cols-3 gap-3 mb-4">
          <div className="bg-blue-50 rounded-xl p-2 text-center">
            <BookOpen className="w-4 h-4 text-blue-600 mx-auto mb-1" />
            <div className="text-xs font-medium text-blue-700">{course.sessions} Challenges</div>
          </div>
          <div className="bg-green-50 rounded-xl p-2 text-center">
            <Clock className="w-4 h-4 text-green-600 mx-auto mb-1" />
            <div className="text-xs font-medium text-green-700">{course.duration}</div>
          </div>
          <div className="bg-cyan-50 rounded-xl p-2 text-center">
            <Users className="w-4 h-4 text-cyan-600 mx-auto mb-1" />
            <div className="text-xs font-medium text-cyan-700">{course.students}</div>
          </div>
        </div>
      );
    } else {
      return (
        <div className="grid grid-cols-3 gap-4 mb-4">
          <div className="flex items-center gap-2 text-sm text-muted-foreground">
            <BookOpen className="w-4 h-4" />
            <span>{course.sessions} sessions</span>
          </div>
          <div className="flex items-center gap-2 text-sm text-muted-foreground">
            <Clock className="w-4 h-4" />
            <span>{course.duration}</span>
          </div>
          <div className="flex items-center gap-2 text-sm text-muted-foreground">
            <Users className="w-4 h-4" />
            <span>{course.students}</span>
          </div>
        </div>
      );
    }
  };

  return (
    <Card className={`overflow-hidden transition-all duration-300 group ${styles.cardStyle}`}>
      {/* Header with gradient background */}
      <div className={`relative h-48 bg-gradient-to-br ${course.color} p-6 text-white`}>
        <div className="absolute top-4 right-4">
          <Badge variant="secondary" className="bg-white/20 text-white border-white/30">
            {course.level}
          </Badge>
        </div>
        
        <div className="absolute bottom-4 left-6 right-6">
          <div className="flex items-center gap-2 mb-2">
            <span className="text-lg">{styles.headerIcon}</span>
            <span className="text-sm opacity-90">
              {ageGroup === 'crib' ? 'Super Fun!' : ageGroup === 'clicker' ? 'Popular Choice' : 'Popular Choice'}
            </span>
          </div>
          <h3 className={`${ageGroup === 'crib' ? 'text-2xl' : 'text-xl'} font-semibold mb-1`}>
            {course.title}
          </h3>
          <p className="text-sm opacity-90">{course.ageRange}</p>
        </div>

        {/* Course image overlay */}
        <div className="absolute inset-0 bg-black/20">
          <ImageWithFallback
            src={course.image} 
            alt={course.title}
            className="w-full h-full object-cover mix-blend-overlay opacity-60"
          />
        </div>
      </div>

      {/* Content */}
      <div className={`${ageGroup === 'crib' ? 'p-8' : 'p-6'}`}>
        <p className={`text-muted-foreground mb-4 leading-relaxed ${
          ageGroup === 'crib' ? 'text-base' : 'text-sm'
        }`}>
          {course.description}
        </p>

        {/* Course stats */}
        {getStatsLayout()}

        {/* Features */}
        <div className="mb-6">
          <h4 className={`font-medium mb-3 ${
            ageGroup === 'crib' ? 'text-base text-pink-700' : 
            ageGroup === 'clicker' ? 'text-sm text-blue-700' : 
            'text-sm'
          }`}>
            {styles.featuresTitle}
          </h4>
          <div className="space-y-2">
            {course.features.slice(0, 3).map((feature, index) => (
              <div key={index} className={`flex items-center gap-2 text-muted-foreground ${
                ageGroup === 'crib' ? 'text-sm' : 'text-sm'
              }`}>
                {ageGroup === 'crib' ? (
                  <span className="text-lg">{'⭐🎨🚀'[index] || '✨'}</span>
                ) : ageGroup === 'clicker' ? (
                  <div className="w-2 h-2 bg-gradient-to-r from-blue-400 to-green-500 rounded-full flex-shrink-0" />
                ) : (
                  <div className="w-1.5 h-1.5 bg-primary rounded-full flex-shrink-0" />
                )}
                <span>{feature}</span>
              </div>
            ))}
            {course.features.length > 3 && (
              <div className="text-sm text-muted-foreground">
                +{course.features.length - 3} more {ageGroup === 'crib' ? 'fun things' : 'features'}
              </div>
            )}
          </div>
        </div>

        {/* Price and action */}
        <div className={`flex items-center justify-between pt-4 border-t border-border ${
          ageGroup === 'crib' ? 'border-pink-100' : ageGroup === 'clicker' ? 'border-blue-100' : ''
        }`}>
          <div>
            <span className={styles.priceStyle}>{course.price}</span>
            <span className="text-sm text-muted-foreground ml-1">
              {ageGroup === 'crib' ? 'total fun!' : 'total'}
            </span>
          </div>
          <Button 
            onClick={() => onSelectCourse(course.id)}
            className={styles.buttonStyle}
          >
            {styles.buttonText}
          </Button>
        </div>
      </div>
    </Card>
  );
}