import { Card, CardContent, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Star, Upload, FileText, CheckCircle, Clock, AlertCircle } from "lucide-react";
import exampleImage from 'figma:asset/893ac20ddb0a941c1457b7d8764e2d2ed6026363.png';

interface AssignmentCardProps {
  assignmentStatus: 'not-issued' | 'issued' | 'submitted' | 'graded' | 're-issued';
  onUploadAssignment: () => void;
}

interface GradingCriteria {
  name: string;
  rating: number;
  status: 'Exceeding' | 'Meeting' | 'Below';
}

export function AssignmentCard({ assignmentStatus, onUploadAssignment }: AssignmentCardProps) {
  const gradingCriteria: GradingCriteria[] = [
    { name: "Relevance", rating: 5, status: "Exceeding" },
    { name: "Working Code", rating: 5, status: "Exceeding" },
    { name: "Creativity", rating: 4, status: "Meeting" },
    { name: "Completion", rating: 5, status: "Exceeding" },
    { name: "Timely Completion", rating: 5, status: "Exceeding" }
  ];

  const getStatusBadge = (status: string) => {
    switch (status) {
      case 'not-issued':
        return <Badge variant="secondary" className="bg-gray-100 text-gray-700">Not Issued</Badge>;
      case 'issued':
        return <Badge variant="secondary" className="bg-blue-100 text-blue-700">Issued</Badge>;
      case 'submitted':
        return <Badge variant="secondary" className="bg-yellow-100 text-yellow-700">Submitted</Badge>;
      case 'graded':
        return <Badge variant="secondary" className="bg-green-100 text-green-700">Graded</Badge>;
      case 're-issued':
        return <Badge variant="secondary" className="bg-orange-100 text-orange-700">Re-issued</Badge>;
      default:
        return null;
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'not-issued':
        return <AlertCircle className="w-5 h-5 text-gray-500" />;
      case 'issued':
        return <FileText className="w-5 h-5 text-blue-500" />;
      case 'submitted':
        return <Clock className="w-5 h-5 text-yellow-500" />;
      case 'graded':
        return <CheckCircle className="w-5 h-5 text-green-500" />;
      case 're-issued':
        return <AlertCircle className="w-5 h-5 text-orange-500" />;
      default:
        return null;
    }
  };

  const renderStars = (rating: number) => {
    return Array.from({ length: 5 }, (_, i) => (
      <Star
        key={i}
        className={`w-4 h-4 ${i < rating ? 'fill-yellow-400 text-yellow-400' : 'text-gray-300'}`}
      />
    ));
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'Exceeding':
        return 'text-red-500';
      case 'Meeting':
        return 'text-orange-500';
      default:
        return 'text-gray-500';
    }
  };

  if (assignmentStatus === 'not-issued') {
    return (
      <Card>
        <CardContent className="pt-6 text-center">
          <div className="text-muted-foreground">
            <FileText className="w-8 h-8 mx-auto mb-2" />
            <p className="text-sm">No assignment issued yet</p>
            <p className="text-xs mt-1">Complete the assignment video to unlock the assignment</p>
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle className="text-base flex items-center gap-2">
            {getStatusIcon(assignmentStatus)}
            Session Assignment
          </CardTitle>
          {getStatusBadge(assignmentStatus)}
        </div>
      </CardHeader>
      <CardContent className="space-y-4">
        {assignmentStatus === 'issued' || assignmentStatus === 're-issued' ? (
          <div className="space-y-4">
            <div className="p-4 bg-blue-50 rounded-lg">
              <h4 className="font-medium mb-2">Assignment Instructions</h4>
              <p className="text-sm text-muted-foreground">
                Create a walking animation using the concepts learned in this session. 
                Upload your project file (.sb3) along with a brief explanation of your approach.
              </p>
            </div>
            <div className="space-y-2">
              <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                <Upload className="w-8 h-8 mx-auto mb-2 text-muted-foreground" />
                <p className="text-sm text-muted-foreground mb-2">
                  Drag and drop your files here, or click to browse
                </p>
                <Button onClick={onUploadAssignment} variant="outline" size="sm">
                  Choose Files
                </Button>
              </div>
              <p className="text-xs text-muted-foreground">
                Supported formats: .sb3, .pdf, .doc, .docx (Max size: 10MB)
              </p>
            </div>
          </div>
        ) : assignmentStatus === 'submitted' ? (
          <div className="space-y-4">
            <div className="p-4 bg-yellow-50 rounded-lg">
              <h4 className="font-medium mb-2">Assignment Submitted</h4>
              <p className="text-sm text-muted-foreground">
                Your assignment has been submitted successfully and is being reviewed by a technical mentor.
              </p>
            </div>
            <div className="space-y-2">
              <div className="text-sm">
                <span className="font-medium">Submitted files:</span>
              </div>
              <div className="flex items-center gap-2 p-2 bg-gray-50 rounded">
                <FileText className="w-4 h-4 text-blue-500" />
                <span className="text-sm">walking_animation.sb3</span>
              </div>
            </div>
          </div>
        ) : assignmentStatus === 'graded' ? (
          <div className="space-y-6">
            <div className="p-4 bg-green-50 rounded-lg">
              <h4 className="font-medium mb-2">Assignment Graded</h4>
              <p className="text-sm text-muted-foreground">
                Your assignment has been reviewed and graded by your technical mentor.
              </p>
            </div>
            
            <div className="space-y-4">
              <div className="space-y-3">
                {gradingCriteria.map((criteria, index) => (
                  <div key={index} className="flex items-center justify-between">
                    <div className="flex items-center gap-3">
                      <span className="text-sm font-medium">{criteria.name}</span>
                      <div className="flex items-center gap-1">
                        {renderStars(criteria.rating)}
                      </div>
                    </div>
                    <span className={`text-sm font-medium ${getStatusColor(criteria.status)}`}>
                      {criteria.status}
                    </span>
                  </div>
                ))}
              </div>

              <div className="p-4 bg-blue-50 rounded-lg">
                <h5 className="font-medium mb-3">Mentor Feedback</h5>
                <div className="flex gap-3 mb-3">
                  <div className="w-10 h-10 bg-gray-300 rounded-full flex items-center justify-center flex-shrink-0">
                    <span className="text-sm font-medium text-gray-600">SJ</span>
                  </div>
                  <div className="flex-1">
                    <div className="flex items-center gap-2 mb-1">
                      <span className="text-sm font-medium">Sarah Johnson</span>
                      <span className="text-xs text-muted-foreground">Technical Mentor</span>
                    </div>
                    <div className="text-xs text-muted-foreground mb-2">
                      September 1, 2025 at 3:45 PM
                    </div>
                    <p className="text-sm text-muted-foreground">
                      Excellent work on the walking animation! Your code structure is clean and the animation flows smoothly. 
                      Great creativity in adding background elements. Keep up the good work!
                    </p>
                  </div>
                </div>
              </div>
            </div>
          </div>
        ) : null}
      </CardContent>
    </Card>
  );
}