import { useState } from 'react';
import { motion } from 'motion/react';
import { Button } from './ui/button';
import { Card } from './ui/card';
import { Badge } from './ui/badge';
import { Sparkles, Target, Users, Clock, BookOpen, Star } from 'lucide-react';
import { ImageWithFallback } from './figma/ImageWithFallback';
import digifunziLogo from 'figma:asset/517ee9d784589cb130eb229f0ac854531acfb4ba.png';

interface AgeGroup {
  id: string;
  title: string;
  subtitle: string;
  description: string;
  longDescription: string;
  icon: string;
  colors: string;
  hoverColors: string;
  features: string[];
  stats: {
    sessions: number;
    duration: string;
    students: string;
  };
}

interface AgeSelectionPageProps {
  onAgeSelect: (ageGroup: string) => void;
}

export function AgeSelectionPage({ onAgeSelect }: AgeSelectionPageProps) {
  const [selectedAge, setSelectedAge] = useState<string | null>(null);

  const ageGroups: AgeGroup[] = [
    {
      id: '5-7',
      title: '5-7 Years',
      subtitle: 'Little Coders',
      description: 'Fun and colorful learning with simple coding concepts',
      longDescription: 'Perfect for young learners who are just starting their coding journey. We use storytelling, colorful characters, and simple drag-and-drop activities to make programming feel like playing with digital toys.',
      icon: '🌈',
      colors: 'from-pink-200 to-yellow-200',
      hoverColors: 'from-pink-300 to-yellow-300',
      features: [
        'Interactive storytelling with code',
        'Colorful characters and animations', 
        'Simple drag-and-drop programming',
        'Parent progress updates',
        'Short 15-minute sessions'
      ],
      stats: {
        sessions: 8,
        duration: '4 weeks',
        students: '2,150+'
      }
    },
    {
      id: '7-12',
      title: '7-12 Years',
      subtitle: 'Young Developers',
      description: 'Game-like learning with achievements and challenges',
      longDescription: 'Designed for curious minds ready to dive deeper into programming. Students learn through game creation, interactive challenges, and hands-on projects that build real coding skills.',
      icon: '🎮',
      colors: 'from-blue-200 to-green-200',
      hoverColors: 'from-blue-300 to-green-300',
      features: [
        'Build interactive games and animations',
        'Learn logic and problem-solving',
        'Achievement system and rewards',
        'Collaborative projects',
        'Portfolio development'
      ],
      stats: {
        sessions: 12,
        duration: '6 weeks',
        students: '3,890+'
      }
    },
    {
      id: '13+',
      title: '13+ Years',
      subtitle: 'Advanced Learners',
      description: 'Comprehensive coding education with advanced concepts',
      longDescription: 'A complete programming curriculum that covers everything from basic concepts to advanced topics like AI, IoT, and professional development practices. Perfect for students ready for serious coding education.',
      icon: '💻',
      colors: 'from-purple-200 to-gray-200',
      hoverColors: 'from-purple-300 to-gray-300',
      features: [
        'Complete programming curriculum',
        'AI and IoT integration',
        'Professional development practices',
        'Technical mentor support',
        'Industry-standard projects'
      ],
      stats: {
        sessions: 15,
        duration: '8 weeks',
        students: '5,247+'
      }
    }
  ];

  const selectedAgeGroup = ageGroups.find(group => group.id === selectedAge);

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 via-indigo-50 to-purple-50">
      {/* Header */}
      <div className="container mx-auto px-4 py-8">
        <motion.div
          initial={{ opacity: 0, y: -20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
          className="text-center mb-12"
        >
          <div className="flex justify-center mb-6">
            <ImageWithFallback 
              src={digifunziLogo} 
              alt="DigiFunzi Logo" 
              className="h-16 w-auto"
            />
          </div>
          
          <div className="flex justify-center items-center gap-2 mb-4">
            <Sparkles className="w-6 h-6 text-indigo-600" />
            <Badge variant="secondary" className="bg-indigo-100 text-indigo-700">
              Personalized Learning Experience
            </Badge>
          </div>
          
          <h1 className="text-4xl md:text-5xl font-bold text-gray-900 mb-6">
            Welcome to{' '}
            <span className="bg-gradient-to-r from-indigo-600 to-purple-600 bg-clip-text text-transparent">
              DigiFunzi
            </span>
          </h1>
          
          <p className="text-xl text-gray-600 mb-8 max-w-3xl mx-auto leading-relaxed">
            Choose your learning path based on your age and experience level. 
            Our adaptive curriculum ensures the perfect balance of fun and learning for every student.
          </p>
        </motion.div>

        {/* Age Group Selection */}
        <div className="max-w-6xl mx-auto">
          <div className="grid md:grid-cols-3 gap-8 mb-12">
            {ageGroups.map((group, index) => (
              <motion.div
                key={group.id}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.6, delay: index * 0.2 }}
                whileHover={{ scale: 1.02 }}
                whileTap={{ scale: 0.98 }}
              >
                <Card 
                  className={`
                    cursor-pointer transition-all duration-300 overflow-hidden
                    ${selectedAge === group.id ? 'ring-2 ring-indigo-500 shadow-xl' : 'hover:shadow-lg'}
                  `}
                  onClick={() => setSelectedAge(group.id)}
                >
                  {/* Card Header with Gradient */}
                  <div className={`bg-gradient-to-br ${group.colors} p-6 text-center relative`}>
                    <div className="text-4xl mb-3">{group.icon}</div>
                    <h3 className="text-2xl font-bold text-gray-800 mb-1">
                      {group.title}
                    </h3>
                    <h4 className="text-lg font-medium text-gray-700 mb-2">
                      {group.subtitle}
                    </h4>
                    <p className="text-gray-600 text-sm">
                      {group.description}
                    </p>
                    
                    {selectedAge === group.id && (
                      <div className="absolute top-4 right-4">
                        <Badge className="bg-indigo-100 text-indigo-700">
                          Selected
                        </Badge>
                      </div>
                    )}
                  </div>

                  {/* Card Content */}
                  <div className="p-6">
                    {/* Stats */}
                    <div className="grid grid-cols-3 gap-4 mb-4">
                      <div className="text-center">
                        <div className="flex items-center justify-center gap-1 mb-1">
                          <BookOpen className="w-4 h-4 text-muted-foreground" />
                        </div>
                        <div className="font-semibold">{group.stats.sessions}</div>
                        <div className="text-xs text-muted-foreground">Sessions</div>
                      </div>
                      <div className="text-center">
                        <div className="flex items-center justify-center gap-1 mb-1">
                          <Clock className="w-4 h-4 text-muted-foreground" />
                        </div>
                        <div className="font-semibold">{group.stats.duration}</div>
                        <div className="text-xs text-muted-foreground">Duration</div>
                      </div>
                      <div className="text-center">
                        <div className="flex items-center justify-center gap-1 mb-1">
                          <Users className="w-4 h-4 text-muted-foreground" />
                        </div>
                        <div className="font-semibold">{group.stats.students}</div>
                        <div className="text-xs text-muted-foreground">Students</div>
                      </div>
                    </div>

                    {/* Features Preview */}
                    <div className="space-y-2">
                      {group.features.slice(0, 3).map((feature, idx) => (
                        <div key={idx} className="flex items-center gap-2 text-sm text-muted-foreground">
                          <div className="w-1.5 h-1.5 bg-indigo-500 rounded-full flex-shrink-0" />
                          <span>{feature}</span>
                        </div>
                      ))}
                      {group.features.length > 3 && (
                        <div className="text-sm text-muted-foreground">
                          +{group.features.length - 3} more features
                        </div>
                      )}
                    </div>
                  </div>
                </Card>
              </motion.div>
            ))}
          </div>

          {/* Detailed Information for Selected Age Group */}
          {selectedAgeGroup && (
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.4 }}
              className="mb-12"
            >
              <Card className="p-8 bg-white shadow-lg">
                <div className="grid md:grid-cols-2 gap-8 items-center">
                  <div>
                    <h3 className="text-2xl font-bold mb-4 flex items-center gap-3">
                      <span className="text-3xl">{selectedAgeGroup.icon}</span>
                      {selectedAgeGroup.title} - {selectedAgeGroup.subtitle}
                    </h3>
                    <p className="text-gray-600 mb-6 leading-relaxed">
                      {selectedAgeGroup.longDescription}
                    </p>
                    
                    <div className="space-y-3">
                      <h4 className="font-semibold text-gray-800">Complete Feature List:</h4>
                      {selectedAgeGroup.features.map((feature, idx) => (
                        <div key={idx} className="flex items-center gap-3">
                          <div className="w-2 h-2 bg-indigo-500 rounded-full flex-shrink-0" />
                          <span className="text-gray-700">{feature}</span>
                        </div>
                      ))}
                    </div>
                  </div>
                  
                  <div className="text-center">
                    <div className={`bg-gradient-to-br ${selectedAgeGroup.colors} rounded-2xl p-8 mb-6`}>
                      <div className="text-6xl mb-4">{selectedAgeGroup.icon}</div>
                      <div className="text-gray-800">
                        <div className="text-sm opacity-75">Perfect for</div>
                        <div className="font-bold text-xl">{selectedAgeGroup.title}</div>
                      </div>
                    </div>
                    
                    <div className="flex items-center justify-center gap-2 mb-4">
                      <Star className="w-5 h-5 text-yellow-500 fill-current" />
                      <span className="font-semibold">4.9/5 Student Rating</span>
                    </div>
                  </div>
                </div>
              </Card>
            </motion.div>
          )}

          {/* Action Button */}
          {selectedAge && (
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.4 }}
              className="text-center"
            >
              <Button
                onClick={() => onAgeSelect(selectedAge)}
                size="lg"
                className="bg-gradient-to-r from-indigo-600 to-purple-600 hover:from-indigo-700 hover:to-purple-700 text-white px-12 py-4 text-lg shadow-lg hover:shadow-xl transition-all duration-200"
              >
                <Target className="w-5 h-5 mr-2" />
                Start My {selectedAgeGroup?.title} Journey
              </Button>
              
              <p className="text-sm text-gray-500 mt-4">
                You can always change your age group later in your profile settings
              </p>
            </motion.div>
          )}
        </div>
      </div>
    </div>
  );
}