import { useState } from 'react';
import { Header } from './components/Header';
import { Header5to7 } from './components/Header5to7';
import { Header7to12 } from './components/Header7to12';
import { PictoBloxTestSuite } from './components/PictoBloxTestSuite';
import digiFunziLogo from 'figma:asset/517ee9d784589cb130eb229f0ac854531acfb4ba.png';
import { AgeSelectionPage } from './components/AgeSelectionPage';
import { CoursesPage } from './components/CoursesPage';
import { CourseHeader } from './components/CourseHeader';
import { CourseInfo } from './components/CourseInfo';
import { ProgressSection } from './components/ProgressSection';
import { SessionsList } from './components/SessionsList';
import { SessionDetail } from './components/SessionDetail';
import { useCourseNavigation } from './hooks/useCourseNavigation';
import { useSessionManagement } from './hooks/useSessionManagement';
import { createSessionsData, createProgressData, getDefaultVideoUrl } from './utils/sessionUtils';
import { courseData } from './constants/courseData';
import { getCourseDataById } from './constants/courseDataMap';

export default function App() {
  const [hasSelectedAge, setHasSelectedAge] = useState(false);
  const [selectedAgeGroup, setSelectedAgeGroup] = useState<string | null>(null);
  const [showTestSuite, setShowTestSuite] = useState(false);

  const {
    currentView,
    selectedCourseId,
    navigateToCourses,
    navigateToCourseDetail,
    navigateToSessionDetail,
  } = useCourseNavigation();

  const {
    currentSession,
    completedSessions,
    handleStartCourse,
    handleSessionClick,
    handleBackToCourse,
    handlePreviousSession,
    handleNextSession,
    handleSessionComplete,
    handleVideoProgress,
    videoTracking
  } = useSessionManagement();

  // Get course data based on selected course or default
  const activeCourseData = selectedCourseId 
    ? getCourseDataById(selectedCourseId) || courseData
    : courseData;

  const sessions = createSessionsData(completedSessions, currentSession);
  const progressData = createProgressData(completedSessions, activeCourseData.sessions);

  const currentSessionData = sessions.find(s => s.id === currentSession);

  const handleAgeSelection = (ageGroup: string) => {
    setSelectedAgeGroup(ageGroup);
    setHasSelectedAge(true);
  };

  // Show test suite if in development mode (check for URL parameter)
  if (showTestSuite || window.location.search.includes('test-pictoblox')) {
    return <PictoBloxTestSuite />;
  }

  // Render age selection page if user hasn't selected age yet
  if (!hasSelectedAge) {
    return <AgeSelectionPage onAgeSelect={handleAgeSelection} />;
  }

  // Render courses page
  if (currentView === 'courses') {
    return <CoursesPage onSelectCourse={navigateToCourseDetail} selectedAgeGroup={selectedAgeGroup} />;
  }

  // Render session detail page
  if (currentView === 'session-detail') {
    const renderHeader = () => {
      if (selectedCourseId === 'pictoblox-5-7') return <Header5to7 />;
      if (selectedCourseId === 'pictoblox-7-12') return <Header7to12 />;
      return <Header />;
    };

    const ageGroupType = selectedCourseId === 'pictoblox-5-7' ? 'crib' :
                        selectedCourseId === 'pictoblox-7-12' ? 'clicker' : 'quest';

    return (
      <div className="min-h-screen bg-gray-50">
        {renderHeader()}
        <SessionDetail
          sessionTitle={currentSessionData?.title || "Introduction to Pictoblox and Coding"}
          sessionNumber={currentSession}
          totalSessions={sessions.length}
          progress={progressData.overallProgress}
          sessionTime={videoTracking.formatTime(videoTracking.getTotalWatchTime(currentSession))}
          videoUrl={getDefaultVideoUrl()}
          onPrevious={handlePreviousSession}
          onNext={() => handleNextSession(sessions.length)}
          onHome={() => {
            handleBackToCourse();
            navigateToCourses();
          }}
          isCompleted={completedSessions.has(currentSession)}
          onSessionComplete={handleSessionComplete}
          onVideoProgress={handleVideoProgress}
          videoTracking={videoTracking}
          ageGroup={ageGroupType}
        />
      </div>
    );
  }

  // Render course detail page (default)
  const renderHeader = () => {
    if (selectedCourseId === 'pictoblox-5-7') return <Header5to7 />;
    if (selectedCourseId === 'pictoblox-7-12') return <Header7to12 />;
    return <Header />;
  };

  const ageGroupType = selectedCourseId === 'pictoblox-5-7' ? 'crib' :
                      selectedCourseId === 'pictoblox-7-12' ? 'clicker' : 'quest';

  return (
    <div className="min-h-screen bg-gray-50">
      {renderHeader()}
      
      <CourseHeader
        title={activeCourseData.title}
        description={activeCourseData.description}
        price={activeCourseData.price}
        sessions={activeCourseData.sessions}
        duration={activeCourseData.duration}
        students={activeCourseData.students}
        image={activeCourseData.image}
        onStartCourse={() => {
          handleStartCourse();
          navigateToSessionDetail();
        }}
        onBackToCourses={navigateToCourses}
        overallProgress={progressData.overallProgress}
        timeSpent={progressData.timeSpent}
        sessionsCompleted={progressData.sessionsCompleted}
        totalSessions={progressData.totalSessions}
        hasStarted={completedSessions.size > 0 || currentSession > 1}
        ageGroup={ageGroupType}
      />

      <ProgressSection
        overallProgress={progressData.overallProgress}
        timeSpent={progressData.timeSpent}
        sessionsCompleted={progressData.sessionsCompleted}
        totalSessions={progressData.totalSessions}
        ageGroup={ageGroupType}
      />

      <CourseInfo />

      <SessionsList
        sessions={sessions}
        onSessionClick={(sessionId) => {
          handleSessionClick(sessionId);
          navigateToSessionDetail();
        }}
        ageGroup={ageGroupType}
      />
    </div>
  );
}