// DigiFunzi TypeScript Utility Functions

import { ClassType, AGE_GROUP_CONFIGS, AgeGroupConfig, CourseData, SessionProgress } from '../types';

/**
 * Get age-appropriate configuration for a class type
 */
export function getAgeGroupConfig(classType: ClassType): AgeGroupConfig {
  return AGE_GROUP_CONFIGS[classType];
}

/**
 * Get age-appropriate terminology
 */
export function getTerminology(classType: ClassType, term: keyof AgeGroupConfig['terminology']): string {
  return AGE_GROUP_CONFIGS[classType].terminology[term];
}

/**
 * Get theme colors for a class type
 */
export function getThemeColors(classType: ClassType) {
  return AGE_GROUP_CONFIGS[classType].theme;
}

/**
 * Format currency in Kenyan Shillings
 */
export function formatKES(amount: number): string {
  return new Intl.NumberFormat('en-KE', {
    style: 'currency',
    currency: 'KES',
    minimumFractionDigits: 0,
    maximumFractionDigits: 2
  }).format(amount);
}

/**
 * Format time duration in minutes and seconds
 */
export function formatDuration(seconds: number): string {
  const minutes = Math.floor(seconds / 60);
  const remainingSeconds = seconds % 60;
  
  if (minutes === 0) {
    return `${remainingSeconds}s`;
  }
  
  return `${minutes}:${remainingSeconds.toString().padStart(2, '0')}`;
}

/**
 * Calculate course progress percentage
 */
export function calculateCourseProgress(currentSession: number, totalSessions: number): number {
  if (totalSessions === 0) return 0;
  return Math.round((currentSession / totalSessions) * 100);
}

/**
 * Check if a session is unlocked based on progression logic
 */
export function isSessionUnlocked(
  sessionNumber: number, 
  currentSession: number, 
  sessionProgress: SessionProgress[]
): boolean {
  // First session is always unlocked
  if (sessionNumber === 1) return true;
  
  // Can access current session or any previous completed session
  if (sessionNumber <= currentSession) return true;
  
  // Can access next session if current is completed
  const previousSession = sessionProgress.find(s => s.sessionNumber === sessionNumber - 1);
  return previousSession?.isCompleted || false;
}

/**
 * Get age-appropriate emoji for different contexts
 */
export function getContextEmoji(classType: ClassType, context: string): string {
  const emojiMap: Record<ClassType, Record<string, string>> = {
    crib: {
      course: "🎨",
      session: "⭐",
      project: "🎭",
      mentor: "🧙‍♂️",
      achievement: "🏆",
      video: "📺",
      assignment: "🎯"
    },
    clicker: {
      course: "🎮",
      session: "⚡",
      project: "🚀",
      mentor: "👨‍💻",
      achievement: "🏅",
      video: "🎬",
      assignment: "💼"
    },
    quest: {
      course: "💻",
      session: "📚",
      project: "⚙️",
      mentor: "👨‍🏫",
      achievement: "🎖️",
      video: "🎥",
      assignment: "📋"
    }
  };
  
  return emojiMap[classType][context] || "✨";
}

/**
 * Generate age-appropriate encouragement messages
 */
export function getEncouragementMessage(classType: ClassType, context: 'start' | 'progress' | 'complete'): string {
  const messages: Record<ClassType, Record<string, string[]>> = {
    crib: {
      start: [
        "Let's start this magical adventure! 🌟",
        "Ready to create something amazing? 🎨",
        "Time for some coding magic! ✨"
      ],
      progress: [
        "You're doing great! Keep going! 🌈",
        "Amazing work, young creator! 🎭",
        "Look how much you've learned! 🌟"
      ],
      complete: [
        "Wow! You completed another adventure! 🎉",
        "You're becoming a coding wizard! 🧙‍♂️",
        "Fantastic job, superstar! ⭐"
      ]
    },
    clicker: {
      start: [
        "Ready to level up your skills? 🎮",
        "Let's build something awesome! 🚀",
        "Time to unlock new abilities! ⚡"
      ],
      progress: [
        "Great progress, tech explorer! 💫",
        "You're mastering new skills! 🛠️",
        "Keep building, you're on fire! 🔥"
      ],
      complete: [
        "Mission accomplished! 🎯",
        "You've earned a new achievement! 🏅",
        "Excellent work, game changer! 🎮"
      ]
    },
    quest: {
      start: [
        "Ready to tackle this challenge? 💪",
        "Let's dive into advanced concepts! 📚",
        "Time to expand your expertise! 🎯"
      ],
      progress: [
        "Excellent progress on your journey! 📈",
        "You're developing professional skills! 💼",
        "Keep pushing forward, developer! 👨‍💻"
      ],
      complete: [
        "Outstanding achievement! 🏆",
        "You've earned professional recognition! 🎖️",
        "Impressive work, future innovator! 🌟"
      ]
    }
  };
  
  const contextMessages = messages[classType][context];
  return contextMessages[Math.floor(Math.random() * contextMessages.length)];
}

/**
 * Validate M-Pesa phone number format
 */
export function validateMpesaNumber(phoneNumber: string): boolean {
  // Kenyan mobile number format: +254XXXXXXXXX or 0XXXXXXXXX
  const kenyaPhoneRegex = /^(\+254|0)[17]\d{8}$/;
  return kenyaPhoneRegex.test(phoneNumber.replace(/\s/g, ''));
}

/**
 * Format M-Pesa phone number
 */
export function formatMpesaNumber(phoneNumber: string): string {
  const cleaned = phoneNumber.replace(/\s/g, '');
  
  if (cleaned.startsWith('0')) {
    return '+254' + cleaned.substring(1);
  }
  
  if (cleaned.startsWith('254')) {
    return '+' + cleaned;
  }
  
  if (cleaned.startsWith('+254')) {
    return cleaned;
  }
  
  return phoneNumber;
}

/**
 * Get session type label based on session number
 */
export function getSessionTypeLabel(sessionNumber: number, totalSessions: number): string {
  if (sessionNumber === 1) return "Introduction";
  if (sessionNumber === totalSessions) return "Capstone Project";
  if (sessionNumber === Math.ceil(totalSessions / 2)) return "Midterm Review";
  return `Session ${sessionNumber}`;
}

/**
 * Calculate estimated completion time for a course
 */
export function calculateEstimatedTime(totalSessions: number, averageSessionLength: number = 45): string {
  const totalMinutes = totalSessions * averageSessionLength;
  const hours = Math.floor(totalMinutes / 60);
  const minutes = totalMinutes % 60;
  
  if (hours === 0) {
    return `${minutes} minutes`;
  }
  
  if (minutes === 0) {
    return `${hours} hour${hours > 1 ? 's' : ''}`;
  }
  
  return `${hours}h ${minutes}m`;
}

/**
 * Generate course difficulty badge color
 */
export function getDifficultyColor(difficulty: 'beginner' | 'intermediate' | 'advanced'): string {
  const colors = {
    beginner: 'bg-green-100 text-green-800',
    intermediate: 'bg-yellow-100 text-yellow-800',
    advanced: 'bg-red-100 text-red-800'
  };
  
  return colors[difficulty];
}

/**
 * Generate random avatar color for users
 */
export function generateAvatarColor(userId: string): string {
  const colors = [
    'bg-red-500', 'bg-blue-500', 'bg-green-500', 'bg-yellow-500',
    'bg-purple-500', 'bg-pink-500', 'bg-indigo-500', 'bg-teal-500'
  ];
  
  // Use user ID to consistently generate same color
  const hash = userId.split('').reduce((a, b) => {
    a = ((a << 5) - a) + b.charCodeAt(0);
    return a & a;
  }, 0);
  
  return colors[Math.abs(hash) % colors.length];
}

/**
 * Debounce function for search and input handling
 */
export function debounce<T extends (...args: any[]) => any>(
  func: T,
  wait: number
): (...args: Parameters<T>) => void {
  let timeout: NodeJS.Timeout;
  
  return (...args: Parameters<T>) => {
    clearTimeout(timeout);
    timeout = setTimeout(() => func(...args), wait);
  };
}

/**
 * Format relative time (e.g., "2 hours ago", "1 day ago")
 */
export function formatRelativeTime(date: string | Date): string {
  const now = new Date();
  const past = new Date(date);
  const diffInSeconds = Math.floor((now.getTime() - past.getTime()) / 1000);
  
  if (diffInSeconds < 60) return 'just now';
  
  const diffInMinutes = Math.floor(diffInSeconds / 60);
  if (diffInMinutes < 60) return `${diffInMinutes} minute${diffInMinutes > 1 ? 's' : ''} ago`;
  
  const diffInHours = Math.floor(diffInMinutes / 60);
  if (diffInHours < 24) return `${diffInHours} hour${diffInHours > 1 ? 's' : ''} ago`;
  
  const diffInDays = Math.floor(diffInHours / 24);
  if (diffInDays < 30) return `${diffInDays} day${diffInDays > 1 ? 's' : ''} ago`;
  
  const diffInMonths = Math.floor(diffInDays / 30);
  return `${diffInMonths} month${diffInMonths > 1 ? 's' : ''} ago`;
}

/**
 * Check if user is in appropriate age group for class type
 */
export function validateAgeGroup(birthDate: string, classType: ClassType): boolean {
  const age = calculateAge(birthDate);
  
  switch (classType) {
    case 'crib':
      return age >= 5 && age <= 7;
    case 'clicker':
      return age >= 8 && age <= 12;
    case 'quest':
      return age >= 13;
    default:
      return false;
  }
}

/**
 * Calculate age from birth date
 */
export function calculateAge(birthDate: string): number {
  const today = new Date();
  const birth = new Date(birthDate);
  let age = today.getFullYear() - birth.getFullYear();
  const monthDiff = today.getMonth() - birth.getMonth();
  
  if (monthDiff < 0 || (monthDiff === 0 && today.getDate() < birth.getDate())) {
    age--;
  }
  
  return age;
}