// DigiFunzi TypeScript Interfaces and Types

// Age-based class categories
export type ClassType = "crib" | "clicker" | "quest";

// Main navigation sections
export type SectionType = "dashboard" | "classes" | "projects" | "billing" | "profile" | "settings" | "lesson" | "mentor-booking";

// Lesson and Course Interfaces
export interface CurrentLesson {
  courseId: string;
  courseTitle: string;
  currentSession: number;
  totalSessions: number;
  classType: ClassType;
}

export interface CourseData {
  title: string;
  currentSession: number;
  totalSessions: number;
  isLocked?: boolean;
  price?: number;
  description?: string;
  difficulty?: "beginner" | "intermediate" | "advanced";
}

// Mentor System Interfaces
export interface SelectedMentor {
  id: string;
  name: string;
  hourlyRate: number;
  expertise: string[];
  avatar?: string;
  rating?: number;
  experience?: string;
  bio?: string;
  availability?: TimeSlot[];
}

export interface TimeSlot {
  id: string;
  startTime: string;
  endTime: string;
  date: string;
  isAvailable: boolean;
}

export interface MentorBooking {
  id: string;
  mentorId: string;
  studentId: string;
  courseTitle?: string;
  classType?: ClassType;
  timeSlot: TimeSlot;
  status: "pending" | "confirmed" | "completed" | "cancelled";
  totalAmount: number;
  sessionType: "homework_help" | "project_review" | "concept_explanation" | "general_support";
}

// Session and Progress Interfaces
export interface SessionProgress {
  sessionNumber: number;
  isCompleted: boolean;
  completionDate?: string;
  timeSpent?: number; // in minutes
  score?: number; // percentage
}

export interface LessonContent {
  introduction?: VideoContent;
  activities: VideoContent[];
  assignment?: VideoContent;
  resources?: Resource[];
}

export interface VideoContent {
  id: string;
  title: string;
  src?: string;
  duration?: number; // in seconds
  isCompleted?: boolean;
  progress?: number; // percentage watched
}

export interface Resource {
  id: string;
  title: string;
  type: "pdf" | "youtube" | "presentation" | "document" | "link";
  url: string;
  description?: string;
}

// Payment and Billing Interfaces
export interface PaymentMethod {
  id: string;
  type: "mpesa" | "card" | "bank";
  name: string;
  isDefault?: boolean;
}

export interface BillingRecord {
  id: string;
  date: string;
  description: string;
  amount: number;
  status: "paid" | "pending" | "failed";
  paymentMethod: string;
  courseId?: string;
  mentorBookingId?: string;
}

// User Profile and Progress
export interface UserProfile {
  id: string;
  name: string;
  email: string;
  classType: ClassType;
  level: number;
  joinDate: string;
  avatar?: string;
  achievements: Achievement[];
  totalSessionsCompleted: number;
  totalCoursesCompleted: number;
  totalProjectsCreated: number;
  totalMentorSessions: number;
}

export interface Achievement {
  id: string;
  title: string;
  description: string;
  icon: string;
  unlockedDate: string;
  category: "courses" | "sessions" | "projects" | "mentorship" | "special";
}

// Age-Appropriate Content Configuration
export interface AgeGroupConfig {
  classType: ClassType;
  ageRange: string;
  theme: {
    primaryColor: string;
    secondaryColor: string;
    gradientFrom: string;
    gradientTo: string;
  };
  terminology: {
    assignments: string; // "Projects" for all groups
    sessions: string;
    courses: string;
  };
  contentStyle: "playful" | "engaging" | "professional";
}

// Component Props Interfaces
export interface SidebarProps {
  activeSection: SectionType;
  onSectionChange: (section: string) => void;
}

export interface ClassesViewProps {
  onViewCourse: (courseId: string, classType: ClassType) => void;
  onPayForCourse: (courseId: string) => void;
}

export interface LessonViewProps {
  courseTitle: string;
  currentSession: number;
  totalSessions: number;
  classType: ClassType;
  onBack: () => void;
  onSessionComplete: () => void;
  onNavigateToSession: (sessionNumber: number) => void;
  onSelectMentor: (mentor: SelectedMentor) => void;
  onNavigateToDashboard: () => void;
}

export interface MentorBookingProps {
  mentor: SelectedMentor;
  onBack: () => void;
  courseTitle?: string;
  classType?: ClassType;
  onNavigateToDashboard: () => void;
  onBackToClasses: () => void;
}

export interface VideoPlayerProps {
  src?: string;
  poster?: string;
  title?: string;
  className?: string;
  onProgress?: (progress: number) => void;
  onEnded?: () => void;
}

export interface PaymentModalProps {
  isOpen: boolean;
  onClose: () => void;
  courseTitle: string;
  amount: number;
  onPaymentSuccess: () => void;
}

// API Response Types
export interface ApiResponse<T> {
  success: boolean;
  data?: T;
  message?: string;
  error?: string;
}

export interface PaginatedResponse<T> extends ApiResponse<T[]> {
  pagination?: {
    page: number;
    limit: number;
    total: number;
    totalPages: number;
  };
}

// Form Data Types
export interface MentorBookingFormData {
  mentorId: string;
  timeSlotId: string;
  sessionType: MentorBooking['sessionType'];
  courseTitle?: string;
  classType?: ClassType;
  message?: string;
}

export interface PaymentFormData {
  paymentMethod: PaymentMethod['type'];
  phoneNumber?: string; // for M-Pesa
  amount: number;
  courseId?: string;
  mentorBookingId?: string;
}

// Constants
export const AGE_GROUP_CONFIGS: Record<ClassType, AgeGroupConfig> = {
  crib: {
    classType: "crib",
    ageRange: "5-7 years",
    theme: {
      primaryColor: "#feb139", // Yellow
      secondaryColor: "#38aae1", // Light Blue
      gradientFrom: "#feb139",
      gradientTo: "#38aae1"
    },
    terminology: {
      assignments: "Fun Projects",
      sessions: "Learning Adventures",
      courses: "Magic Courses"
    },
    contentStyle: "playful"
  },
  clicker: {
    classType: "clicker",
    ageRange: "8-12 years",
    theme: {
      primaryColor: "#38aae1", // Light Blue
      secondaryColor: "#25476a", // Navy
      gradientFrom: "#38aae1",
      gradientTo: "#25476a"
    },
    terminology: {
      assignments: "Cool Projects",
      sessions: "Tech Sessions",
      courses: "Skill Courses"
    },
    contentStyle: "engaging"
  },
  quest: {
    classType: "quest",
    ageRange: "13+ years",
    theme: {
      primaryColor: "#25476a", // Navy
      secondaryColor: "#38aae1", // Light Blue
      gradientFrom: "#25476a",
      gradientTo: "#38aae1"
    },
    terminology: {
      assignments: "Projects",
      sessions: "Sessions",
      courses: "Courses"
    },
    contentStyle: "professional"
  }
};

// Utility Types
export type Nullable<T> = T | null;
export type Optional<T, K extends keyof T> = Omit<T, K> & Partial<Pick<T, K>>;
export type RequiredFields<T, K extends keyof T> = T & Required<Pick<T, K>>;